# Homepage Performance Analysis

## 📊 Current Data Fetching Analysis

### **Database Queries (On Cache Miss):**
```
Total: 5 queries
Total Time: ~5.28ms (VERY FAST!)

1. Select pages (0.71ms)
2. Select content blocks (1.08ms)
3. Select SEO data (1.95ms)
4. Select media files (0.99ms)
5. Select relationships (0.55ms)
```

**✅ Good News:** No N+1 queries! All relationships are properly eager-loaded.

---

## 📦 Data Size Breakdown

### **Total Data Sent to Frontend: ~29.9KB**

```
Content Blocks: 25.1 KB (84% of total) ← LARGEST
SEO Data:        4.4 KB (15% of total)
Media Files:     ~0 KB (empty array)
Relationships:   ~0 KB (empty array)
Page Object:     ~0.4 KB
```

---

## 🔍 What's Actually Used vs What's Fetched

### **FETCHED from Database:**
```php
// HomeController.php line 31-111
$page                     // Full page object
$page->contentBlocks      // All 8 blocks (25KB)
$page->seo               // All SEO fields (4.4KB)
$page->media             // Empty array (0 items)
$page->relationships     // Empty array (0 items)
```

### **SENT to Frontend:**
```php
return Inertia::render('Home', [
    'page' => $page,                    // ❓ UNUSED (except indirectly)
    'contentBlocks' => $contentBlocks,  // ✅ USED
    'mediaFiles' => $mediaFiles,        // ❌ UNUSED (empty + not referenced)
    'pageData' => $pageData,            // ✅ USED (for SEO)
    'openGraph' => $openGraphData,      // ❌ REDUNDANT (duplicate of pageData)
    'twitterCard' => $twitterData,      // ❌ REDUNDANT (duplicate of pageData)
    'isCMS' => true,                    // ✅ USED
]);
```

### **USED in Home.jsx:**
```jsx
// Only 3 props are actually used:
contentBlocks  // ✅ Used for ContentBlockRenderer
pageData       // ✅ Used for MainLayout SEO
isCMS          // ✅ Used for conditional rendering

// These are NOT used anywhere:
page           // ❌ Never referenced
mediaFiles     // ❌ Never referenced (and empty anyway)
openGraph      // ❌ Not passed to MainLayout
twitterCard    // ❌ Not passed to MainLayout
```

---

## 🚨 Issues Identified

### **1. Duplicate SEO Data (MEDIUM Impact)**
```php
// In HomeController, lines 55-94
$openGraphData = $this->seoService->generateOpenGraphData(...);  // ❌ DUPLICATE
$twitterData = $this->seoService->generateTwitterCardData(...);  // ❌ DUPLICATE

$pageData = [
    'ogImage' => $seoData->og_image,      // ✅ Already here
    'ogUrl' => $seoData->og_url,          // ✅ Already here
    'twitterCard' => $seoData->twitter_card,  // ✅ Already here
    // ... etc - all duplicated
];
```

**Problem:** SEO data is being sent TWICE:
1. Inside `$pageData` array
2. As separate `$openGraph` and `$twitterCard` props

**Impact:** ~2KB of redundant data sent to frontend

---

### **2. Unused Page Object (LOW Impact)**
```php
'page' => $page,  // Full page model with all timestamps, IDs, etc.
```

**Problem:** The entire page object is sent but never used in Home.jsx

**Impact:** ~0.4KB unnecessary data

---

### **3. Empty/Unused Arrays (VERY LOW Impact)**
```php
'mediaFiles' => $mediaFiles,  // Empty array, never used
```

**Problem:** Sending empty arrays that serve no purpose

**Impact:** Minimal (~20 bytes)

---

### **4. Unnecessary Relationship Loading (LOW Impact)**
```php
// In PageService.php line 110
->with(['relationships.relatedPage'])
```

**Problem:** Homepage has 0 relationships, yet we're loading this every time

**Impact:** 1 extra query (0.55ms) + minimal data

---

## ✅ What's Working Well

### **1. Caching (EXCELLENT!)**
✅ PageService uses 1-hour cache
✅ On cache hit: Only 1 query (2.14ms) to check cache
✅ On cache miss: Only 5 queries (~5.28ms total)
✅ Proper cache invalidation on page updates

### **2. Eager Loading (EXCELLENT!)**
✅ No N+1 queries
✅ All relationships loaded efficiently
✅ Proper use of `with()` statements

### **3. Query Filtering (GOOD)**
✅ Only active content blocks loaded
✅ Only published pages loaded
✅ Content blocks ordered correctly

---

## 🎯 Optimization Opportunities

### **Priority 1: Remove Duplicate SEO Data (EASY WIN)**
**Impact:** Save ~2KB per request, simplify code

**Current:**
```php
return Inertia::render('Home', [
    'pageData' => $pageData,      // Has all SEO data
    'openGraph' => $openGraphData, // ← DUPLICATE
    'twitterCard' => $twitterData, // ← DUPLICATE
]);
```

**Optimized:**
```php
return Inertia::render('Home', [
    'pageData' => $pageData,  // All SEO data here
    // Remove openGraph and twitterCard
]);
```

---

### **Priority 2: Remove Unused Props (EASY WIN)**
**Impact:** Save ~0.5KB per request, cleaner code

**Current:**
```php
return Inertia::render('Home', [
    'page' => $page,          // ← UNUSED
    'mediaFiles' => $mediaFiles, // ← UNUSED & EMPTY
]);
```

**Optimized:**
```php
return Inertia::render('Home', [
    // Remove page and mediaFiles
]);
```

---

### **Priority 3: Optimize Relationship Loading (SMALL WIN)**
**Impact:** Save 1 query (0.55ms), minor improvement

**Current:**
```php
->with(['contentBlocks', 'seo', 'media', 'relationships.relatedPage'])
```

**Optimized (if homepage never has relationships):**
```php
->with(['contentBlocks', 'seo'])  // Remove media and relationships if unused
```

---

### **Priority 4: Reduce SEO Fields in PageData (MEDIUM WIN)**
**Impact:** Save ~1-2KB, only if many fields are unused

**Current:** Sending ALL 22 SEO fields to frontend
**Question:** Does MainLayout/StructuredData use all of them?

Let's check what StructuredData actually needs...

---

### **Priority 5: Content Block Optimization (OPTIONAL)**
**Impact:** Depends on block complexity

**Current:** All content blocks loaded at once (25KB)
**Consideration:** 
- Hero block: 1.1KB ✅ (Above fold, needed immediately)
- FAQs block: 7.7KB ⚠️ (Below fold, could lazy load)
- Testimonials: 6.3KB ⚠️ (Below fold, could lazy load)

**Potential:** Lazy load below-fold blocks (save ~14KB on initial load)

---

## 📈 Performance Score

### **Current Performance:**
```
✅ Caching:           EXCELLENT (1hr cache, proper invalidation)
✅ Query Count:       EXCELLENT (5 queries on miss, 1 on hit)
✅ Query Time:        EXCELLENT (~5ms total)
⚠️  Data Redundancy:  MEDIUM (duplicate SEO data)
⚠️  Unused Props:     LOW (page, mediaFiles, openGraph, twitterCard)
⚠️  Data Size:        GOOD (30KB total, 25KB is actual content)
```

### **Overall Grade: B+ (Very Good)**

---

## 🚀 Recommended Optimizations (In Order of Impact)

### **Quick Wins (Do These Now):**

1. **Remove Duplicate SEO Props**
   - Remove: `openGraph`, `twitterCard` props
   - Impact: -2KB, cleaner code
   - Effort: 5 minutes

2. **Remove Unused Props**
   - Remove: `page`, `mediaFiles` props
   - Impact: -0.5KB, cleaner code
   - Effort: 3 minutes

3. **Optimize Relationship Loading**
   - Don't load `media` and `relationships` if not needed
   - Impact: -1 query (-0.55ms)
   - Effort: 2 minutes

**Total Savings: ~2.5KB + 1 query**

---

### **Advanced Optimizations (Consider Later):**

4. **Lazy Load Below-Fold Blocks**
   - Load hero/services immediately
   - Lazy load FAQs/testimonials
   - Impact: -14KB initial load
   - Effort: 2-3 hours (requires code changes)

5. **Compress JSON Content**
   - Gzip content blocks before sending
   - Impact: 50-70% size reduction
   - Effort: 1 hour (add middleware)

---

## 🎯 Immediate Action Plan

### **Step 1: Clean Up HomeController**
Remove unused props:
```php
// Remove these from Inertia::render()
- 'page' => $page,
- 'mediaFiles' => $mediaFiles,
- 'openGraph' => $openGraphData,
- 'twitterCard' => $twitterData,
```

### **Step 2: Update PageService**
Only load needed relationships:
```php
->with(['contentBlocks' => function ($query) {
    $query->where('is_active', true)->orderBy('order');
}, 'seo'])  // Remove 'media' and 'relationships.relatedPage'
```

### **Step 3: Update Home.jsx**
Remove unused props from component signature.

---

## 📝 Summary

**Current State:**
- ✅ Database queries are FAST and efficient (5ms total)
- ✅ Caching is working perfectly
- ⚠️  Sending ~2.5KB of unnecessary data
- ⚠️  Loading 2 unused relationships

**Easy Wins Available:**
- Remove duplicate SEO data → Save 2KB
- Remove unused props → Save 0.5KB
- Don't load unused relationships → Save 1 query

**Bottom Line:**
Your homepage is already pretty well optimized! The main issue is **duplicate/unused data being sent to frontend**, not slow database queries. The quick fixes above will make it even better.

---

## ⏱️ Estimated Performance Improvement

**Before:** 30KB data transfer, 5 queries on cache miss
**After:**  27.5KB data transfer, 4 queries on cache miss

**Impact:** ~8% reduction in data transfer, 20% reduction in queries

---

## 🤔 Questions to Consider

1. **Does homepage ever use media files?** If not, remove from query
2. **Does homepage ever have relationships?** If not, remove from query
3. **Do you need all SEO fields in frontend?** Some might only be needed server-side
4. **Are FAQs/Testimonials needed immediately?** Consider lazy loading

Want me to implement the quick wins now? 🚀

