# Fix 422 Error on File Uploads at Namecheap

## Issue
Getting `422 Unprocessable Entity` error when uploading files on live Namecheap server.

## Root Causes & Solutions

### 1. **PHP File Upload Limits** ⭐ Most Common Issue

**Check your current limits:**
Create a file named `info.php` in your public directory:
```php
<?php phpinfo(); ?>
```
Visit `https://academicscribe.com/info.php` and look for:
- `upload_max_filesize`
- `post_max_size`
- `max_execution_time`
- `memory_limit`

**Solution: Update `.htaccess` in your public directory**

Add this to `/public_html/.htaccess`:
```apache
# Increase PHP upload limits
php_value upload_max_filesize 40M
php_value post_max_size 50M
php_value max_execution_time 300
php_value memory_limit 256M
php_value max_input_time 300
```

Or create/update `php.ini` in your root directory:
```ini
upload_max_filesize = 40M
post_max_size = 50M
max_execution_time = 300
memory_limit = 256M
max_input_time = 300
```

---

### 2. **Storage Directory Permissions**

**Issue:** Laravel can't write files to storage directory.

**Solution:**
```bash
# SSH into your server and run:
cd /home/yourusername/public_html
chmod -R 775 storage
chmod -R 775 bootstrap/cache
chown -R yourusername:yourusername storage bootstrap/cache

# Create symlink if not exists
php artisan storage:link
```

---

### 3. **CSRF Token Issues**

**Issue:** Token expired or missing in production.

**Current Code Check:**
The code already includes CSRF token in axios headers (good!), but verify:

**In `resources/js/app.jsx`:**
```jsx
window.axios.defaults.headers.common['X-CSRF-TOKEN'] = document.querySelector('meta[name="csrf-token"]')?.content;
window.axios.defaults.headers.common['X-Requested-With'] = 'XMLHttpRequest';
```

**In your blade template:**
```html
<head>
    <meta name="csrf-token" content="{{ csrf_token() }}">
</head>
```

---

### 4. **Server Disk Space**

**Check disk space:**
```bash
df -h
```

If disk is full (100%), files can't be uploaded.

---

### 5. **Temporary Upload Directory**

**Issue:** PHP temp directory doesn't exist or has wrong permissions.

**Solution:**
Create `.user.ini` in your public_html:
```ini
upload_tmp_dir = /home/yourusername/tmp
```

Then create the directory:
```bash
mkdir -p /home/yourusername/tmp
chmod 777 /home/yourusername/tmp
```

---

### 6. **ModSecurity Rules** (Namecheap Specific)

**Issue:** Namecheap's ModSecurity may block file uploads.

**Solution:**
Contact Namecheap support to:
1. Check ModSecurity logs
2. Whitelist your file upload endpoints
3. Temporarily disable ModSecurity for testing

---

### 7. **File Validation Issues**

**Check if specific file types are being blocked:**

Update validation in `OrderFilesController.php` to be more permissive for testing:

```php
// Temporarily add this for debugging
Log::info('File upload attempt', [
    'file_mime' => $request->file('file')?->getMimeType(),
    'file_size' => $request->file('file')?->getSize(),
    'file_name' => $request->file('file')?->getClientOriginalName(),
    'order_id' => $request->input('order_id'),
    'category' => $request->input('file_category'),
]);
```

---

## Quick Diagnostic Steps

### Step 1: Check PHP Configuration
Add to your `.env`:
```env
APP_DEBUG=true
LOG_LEVEL=debug
```

### Step 2: Check Laravel Logs
Download and check:
- `storage/logs/laravel.log`

Look for the actual validation error message.

### Step 3: Test with Small File
Try uploading a tiny text file (< 1KB) to isolate size issues.

### Step 4: Check Browser Console
The error response should show exact validation errors:
```javascript
// In ClientFileUpload.jsx, update error handling:
catch (error) {
    console.error('Upload error:', error);
    console.error('Response data:', error.response?.data);
    console.error('Validation errors:', error.response?.data?.errors);
    setError(error.response?.data?.message || 'Upload failed');
}
```

---

## Recommended .htaccess for Namecheap

Complete `.htaccess` file for `/public_html/`:

```apache
<IfModule mod_rewrite.c>
    <IfModule mod_negotiation.c>
        Options -MultiViews -Indexes
    </IfModule>

    RewriteEngine On

    # Handle Authorization Header
    RewriteCond %{HTTP:Authorization} .
    RewriteRule .* - [E=HTTP_AUTHORIZATION:%{HTTP:Authorization}]

    # Redirect Trailing Slashes If Not A Folder...
    RewriteCond %{REQUEST_FILENAME} !-d
    RewriteCond %{REQUEST_URI} (.+)/$
    RewriteRule ^ %1 [L,R=301]

    # Send Requests To Front Controller...
    RewriteCond %{REQUEST_FILENAME} !-d
    RewriteCond %{REQUEST_FILENAME} !-f
    RewriteRule ^ index.php [L]
</IfModule>

# Increase PHP limits
php_value upload_max_filesize 40M
php_value post_max_size 50M
php_value max_execution_time 300
php_value memory_limit 256M
php_value max_input_time 300

# Security headers
<IfModule mod_headers.c>
    Header set X-Content-Type-Options "nosniff"
    Header set X-Frame-Options "SAMEORIGIN"
    Header set X-XSS-Protection "1; mode=block"
</IfModule>

# Compression
<IfModule mod_deflate.c>
    AddOutputFilterByType DEFLATE text/html text/plain text/xml text/css text/javascript application/javascript
</IfModule>
```

---

## Testing Checklist

- [ ] Create info.php and check current PHP limits
- [ ] Update .htaccess with increased limits
- [ ] Verify storage directory permissions (775)
- [ ] Check storage symlink exists
- [ ] Test with small file first
- [ ] Check Laravel logs for actual error
- [ ] Verify CSRF token is present
- [ ] Check disk space on server
- [ ] Contact Namecheap about ModSecurity if needed

---

## Expected Error Messages

**If it's a size issue:**
```json
{
  "errors": {
    "file": ["The file may not be greater than 40960 kilobytes."]
  }
}
```

**If it's a permission issue:**
```json
{
  "message": "File upload failed: Unable to write file to disk"
}
```

**If it's a CSRF issue:**
```json
{
  "message": "CSRF token mismatch."
}
```

**If it's a validation issue:**
```json
{
  "errors": {
    "file": ["The file must be one of the following types: DOC, DOCX, XLS, XLSX, PPT, PPTX, PDF, ZIP"]
  }
}
```

---

## Contact Namecheap Support

If none of these work, open a ticket with Namecheap including:
1. The exact error from Laravel logs
2. Screenshot of the browser console error
3. Request that they check ModSecurity logs
4. Ask them to verify your account's upload limits
