<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\Conversation;
use App\Models\User;
use Carbon\Carbon;
use Illuminate\Support\Facades\Log;

class ArchiveOldConversations extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'messaging:auto-archive {--dry-run : Show what would be archived without actually archiving}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Automatically archive resolved conversations that are older than the specified period';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $isDryRun = $this->option('dry-run');
        
        // Check if auto-archiving is enabled
        if (!config('messaging.automation.auto_archive_enabled')) {
            $this->info('Auto-archiving is disabled in configuration.');
            return 0;
        }

        $daysOld = config('messaging.automation.auto_archive_after_days');
        
        if (!$daysOld) {
            $this->error('Auto-archive days not configured.');
            return 1;
        }

        $cutoffDate = Carbon::now()->subDays($daysOld);
        
        $this->info("Looking for resolved conversations older than: {$cutoffDate->format('Y-m-d H:i:s')}");

        // Find conversations that should be auto-archived
        $conversations = Conversation::where('status', 'resolved')
            ->where('resolved_at', '<', $cutoffDate)
            ->with(['participants'])
            ->get();

        if ($conversations->isEmpty()) {
            $this->info('No conversations found for auto-archiving.');
            return 0;
        }

        $this->info("Found {$conversations->count()} conversation(s) for auto-archiving:");

        $archivedCount = 0;
        $systemUser = $this->getSystemUser();

        foreach ($conversations as $conversation) {
            $resolvedAt = $conversation->resolved_at;
            $participantNames = $conversation->participants->pluck('name')->join(', ');
            
            $this->line("- ID: {$conversation->id} | Participants: {$participantNames} | Resolved: {$resolvedAt->format('Y-m-d H:i:s')}");

            if (!$isDryRun) {
                try {
                    if ($conversation->archive($systemUser, true)) {
                        $archivedCount++;
                        
                        // Log the auto-archiving
                        Log::info('Conversation auto-archived', [
                            'conversation_id' => $conversation->id,
                            'resolved_at' => $resolvedAt,
                            'participants' => $conversation->participants->pluck('id')->toArray()
                        ]);

                        // TODO: Send notification to participants about auto-archiving
                        // This will be implemented in the notification phase
                    }
                } catch (\Exception $e) {
                    $this->error("Failed to archive conversation {$conversation->id}: {$e->getMessage()}");
                    Log::error('Auto-archive failed', [
                        'conversation_id' => $conversation->id,
                        'error' => $e->getMessage()
                    ]);
                }
            }
        }

        if ($isDryRun) {
            $this->info("DRY RUN: Would archive {$conversations->count()} conversation(s).");
        } else {
            $this->info("Successfully auto-archived {$archivedCount} conversation(s).");
        }

        return 0;
    }

    /**
     * Get or create a system user for automated actions.
     */
    private function getSystemUser(): User
    {
        // Try to find an existing system/admin user
        $systemUser = User::where('user_type', 'super_admin')->first();
        
        if (!$systemUser) {
            $systemUser = User::where('user_type', 'admin')->first();
        }

        if (!$systemUser) {
            // If no admin exists, create a system user
            $systemUser = User::create([
                'name' => 'System',
                'email' => 'system@messaging.local',
                'user_type' => 'admin',
                'password' => bcrypt('system-auto-generated'),
                'email_verified_at' => now(),
            ]);
        }

        return $systemUser;
    }
}
