<?php

namespace App\Console\Commands;

use App\Models\BlogPost;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Storage;

class BlogRestoreCommand extends Command
{
    protected $signature = 'blog:restore {timestamp : Backup timestamp to restore}';
    
    protected $description = 'Restore blog posts from a previous backup';

    public function handle()
    {
        $timestamp = $this->argument('timestamp');
        $filename = "blog_backup_{$timestamp}.json";
        $path = "backups/blog/{$filename}";
        
        if (!Storage::disk('local')->exists($path)) {
            $this->error("Backup file not found: {$filename}");
            $this->info("Available backups:");
            
            $backups = Storage::disk('local')->files('backups/blog');
            foreach ($backups as $backup) {
                $this->info("  - " . basename($backup));
            }
            
            return 1;
        }

        $backupData = json_decode(Storage::disk('local')->get($path), true);
        
        $this->info("Backup created: {$backupData['created_at']}");
        $this->info("Posts in backup: {$backupData['post_count']}");
        $this->newLine();
        
        if (!$this->confirm('Are you sure you want to restore this backup? This will overwrite current data.')) {
            $this->info('Restore cancelled');
            return 0;
        }

        $restored = 0;
        $errors = [];

        foreach ($backupData['posts'] as $postData) {
            try {
                $post = BlogPost::find($postData['id']);
                
                if (!$post) {
                    $errors[] = "Post ID {$postData['id']} not found - skipped";
                    continue;
                }

                $post->update([
                    'content' => $postData['content'],
                    'structured_data' => $postData['structured_data'],
                    'excerpt' => $postData['excerpt'],
                    'meta_title' => $postData['meta_title'] ?? null,
                    'meta_description' => $postData['meta_description'] ?? null,
                    'meta_keywords' => $postData['meta_keywords'] ?? null,
                ]);

                $restored++;
                $this->info("✓ Restored: {$post->title}");
                
            } catch (\Exception $e) {
                $errors[] = "Error restoring post {$postData['id']}: {$e->getMessage()}";
            }
        }

        $this->newLine();
        $this->info("✓ Restoration completed!");
        $this->info("✓ {$restored} posts restored");
        
        if (!empty($errors)) {
            $this->newLine();
            $this->warn("Errors encountered:");
            foreach ($errors as $error) {
                $this->error("  - {$error}");
            }
        }

        return 0;
    }
}
