<?php

namespace App\Console\Commands;

use App\Models\Order;
use App\Services\OrderStatusService;
use App\Notifications\OrderApprovalReminderNotification;
use App\Notifications\WorkReadyForReviewNotification;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Log;

class ProcessOrderStatusChanges extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'orders:process-status-changes';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Process automatic order status changes like auto-approval and reminders';

    /**
     * Execute the console command.
     */
    public function handle(OrderStatusService $statusService)
    {
        $this->info('Processing order status changes...');

        // Process auto-approvals (48 hours)
        $this->processAutoApprovals($statusService);

        // Process reminders (24 hours)
        $this->processReminders($statusService);

        $this->info('Order status changes processed successfully!');
    }

    /**
     * Process orders that need auto-approval.
     */
    private function processAutoApprovals(OrderStatusService $statusService): void
    {
        $orders = $statusService->getOrdersNeedingAutoApproval();
        
        if ($orders->isEmpty()) {
            $this->info('No orders need auto-approval.');
            return;
        }

        $this->info("Processing {$orders->count()} orders for auto-approval...");

        foreach ($orders as $order) {
            try {
                $statusService->handleAutoApproval($order);
                $this->info("Order #{$order->order_number} auto-approved.");
                
                // Log the auto-approval
                Log::info("Order #{$order->order_number} auto-approved after 48 hours", [
                    'order_id' => $order->id,
                    'client_id' => $order->user_id,
                    'writer_id' => $order->writer_id,
                ]);
            } catch (\Exception $e) {
                $this->error("Failed to auto-approve order #{$order->order_number}: {$e->getMessage()}");
                Log::error("Failed to auto-approve order #{$order->order_number}", [
                    'order_id' => $order->id,
                    'error' => $e->getMessage(),
                ]);
            }
        }
    }

    /**
     * Process orders that need reminders.
     */
    private function processReminders(OrderStatusService $statusService): void
    {
        $orders = $statusService->getOrdersNeedingReminder();
        
        if ($orders->isEmpty()) {
            $this->info('No orders need reminders.');
            return;
        }

        $this->info("Processing {$orders->count()} orders for reminders...");

        foreach ($orders as $order) {
            try {
                // Calculate hours remaining
                $hoursRemaining = 48 - $order->client_review_started_at->diffInHours(now());
                
                // Send reminder notification to client
                $client = $order->user;
                if ($client) {
                    $client->notify(new OrderApprovalReminderNotification($order, $hoursRemaining));
                    $this->info("Reminder sent for order #{$order->order_number} ({$hoursRemaining} hours remaining).");
                }
                
                // Log the reminder
                Log::info("Reminder sent for order #{$order->order_number}", [
                    'order_id' => $order->id,
                    'client_id' => $order->user_id,
                    'hours_remaining' => $hoursRemaining,
                ]);
            } catch (\Exception $e) {
                $this->error("Failed to send reminder for order #{$order->order_number}: {$e->getMessage()}");
                Log::error("Failed to send reminder for order #{$order->order_number}", [
                    'order_id' => $order->id,
                    'error' => $e->getMessage(),
                ]);
            }
        }
    }
}
