<?php

namespace App\Console\Commands;

use App\Models\Order;
use App\Models\User;
use App\Notifications\WorkSubmittedNotification;
use App\Notifications\WorkSubmittedAdminNotification;
use App\Notifications\OrderApprovedNotification;
use App\Notifications\RevisionRequestedNotification;
use App\Notifications\OrderApprovalReminderNotification;
use App\Notifications\WorkReadyForReviewNotification;
use App\Services\AdminNotificationService;
use Illuminate\Console\Command;

class TestOrderNotifications extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'test:order-notifications {--order-id=} {--type=all}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Test order notifications for development purposes';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $orderId = $this->option('order-id');
        $type = $this->option('type');

        if (!$orderId) {
            $this->error('Please provide an order ID using --order-id option');
            return 1;
        }

        $order = Order::find($orderId);
        if (!$order) {
            $this->error("Order with ID {$orderId} not found");
            return 1;
        }

        $this->info("Testing notifications for Order #{$order->order_number}");

        switch ($type) {
            case 'work-submitted':
                $this->testWorkSubmittedNotification($order);
                break;
            case 'work-ready':
                $this->testWorkReadyForReviewNotification($order);
                break;
            case 'approved':
                $this->testOrderApprovedNotification($order);
                break;
            case 'revision':
                $this->testRevisionRequestedNotification($order);
                break;
            case 'reminder':
                $this->testOrderApprovalReminderNotification($order);
                break;
            case 'all':
            default:
                $this->testAllNotifications($order);
                break;
        }

        $this->info('Notification testing completed!');
        return 0;
    }

    private function testWorkSubmittedNotification(Order $order)
    {
        $this->info('Testing WorkSubmittedNotification...');
        
        $client = $order->user;
        if ($client) {
            $client->notify(new WorkSubmittedNotification($order));
            $this->info("✓ WorkSubmittedNotification sent to client: {$client->email}");
        }

        AdminNotificationService::notifyAllAdmins(new WorkSubmittedAdminNotification($order));
        $this->info('✓ WorkSubmittedAdminNotification sent to all admins');
    }

    private function testWorkReadyForReviewNotification(Order $order)
    {
        $this->info('Testing WorkReadyForReviewNotification...');
        
        $client = $order->user;
        if ($client) {
            $client->notify(new WorkReadyForReviewNotification($order));
            $this->info("✓ WorkReadyForReviewNotification sent to client: {$client->email}");
        }
    }

    private function testOrderApprovedNotification(Order $order)
    {
        $this->info('Testing OrderApprovedNotification...');
        
        $writer = $order->writer;
        if ($writer) {
            $writer->notify(new OrderApprovedNotification($order));
            $this->info("✓ OrderApprovedNotification sent to writer: {$writer->email}");
        }
    }

    private function testRevisionRequestedNotification(Order $order)
    {
        $this->info('Testing RevisionRequestedNotification...');
        
        $writer = $order->writer;
        if ($writer) {
            $writer->notify(new RevisionRequestedNotification($order, 'Test revision notes'));
            $this->info("✓ RevisionRequestedNotification sent to writer: {$writer->email}");
        }
    }

    private function testOrderApprovalReminderNotification(Order $order)
    {
        $this->info('Testing OrderApprovalReminderNotification...');
        
        $client = $order->user;
        if ($client) {
            $client->notify(new OrderApprovalReminderNotification($order, 24));
            $this->info("✓ OrderApprovalReminderNotification sent to client: {$client->email}");
        }
    }

    private function testAllNotifications(Order $order)
    {
        $this->info('Testing all notifications...');
        
        $this->testWorkSubmittedNotification($order);
        $this->testWorkReadyForReviewNotification($order);
        $this->testOrderApprovedNotification($order);
        $this->testRevisionRequestedNotification($order);
        $this->testOrderApprovalReminderNotification($order);
    }
}
