<?php

namespace App\Console\Commands;

use App\Models\Order;
use Illuminate\Console\Command;

class UpdateAllOrderDeadlines extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'orders:update-all-deadlines 
                            {--source-order= : The order number to copy deadlines from}
                            {--client-deadline= : The client deadline to set (YYYY-MM-DD HH:MM:SS)}
                            {--writer-deadline= : The writer deadline to set (YYYY-MM-DD HH:MM:SS)}
                            {--force : Skip confirmation prompt}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Update client and writer deadlines for all orders';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $sourceOrderNumber = $this->option('source-order');
        $clientDeadline = $this->option('client-deadline');
        $writerDeadline = $this->option('writer-deadline');

        // Get deadlines from source order if specified
        if ($sourceOrderNumber) {
            $sourceOrder = Order::where('order_number', $sourceOrderNumber)->first();
            
            if (!$sourceOrder) {
                $this->error("Order #{$sourceOrderNumber} not found!");
                return 1;
            }

            $clientDeadline = $sourceOrder->deadline;
            $writerDeadline = $sourceOrder->writer_deadline;

            $this->info("Found source order #{$sourceOrderNumber}:");
            $this->line("  Client Deadline: {$clientDeadline}");
            $this->line("  Writer Deadline: {$writerDeadline}");
            $this->newLine();
        }

        // Validate that we have deadlines
        if (!$clientDeadline || !$writerDeadline) {
            $this->error('Both client deadline and writer deadline must be specified!');
            $this->line('Use either --source-order or both --client-deadline and --writer-deadline options.');
            return 1;
        }

        // Get all orders
        $orders = Order::all();
        $totalOrders = $orders->count();

        if ($totalOrders === 0) {
            $this->warn('No orders found in the database.');
            return 0;
        }

        // Show what will be updated
        $this->warn("⚠️  WARNING: This will update ALL {$totalOrders} orders!");
        $this->newLine();
        $this->table(
            ['Field', 'New Value'],
            [
                ['Client Deadline', $clientDeadline],
                ['Writer Deadline', $writerDeadline],
            ]
        );
        $this->newLine();

        // Confirm unless --force is used
        if (!$this->option('force')) {
            if (!$this->confirm('Do you want to proceed with updating all orders?', false)) {
                $this->info('Operation cancelled.');
                return 0;
            }
        }

        // Perform the update
        $this->info('Updating orders...');
        $progressBar = $this->output->createProgressBar($totalOrders);
        $progressBar->start();

        $updatedCount = 0;
        $errors = [];

        foreach ($orders as $order) {
            try {
                $order->deadline = $clientDeadline;
                $order->writer_deadline = $writerDeadline;
                $order->save();
                
                $updatedCount++;
                
                \Log::info('Bulk deadline update', [
                    'order_id' => $order->id,
                    'order_number' => $order->order_number,
                    'new_client_deadline' => $clientDeadline,
                    'new_writer_deadline' => $writerDeadline,
                ]);
            } catch (\Exception $e) {
                $errors[] = [
                    'order_number' => $order->order_number,
                    'error' => $e->getMessage(),
                ];
                
                \Log::error('Failed to update order deadline', [
                    'order_id' => $order->id,
                    'order_number' => $order->order_number,
                    'error' => $e->getMessage(),
                ]);
            }

            $progressBar->advance();
        }

        $progressBar->finish();
        $this->newLine(2);

        // Show results
        $this->info("✓ Successfully updated {$updatedCount} out of {$totalOrders} orders.");

        if (count($errors) > 0) {
            $this->newLine();
            $this->error('Errors occurred for the following orders:');
            $this->table(
                ['Order Number', 'Error'],
                $errors
            );
        }

        $this->newLine();
        $this->info('Operation completed!');
        
        return 0;
    }
}

