<?php

namespace App\Http\Controllers\Admin;

use Carbon\Carbon;
use App\Models\User;
use Inertia\Inertia;
use App\Models\Order;
use App\Models\Coupon;
use App\Models\LoyaltyTier;
use App\Models\SeasonalType;
use Illuminate\Http\Request;
use App\Services\CouponService;
use Illuminate\Validation\Rule;
use Illuminate\Support\Facades\Log;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;

class CouponController extends Controller
{
    protected $couponService;

    public function __construct(CouponService $couponService)
    {
        $this->couponService = $couponService;
    }

    /**
     * Display a listing of the coupons.
     */
    public function index(Request $request)
    {
        $filters = $request->only([
            'search',
            'type',
            'status',
            'sort',
            'direction',
            'date_range'
        ]);

        $query = Coupon::query()
            ->with(['loyaltyTier', 'seasonalType', 'referrer', 'creator', 'updater'])
            ->withCount(['userUsages', 'orders']);

        // Apply search filter
        if (!empty($filters['search'])) {
            $search = $filters['search'];
            $query->where(function ($q) use ($search) {
                $q->where('coupon_code', 'like', "%{$search}%")
                    ->orWhere('coupon_type', 'like', "%{$search}%")
                    ->orWhere('discount_amount', 'like', "%{$search}%");
            });
        }

        // Apply type filter
        if (!empty($filters['type'])) {
            $query->where('coupon_type', $filters['type']);
        }

        // Apply status filter
        if (isset($filters['status'])) {
            $status = $filters['status'] === 'active';
            $query->where('is_active', $status);
        }

        // Apply date range filter
        if (!empty($filters['date_range'])) {
            $dates = explode(' to ', $filters['date_range']);
            if (count($dates) === 2) {
                $startDate = Carbon::parse($dates[0])->startOfDay();
                $endDate = Carbon::parse($dates[1])->endOfDay();

                $query->where(function ($q) use ($startDate, $endDate) {
                    $q->whereBetween('created_at', [$startDate, $endDate])
                        ->orWhereBetween('updated_at', [$startDate, $endDate]);
                });
            }
        }

        // Apply sorting
        $sortField = $filters['sort'] ?? 'created_at';
        $direction = $filters['direction'] ?? 'desc';
        $allowedSorts = ['coupon_code', 'coupon_type', 'discount_amount', 'created_at', 'updated_at', 'uses_count', 'expiry_date'];

        if (in_array($sortField, $allowedSorts)) {
            $query->orderBy($sortField, $direction);
        } else {
            $query->orderBy('created_at', 'desc');
        }

        // $coupons = $query->paginate(15)->withQueryString();
        $coupons = $query->paginate(15)->appends(request()->query());

        // Get coupon types for filter dropdown
        $couponTypes = [
            Coupon::TYPE_FIRST_ORDER => 'First Order',
            Coupon::TYPE_FLASH => 'Flash Sale',
            Coupon::TYPE_REFERRAL => 'Referral',
            Coupon::TYPE_LOYALTY => 'Loyalty',
            Coupon::TYPE_ANNIVERSARY => 'Anniversary',
            Coupon::TYPE_SEASONAL => 'Seasonal',
            Coupon::TYPE_VOLUME => 'Volume',
        ];

        return Inertia::render('Admin/Coupons/Index', [
            'coupons' => $coupons,
            'filters' => $filters,
            'couponTypes' => $couponTypes,
        ]);
    }

    /**
     * Show the form for creating a new coupon.
     */
    public function create()
    {
        $loyaltyTiers = LoyaltyTier::active()->get();
        $seasonalTypes = SeasonalType::active()->get();

        $couponTypes = [
            Coupon::TYPE_FIRST_ORDER => 'First Order',
            Coupon::TYPE_FLASH => 'Flash Sale',
            Coupon::TYPE_REFERRAL => 'Referral',
            Coupon::TYPE_LOYALTY => 'Loyalty',
            Coupon::TYPE_ANNIVERSARY => 'Anniversary',
            Coupon::TYPE_SEASONAL => 'Seasonal',
            Coupon::TYPE_VOLUME => 'Volume',
        ];

        $discountTypes = [
            Coupon::DISCOUNT_FIXED => 'Fixed Amount',
            Coupon::DISCOUNT_PERCENTAGE => 'Percentage',
        ];

        return Inertia::render('Admin/Coupons/Create', [
            'couponTypes' => $couponTypes,
            'discountTypes' => $discountTypes,
            'loyaltyTiers' => $loyaltyTiers,
            'seasonalTypes' => $seasonalTypes,
        ]);
    }

    /**
     * Store a newly created coupon in storage.
     */
    public function store(Request $request)
    {
        try {
            $data = $this->validateCoupon($request);

            // Add creator information
            $data['created_by'] = Auth::id();

            // Generate unique code if not provided
            if (empty($data['coupon_code'])) {
                $data['coupon_code'] = Coupon::generateUniqueCode();
            }

            // Convert date strings to Carbon instances
            if (!empty($data['start_date'])) {
                $data['start_date'] = Carbon::parse($data['start_date']);
            }

            if (!empty($data['expiry_date'])) {
                $data['expiry_date'] = Carbon::parse($data['expiry_date']);
            }

            // Set uses_count to 0 for new coupons
            $data['uses_count'] = 0;

            $coupon = Coupon::create($data);

            return redirect()->route('admin.coupons.index')
                ->with('success', 'Coupon created successfully.');
        } catch (\Exception $e) {
            Log::error('Failed to create coupon: ' . $e->getMessage(), [
                'request' => $request->all(),
                'trace' => $e->getTraceAsString()
            ]);

            return redirect()->back()
                ->withInput()
                ->with('error', 'Failed to create coupon: ' . $e->getMessage());
        }
    }

    /**
     * Show the form for editing the specified coupon.
     */
    public function edit(Coupon $coupon)
    {
        $coupon->load(['loyaltyTier', 'seasonalType', 'referrer', 'creator', 'updater']);

        $loyaltyTiers = LoyaltyTier::active()->get();
        $seasonalTypes = SeasonalType::active()->get();

        $couponTypes = [
            Coupon::TYPE_FIRST_ORDER => 'First Order',
            Coupon::TYPE_FLASH => 'Flash Sale',
            Coupon::TYPE_REFERRAL => 'Referral',
            Coupon::TYPE_LOYALTY => 'Loyalty',
            Coupon::TYPE_ANNIVERSARY => 'Anniversary',
            Coupon::TYPE_SEASONAL => 'Seasonal',
            Coupon::TYPE_VOLUME => 'Volume',
        ];

        $discountTypes = [
            Coupon::DISCOUNT_FIXED => 'Fixed Amount',
            Coupon::DISCOUNT_PERCENTAGE => 'Percentage',
        ];

        return Inertia::render('Admin/Coupons/Edit', [
            'coupon' => $coupon,
            'couponTypes' => $couponTypes,
            'discountTypes' => $discountTypes,
            'loyaltyTiers' => $loyaltyTiers,
            'seasonalTypes' => $seasonalTypes,
        ]);
    }

    /**
     * Update the specified coupon in storage.
     */
    public function update(Request $request, Coupon $coupon)
    {
        try {
            $data = $this->validateCoupon($request, $coupon->id);

            // Handle null values for unlimited usage
            if (isset($data['per_user_limit'])) {
                $data['per_user_limit'] = ($data['per_user_limit'] === "" || $data['per_user_limit'] === null || $data['per_user_limit'] === 0) ? null : $data['per_user_limit'];
            }
            
            if (isset($data['max_uses'])) {
                $data['max_uses'] = ($data['max_uses'] === "" || $data['max_uses'] === null || $data['max_uses'] === 0) ? null : $data['max_uses'];
            }

            // Add updater information
            $data['updated_by'] = Auth::id();

            // Convert date strings to Carbon instances
            if (!empty($data['start_date'])) {
                $data['start_date'] = Carbon::parse($data['start_date']);
            }

            if (!empty($data['expiry_date'])) {
                $data['expiry_date'] = Carbon::parse($data['expiry_date']);
            }

            $coupon->update($data);

            // Build success message with usage details
            $usageInfo = [];
            if ($coupon->per_user_limit === null) {
                $usageInfo[] = "unlimited uses per user";
            } else {
                $usageInfo[] = "{$coupon->per_user_limit} use(s) per user";
            }
            
            if ($coupon->max_uses === null) {
                $usageInfo[] = "unlimited total uses";
            } else {
                $usageInfo[] = "{$coupon->max_uses} total uses";
            }
            
            $detailedMessage = "Coupon '{$coupon->coupon_code}' updated successfully! Settings: " . implode(", ", $usageInfo) . ".";

            return redirect()->back()
                ->with('success', 'Coupon updated successfully.')
                ->with('message', $detailedMessage);
        } catch (\Exception $e) {
            Log::error('Failed to update coupon: ' . $e->getMessage(), [
                'coupon_id' => $coupon->id,
                'request' => $request->all(),
                'trace' => $e->getTraceAsString()
            ]);

            return redirect()->back()
                ->withInput()
                ->with('error', 'Failed to update coupon: ' . $e->getMessage());
        }
    }

    /**
     * Toggle the active status of the specified coupon.
     */
    public function toggleStatus(Coupon $coupon)
    {
        try {
            $coupon->is_active = !$coupon->is_active;
            $coupon->updated_by = Auth::id();
            $coupon->save();

            $status = $coupon->is_active ? 'activated' : 'deactivated';

            return redirect()->back()
                ->with('success', "Coupon {$status} successfully.");
        } catch (\Exception $e) {
            Log::error('Failed to toggle coupon status: ' . $e->getMessage(), [
                'coupon_id' => $coupon->id,
                'trace' => $e->getTraceAsString()
            ]);

            return redirect()->back()
                ->with('error', 'Failed to update coupon status.');
        }
    }

    /**
     * Remove the specified coupon from storage.
     */
    public function destroy(Coupon $coupon)
    {
        try {
            // Record who deleted this coupon before it's deleted
            $coupon->deleted_by = Auth::id();
            $coupon->save();

            $coupon->delete();

            return redirect()->route('admin.coupons.index')
                ->with('success', 'Coupon deleted successfully.');
        } catch (\Exception $e) {
            Log::error('Failed to delete coupon: ' . $e->getMessage(), [
                'coupon_id' => $coupon->id,
                'trace' => $e->getTraceAsString()
            ]);

            return redirect()->back()
                ->with('error', 'Failed to delete coupon.');
        }
    }

    /**
     * Display the form for generating loyalty coupons.
     */
    public function showGenerateLoyaltyCoupons()
    {
        $loyaltyTiers = LoyaltyTier::active()->get();
        $discountTypes = [
            Coupon::DISCOUNT_FIXED => 'Fixed Amount',
            Coupon::DISCOUNT_PERCENTAGE => 'Percentage',
        ];

        return Inertia::render('Admin/Coupons/GenerateLoyalty', [
            'loyaltyTiers' => $loyaltyTiers,
            'discountTypes' => $discountTypes,
        ]);
    }

    /**
     * Generate loyalty coupons for all tiers.
     */
    public function generateLoyaltyCoupons(Request $request)
    {
        try {
            $request->validate([
                'discount_type' => ['required', Rule::in([Coupon::DISCOUNT_FIXED, Coupon::DISCOUNT_PERCENTAGE])],
                'tier_discounts' => 'required|array',
                'prefix' => 'nullable|string|max:10',
                'start_date' => 'nullable|date',
                'expiry_date' => 'nullable|date|after_or_equal:start_date',
                'max_uses' => 'nullable|integer|min:1',
            ]);

            $options = [
                'prefix' => $request->prefix,
                'start_date' => $request->filled('start_date') ? Carbon::parse($request->start_date) : now(),
                'expiry_date' => $request->filled('expiry_date') ? Carbon::parse($request->expiry_date) : null,
                'max_uses' => $request->max_uses,
                'created_by' => Auth::id(),
            ];

            $coupons = $this->couponService->generateLoyaltyCoupons(
                $request->discount_type,
                $request->tier_discounts,
                $options
            );

            $count = $coupons->count();

            return redirect()->route('admin.coupons.index')
                ->with('success', "{$count} loyalty coupons generated successfully.");
        } catch (\Exception $e) {
            Log::error('Failed to generate loyalty coupons: ' . $e->getMessage(), [
                'request' => $request->all(),
                'trace' => $e->getTraceAsString()
            ]);

            return redirect()->back()
                ->withInput()
                ->with('error', 'Failed to generate loyalty coupons: ' . $e->getMessage());
        }
    }

    /**
     * Display the form for generating seasonal coupons.
     */
    public function showGenerateSeasonalCoupons()
    {
        $seasonalTypes = SeasonalType::active()->get();
        $discountTypes = [
            Coupon::DISCOUNT_FIXED => 'Fixed Amount',
            Coupon::DISCOUNT_PERCENTAGE => 'Percentage',
        ];

        return Inertia::render('Admin/Coupons/GenerateSeasonal', [
            'seasonalTypes' => $seasonalTypes,
            'discountTypes' => $discountTypes,
        ]);
    }

    /**
     * Generate seasonal coupons for all types.
     */
    public function generateSeasonalCoupons(Request $request)
    {
        try {
            $request->validate([
                'discount_type' => ['required', Rule::in([Coupon::DISCOUNT_FIXED, Coupon::DISCOUNT_PERCENTAGE])],
                'seasonal_discounts' => 'required|array',
                'prefix' => 'nullable|string|max:10',
                'start_date' => 'nullable|date',
                'expiry_date' => 'nullable|date|after_or_equal:start_date',
                'max_uses' => 'nullable|integer|min:1',
            ]);
            $options = [
                'prefix' => $request->prefix,
                'start_date' => $request->filled('start_date') ? Carbon::parse($request->start_date) : null,
                'expiry_date' => $request->filled('expiry_date') ? Carbon::parse($request->expiry_date) : null,
                'max_uses' => $request->max_uses,
                'created_by' => Auth::id(),
            ];

            $coupons = $this->couponService->generateSeasonalCoupons(
                $request->discount_type,
                $request->seasonal_discounts,
                $options
            );

            $count = $coupons->count();

            return redirect()->route('admin.coupons.index')
                ->with('success', "{$count} seasonal coupons generated successfully.");
        } catch (\Exception $e) {
            Log::error('Failed to generate seasonal coupons: ' . $e->getMessage(), [
                'request' => $request->all(),
                'trace' => $e->getTraceAsString()
            ]);

            return redirect()->back()
                ->withInput()
                ->with('error', 'Failed to generate seasonal coupons: ' . $e->getMessage());
        }
    }

    /**
     * Display the form for generating flash coupons.
     */
    public function showGenerateFlashCoupon()
    {
        $discountTypes = [
            Coupon::DISCOUNT_FIXED => 'Fixed Amount',
            Coupon::DISCOUNT_PERCENTAGE => 'Percentage',
        ];

        return Inertia::render('Admin/Coupons/GenerateFlash', [
            'discountTypes' => $discountTypes,
        ]);
    }

    /**
     * Generate a flash sale coupon.
     */
    public function generateFlashCoupon(Request $request)
    {
        try {
            $request->validate([
                'discount_type' => ['required', Rule::in([Coupon::DISCOUNT_FIXED, Coupon::DISCOUNT_PERCENTAGE])],
                'discount_amount' => 'required|numeric|min:0.01',
                'prefix' => 'nullable|string|max:10',
                'duration_hours' => 'required|integer|min:1|max:720', // Max 30 days
                'start_date' => 'required|date',
                'max_uses' => 'nullable|integer|min:1',
                'per_user_limit' => 'nullable|integer|min:1',
                'min_order_amount' => 'nullable|numeric|min:0',
            ]);

            $startDate = Carbon::parse($request->start_date);
            $expiryDate = $startDate->copy()->addHours($request->duration_hours);

            $options = [
                'prefix' => $request->prefix,
                'start_date' => $startDate,
                'per_user_limit' => $request->per_user_limit,
                'min_order_amount' => $request->min_order_amount,
                'max_uses' => $request->max_uses,
                'created_by' => Auth::id(),
            ];

            $coupon = $this->couponService->generateFlashCoupon(
                $request->discount_type,
                $request->discount_amount,
                $expiryDate,
                $options
            );

            return redirect()->route('admin.coupons.index')
                ->with('success', "Flash sale coupon generated successfully. Code: {$coupon->coupon_code}");
        } catch (\Exception $e) {
            Log::error('Failed to generate flash coupon: ' . $e->getMessage(), [
                'request' => $request->all(),
                'trace' => $e->getTraceAsString()
            ]);

            return redirect()->back()
                ->withInput()
                ->with('error', 'Failed to generate flash coupon: ' . $e->getMessage());
        }
    }

    /**
     * Display the form for generating first order coupon.
     */
    public function showGenerateFirstOrderCoupon()
    {
        $discountTypes = [
            Coupon::DISCOUNT_FIXED => 'Fixed Amount',
            Coupon::DISCOUNT_PERCENTAGE => 'Percentage',
        ];

        return Inertia::render('Admin/Coupons/GenerateFirstOrder', [
            'discountTypes' => $discountTypes,
        ]);
    }

    /**
     * Generate a first order coupon.
     */
    public function generateFirstOrderCoupon(Request $request)
    {
        try {
            $request->validate([
                'discount_type' => ['required', Rule::in([Coupon::DISCOUNT_FIXED, Coupon::DISCOUNT_PERCENTAGE])],
                'discount_amount' => 'required|numeric|min:0.01',
                'prefix' => 'nullable|string|max:10',
                'start_date' => 'nullable|date',
                'expiry_date' => 'nullable|date|after_or_equal:start_date',
                'max_uses' => 'nullable|integer|min:1',
                'min_order_amount' => 'nullable|numeric|min:0',
            ]);

            $options = [
                'prefix' => $request->prefix,
                'start_date' => $request->filled('start_date') ? Carbon::parse($request->start_date) : now(),
                'expiry_date' => $request->filled('expiry_date') ? Carbon::parse($request->expiry_date) : null,
                'max_uses' => $request->max_uses,
                'min_order_amount' => $request->min_order_amount,
                'created_by' => Auth::id(),
            ];

            $coupon = $this->couponService->generateFirstOrderCoupon(
                $request->discount_type,
                $request->discount_amount,
                $options
            );

            return redirect()->route('admin.coupons.index')
                ->with('success', "First order coupon generated successfully. Code: {$coupon->coupon_code}");
        } catch (\Exception $e) {
            Log::error('Failed to generate first order coupon: ' . $e->getMessage(), [
                'request' => $request->all(),
                'trace' => $e->getTraceAsString()
            ]);

            return redirect()->back()
                ->withInput()
                ->with('error', 'Failed to generate first order coupon: ' . $e->getMessage());
        }
    }

    /**
     * Display the form for generating volume coupons.
     */
    public function showGenerateVolumeCoupons()
    {
        $discountTypes = [
            Coupon::DISCOUNT_FIXED => 'Fixed Amount',
            Coupon::DISCOUNT_PERCENTAGE => 'Percentage',
        ];

        return Inertia::render('Admin/Coupons/GenerateVolume', [
            'discountTypes' => $discountTypes,
        ]);
    }

    /**
     * Generate volume discount coupons.
     */
    public function generateVolumeCoupons(Request $request)
    {
        try {
            $request->validate([
                'discount_type' => ['required', Rule::in([Coupon::DISCOUNT_FIXED, Coupon::DISCOUNT_PERCENTAGE])],
                'volume_thresholds' => 'required|array',
                'volume_thresholds.*.pages' => 'required|integer|min:1',
                'volume_thresholds.*.discount' => 'required|numeric|min:0.01',
                'prefix' => 'nullable|string|max:10',
                'start_date' => 'nullable|date',
                'expiry_date' => 'nullable|date|after_or_equal:start_date',
                'max_uses' => 'nullable|integer|min:1',
            ]);

            $volumeThresholds = [];
            foreach ($request->volume_thresholds as $threshold) {
                $volumeThresholds[$threshold['pages']] = $threshold['discount'];
            }

            $options = [
                'prefix' => $request->prefix,
                'start_date' => $request->filled('start_date') ? Carbon::parse($request->start_date) : now(),
                'expiry_date' => $request->filled('expiry_date') ? Carbon::parse($request->expiry_date) : null,
                'max_uses' => $request->max_uses,
                'created_by' => Auth::id(),
            ];

            $coupons = $this->couponService->generateVolumeCoupons(
                $volumeThresholds,
                $request->discount_type,
                $options
            );

            $count = $coupons->count();

            return redirect()->route('admin.coupons.index')
                ->with('success', "{$count} volume discount coupons generated successfully.");
        } catch (\Exception $e) {
            Log::error('Failed to generate volume coupons: ' . $e->getMessage(), [
                'request' => $request->all(),
                'trace' => $e->getTraceAsString()
            ]);

            return redirect()->back()
                ->withInput()
                ->with('error', 'Failed to generate volume coupons: ' . $e->getMessage());
        }
    }

    /**
     * Display the form for generating referral coupon.
     */
    public function showGenerateReferralCoupon()
    {
        $discountTypes = [
            Coupon::DISCOUNT_FIXED => 'Fixed Amount',
            Coupon::DISCOUNT_PERCENTAGE => 'Percentage',
        ];

        $clients = User::where('user_type', 'client')
            ->where('status', 'active')
            ->select('id', 'name', 'email')
            ->orderBy('name')
            ->get();

        return Inertia::render('Admin/Coupons/GenerateReferral', [
            'discountTypes' => $discountTypes,
            'clients' => $clients,
        ]);
    }

    /**
     * Generate a referral coupon.
     */
    public function generateReferralCoupon(Request $request)
    {
        try {
            $request->validate([
                'discount_type' => ['required', Rule::in([Coupon::DISCOUNT_FIXED, Coupon::DISCOUNT_PERCENTAGE])],
                'discount_amount' => 'required|numeric|min:0.01',
                'prefix' => 'nullable|string|max:10',
                'start_date' => 'nullable|date',
                'expiry_date' => 'nullable|date|after_or_equal:start_date',
                'max_uses' => 'nullable|integer|min:1',
                'per_user_limit' => 'nullable|integer|min:1',
                'referrer_user_id' => 'nullable|exists:users,id',
                'user_specific' => 'boolean',
            ]);

            $options = [
                'prefix' => $request->prefix,
                'start_date' => $request->filled('start_date') ? Carbon::parse($request->start_date) : now(),
                'expiry_date' => $request->filled('expiry_date') ? Carbon::parse($request->expiry_date) : null,
                'max_uses' => $request->max_uses,
                'per_user_limit' => $request->per_user_limit,
                'created_by' => Auth::id(),
            ];

            $referrerId = $request->user_specific ? $request->referrer_user_id : null;

            $coupon = $this->couponService->generateReferralCoupon(
                $referrerId,
                $request->discount_type,
                $request->discount_amount,
                $options
            );

            return redirect()->route('admin.coupons.index')
                ->with('success', "Referral coupon generated successfully. Code: {$coupon->coupon_code}");
        } catch (\Exception $e) {
            Log::error('Failed to generate referral coupon: ' . $e->getMessage(), [
                'request' => $request->all(),
                'trace' => $e->getTraceAsString()
            ]);

            return redirect()->back()
                ->withInput()
                ->with('error', 'Failed to generate referral coupon: ' . $e->getMessage());
        }
    }

    /**
     * Display the form for generating anniversary coupon.
     */
    public function showGenerateAnniversaryCoupon()
    {
        $discountTypes = [
            Coupon::DISCOUNT_FIXED => 'Fixed Amount',
            Coupon::DISCOUNT_PERCENTAGE => 'Percentage',
        ];

        return Inertia::render('Admin/Coupons/GenerateAnniversary', [
            'discountTypes' => $discountTypes,
        ]);
    }

    /**
     * Generate an anniversary coupon.
     */
    public function generateAnniversaryCoupon(Request $request)
    {
        try {
            $validated = $request->validate([
                'discount_type' => ['required', Rule::in([Coupon::DISCOUNT_FIXED, Coupon::DISCOUNT_PERCENTAGE])],
                'discount_amount' => 'required|numeric|min:0.01',
                'prefix' => 'nullable|string|max:10',
                'years_active' => 'required|integer|min:1',
                'valid_days' => 'required|integer|min:1|max:30',
                'max_uses' => 'nullable|integer|min:1',
            ]);

            // Cast string inputs to appropriate types
            $validDays = (int)$validated['valid_days'];
            $yearsActive = (int)$validated['years_active'];
            $maxUses = isset($validated['max_uses']) ? (int)$validated['max_uses'] : null;

            $options = [
                'prefix' => $validated['prefix'],
                'start_date' => now(),
                'expiry_date' => now()->addDays($validDays), // Using the cast integer
                'max_uses' => $maxUses,
                'created_by' => Auth::id(),
            ];

            $coupon = $this->couponService->generateAnniversaryCoupon(
                $yearsActive,
                $validated['discount_type'],
                (float)$validated['discount_amount'],
                $options
            );

            return redirect()->route('admin.coupons.index')
                ->with('success', "Anniversary coupon generated successfully. Code: {$coupon->coupon_code}");
        } catch (\Exception $e) {
            Log::error('Failed to generate anniversary coupon: ' . $e->getMessage(), [
                'request' => $request->all(),
                'trace' => $e->getTraceAsString()
            ]);

            return redirect()->back()
                ->withInput()
                ->with('error', 'Failed to generate anniversary coupon: ' . $e->getMessage());
        }
    }



    /**
     * Show coupon usage statistics
     */
    public function showStatistics(Request $request)
    {
        // Get date range filter
        $startDate = $request->input('start_date') ? Carbon::parse($request->input('start_date')) : Carbon::now()->subDays(30);
        $endDate = $request->input('end_date') ? Carbon::parse($request->input('end_date')) : Carbon::now();

        // Get coupons used within date range
        $usageQuery = \App\Models\UserCoupon::whereBetween('used_at', [$startDate, $endDate])
            ->with(['coupon', 'user'])
            ->selectRaw('coupon_id, COUNT(*) as usage_count, SUM(discount_amount) as total_discount')
            ->groupBy('coupon_id');

        $couponUsageData = $usageQuery->get();

        // Get most used coupon types
        $couponTypeStats = \App\Models\UserCoupon::whereBetween('used_at', [$startDate, $endDate])
            ->join('coupons', 'user_coupons.coupon_id', '=', 'coupons.id')
            ->selectRaw('coupons.coupon_type, COUNT(*) as usage_count, SUM(user_coupons.discount_amount) as total_discount')
            ->groupBy('coupons.coupon_type')
            ->orderBy('usage_count', 'desc')
            ->get();

        // Get daily usage data for chart
        $dailyStats = \App\Models\UserCoupon::whereBetween('used_at', [$startDate, $endDate])
            ->selectRaw('DATE(used_at) as date, COUNT(*) as usage_count, SUM(discount_amount) as total_discount')
            ->groupBy('date')
            ->orderBy('date')
            ->get();

        // Get top users by discount amount
        $topUsers = \App\Models\UserCoupon::whereBetween('used_at', [$startDate, $endDate])
            ->with('user:id,name,email')
            ->selectRaw('user_id, COUNT(*) as usage_count, SUM(discount_amount) as total_discount')
            ->groupBy('user_id')
            ->orderBy('total_discount', 'desc')
            ->limit(10)
            ->get();

        return Inertia::render('Admin/Coupons/Statistics', [
            'couponUsageData' => $couponUsageData,
            'couponTypeStats' => $couponTypeStats,
            'dailyStats' => $dailyStats,
            'topUsers' => $topUsers,
            'startDate' => $startDate->format('Y-m-d'),
            'endDate' => $endDate->format('Y-m-d'),
        ]);
    }

    /**
     * Show details for a specific coupon including usage history
     */
    public function show(Coupon $coupon)
    {
        $coupon->load(['loyaltyTier', 'seasonalType', 'referrer', 'creator', 'updater']);

        // Get usage history
        $usageHistory = \App\Models\UserCoupon::where('coupon_id', $coupon->id)
            ->with(['user:id,name,email', 'order:id,order_number'])
            ->orderBy('used_at', 'desc')
            ->paginate(15);

        // Get orders that used this coupon
        $orders = Order::where('coupon_id', $coupon->id)
            ->with('user')
            ->orderBy('id', 'desc')
            ->paginate(15);

        // Get stats
        $stats = [
            'total_uses' => $coupon->uses_count,
            'total_discount' => \App\Models\UserCoupon::where('coupon_id', $coupon->id)->sum('discount_amount'),
            'average_discount' => \App\Models\UserCoupon::where('coupon_id', $coupon->id)->avg('discount_amount') ?? 0,
            'unique_users' => \App\Models\UserCoupon::where('coupon_id', $coupon->id)->distinct('user_id')->count('user_id'),
            'total_revenue' => Order::where('coupon_id', $coupon->id)->sum('order_amount') ?? 0,
        ];

        // Get coupon types for display
        $couponTypes = [
            Coupon::TYPE_FIRST_ORDER => 'First Order',
            Coupon::TYPE_FLASH => 'Flash Sale',
            Coupon::TYPE_REFERRAL => 'Referral',
            Coupon::TYPE_LOYALTY => 'Loyalty',
            Coupon::TYPE_ANNIVERSARY => 'Anniversary',
            Coupon::TYPE_SEASONAL => 'Seasonal',
            Coupon::TYPE_VOLUME => 'Volume',
        ];

        // Get discount types for display
        $discountTypes = [
            Coupon::DISCOUNT_FIXED => 'Fixed Amount',
            Coupon::DISCOUNT_PERCENTAGE => 'Percentage',
        ];
        return Inertia::render('Admin/Coupons/Show', [
            'coupon' => $coupon,
            'userUsages' => $usageHistory,
            'couponTypes' => $couponTypes,
            'discountTypes' => $discountTypes,
            'usageStats' => $stats,
            'orders' => $orders,
        ]);
    }

    /**
     * Validate coupon data
     */
    private function validateCoupon(Request $request, $couponId = null)
    {
        $rules = [
            'coupon_code' => [
                'nullable',
                'string',
                'max:20',
                $couponId
                    ? Rule::unique('coupons', 'coupon_code')->ignore($couponId)
                    : Rule::unique('coupons', 'coupon_code')
            ],
            'discount_type' => ['required', Rule::in([Coupon::DISCOUNT_FIXED, Coupon::DISCOUNT_PERCENTAGE])],
            'discount_amount' => 'required|numeric|min:0.01',
            'coupon_type' => [
                'required',
                Rule::in([
                    Coupon::TYPE_FIRST_ORDER,
                    Coupon::TYPE_FLASH,
                    Coupon::TYPE_REFERRAL,
                    Coupon::TYPE_LOYALTY,
                    Coupon::TYPE_ANNIVERSARY,
                    Coupon::TYPE_SEASONAL,
                    Coupon::TYPE_VOLUME,
                    Coupon::TYPE_DEFAULT
                ])
            ],
            'loyalty_tier_id' => [
                'nullable',
                'exists:loyalty_tiers,id',
                Rule::requiredIf(function () use ($request) {
                    return $request->coupon_type === Coupon::TYPE_LOYALTY;
                }),
            ],
            'seasonal_type_id' => [
                'nullable',
                'exists:seasonal_types,id',
                Rule::requiredIf(function () use ($request) {
                    return $request->coupon_type === Coupon::TYPE_SEASONAL;
                }),
            ],
            'start_date' => 'nullable|date',
            'expiry_date' => 'nullable|date|after_or_equal:start_date',
            'max_uses' => 'nullable|integer|min:1',
            'per_user_limit' => 'nullable|integer|min:1',
            'min_order_amount' => 'nullable|numeric|min:0',
            'min_pages' => [
                'nullable',
                'integer',
                'min:1',
                Rule::requiredIf(function () use ($request) {
                    return $request->coupon_type === Coupon::TYPE_VOLUME;
                }),
            ],
            'is_first_order_only' => 'boolean',
            'user_specific' => 'boolean',
            'referrer_user_id' => [
                'nullable',
                'exists:users,id',
                Rule::requiredIf(function () use ($request) {
                    return $request->user_specific === true;
                }),
            ],
            'is_referral' => 'boolean',
            'is_active' => 'boolean',
        ];

        return $request->validate($rules);
    }

    /**
     * Bulk toggle status for multiple coupons
     */
    public function bulkToggleStatus(Request $request)
    {
        try {
            $request->validate([
                'coupon_ids' => 'required|array',
                'coupon_ids.*' => 'exists:coupons,id',
                'status' => 'required|boolean',
            ]);

            $couponIds = $request->coupon_ids;
            $status = $request->status;
            $userId = Auth::id();

            Coupon::whereIn('id', $couponIds)
                ->update([
                    'is_active' => $status,
                    'updated_by' => $userId,
                    'updated_at' => now(),
                ]);

            $statusText = $status ? 'activated' : 'deactivated';
            $count = count($couponIds);

            return redirect()->back()
                ->with('success', "{$count} coupons {$statusText} successfully.");
        } catch (\Exception $e) {
            Log::error('Failed to bulk update coupon status: ' . $e->getMessage(), [
                'request' => $request->all(),
                'trace' => $e->getTraceAsString()
            ]);

            return redirect()->back()
                ->with('error', 'Failed to update coupon status: ' . $e->getMessage());
        }
    }

    /**
     * Bulk delete multiple coupons
     */
    public function bulkDelete(Request $request)
    {
        try {
            $request->validate([
                'coupon_ids' => 'required|array',
                'coupon_ids.*' => 'exists:coupons,id',
            ]);

            $couponIds = $request->coupon_ids;
            $userId = Auth::id();

            // First update the deleted_by field for all coupons
            Coupon::whereIn('id', $couponIds)
                ->update([
                    'deleted_by' => $userId,
                    'updated_at' => now(),
                ]);

            // Then delete them
            Coupon::whereIn('id', $couponIds)->delete();

            $count = count($couponIds);

            return redirect()->back()
                ->with('success', "{$count} coupons deleted successfully.");
        } catch (\Exception $e) {
            Log::error('Failed to bulk delete coupons: ' . $e->getMessage(), [
                'request' => $request->all(),
                'trace' => $e->getTraceAsString()
            ]);

            return redirect()->back()
                ->with('error', 'Failed to delete coupons: ' . $e->getMessage());
        }
    }
}
