<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Coupon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Inertia\Inertia;

class DefaultCouponController extends Controller
{
    /**
     * Display a listing of default coupons.
     */
    public function index()
    {
        $defaultCoupons = Coupon::where('coupon_type', Coupon::TYPE_DEFAULT)
            ->orderBy('created_at', 'desc')
            ->paginate(10);

        return Inertia::render('Admin/DefaultCoupons/Index', [
            'defaultCoupons' => $defaultCoupons
        ]);
    }

    /**
     * Show the form for creating a new default coupon.
     */
    public function create()
    {
        return Inertia::render('Admin/DefaultCoupons/Create');
    }

    /**
     * Store a newly created default coupon.
     */
    public function store(Request $request)
    {
        Log::info('Default coupon creation started', [
            'request_data' => $request->all(),
            'user_id' => auth()->id()
        ]);

        try {
            $validated = $request->validate([
                'coupon_code' => 'required|string|max:50|unique:coupons,coupon_code',
                'discount_type' => 'required|in:percentage,fixed',
                'discount_amount' => 'required|numeric|min:0',
                'per_user_limit' => 'nullable|integer|min:0',
                'max_uses' => 'nullable|integer|min:0',
            ]);

            Log::info('Default coupon validation passed', [
                'validated_data' => $validated
            ]);

        } catch (\Illuminate\Validation\ValidationException $e) {
            Log::error('Default coupon validation failed', [
                'errors' => $e->errors(),
                'request_data' => $request->all(),
                'user_id' => auth()->id()
            ]);
            
            return redirect()->back()
                ->withErrors($e->errors())
                ->withInput()
                ->with('error', 'Validation failed. Please check the form and try again.');
        }

        // Handle per_user_limit: treat 0 as null (unlimited)
        $perUserLimit = $validated['per_user_limit'] ?? null;
        if ($perUserLimit === 0) {
            $perUserLimit = null;
        }

        // Handle max_uses: treat 0 as null (unlimited)
        $maxUses = $validated['max_uses'] ?? null;
        if ($maxUses === 0) {
            $maxUses = null;
        }

        try {
            // Create the default coupon with specific settings
            $coupon = Coupon::create([
                'coupon_code' => strtoupper($validated['coupon_code']),
                'coupon_type' => Coupon::TYPE_DEFAULT,
                'discount_type' => $validated['discount_type'],
                'discount_amount' => $validated['discount_amount'],
                'is_active' => true,
                'uses_count' => 0,
                'max_uses' => $maxUses, // Null = unlimited
                'per_user_limit' => $perUserLimit, // Null = unlimited
                'start_date' => now(),
                'expiry_date' => null, // Never expires
                'created_by' => auth()->id(),
            ]);

            Log::info('Default coupon created successfully', [
                'coupon_id' => $coupon->id,
                'coupon_code' => $coupon->coupon_code,
                'per_user_limit' => $coupon->per_user_limit,
                'max_uses' => $coupon->max_uses,
                'created_by' => auth()->id()
            ]);

            // Build detailed success message
            $usageInfo = [];
            
            if ($coupon->per_user_limit === null) {
                $usageInfo[] = "unlimited uses per user";
            } else {
                $usageInfo[] = "{$coupon->per_user_limit} use(s) per user";
            }
            
            if ($coupon->max_uses === null) {
                $usageInfo[] = "unlimited total uses";
            } else {
                $usageInfo[] = "{$coupon->max_uses} total uses";
            }

            $detailedMessage = "Coupon '{$coupon->coupon_code}' created successfully! Settings: " . implode(", ", $usageInfo) . ".";

            return redirect()->route('admin.default-coupons.index')
                ->with('success', 'Default coupon created successfully!')
                ->with('message', $detailedMessage);

        } catch (\Exception $e) {
            Log::error('Exception during default coupon creation', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'validated_data' => $validated,
                'user_id' => auth()->id()
            ]);

            return redirect()->back()
                ->withInput()
                ->with('error', 'An error occurred while creating the coupon: ' . $e->getMessage());
        }
    }

    /**
     * Display the specified default coupon.
     */
    public function show(Coupon $defaultCoupon)
    {
        // Ensure this is a default coupon
        if ($defaultCoupon->coupon_type !== Coupon::TYPE_DEFAULT) {
            abort(404);
        }

        return Inertia::render('Admin/DefaultCoupons/Show', [
            'defaultCoupon' => $defaultCoupon
        ]);
    }

    /**
     * Show the form for editing the specified default coupon.
     */
    public function edit(Coupon $defaultCoupon)
    {
        // Ensure this is a default coupon
        if ($defaultCoupon->coupon_type !== Coupon::TYPE_DEFAULT) {
            abort(404);
        }

        // Force refresh from database
        $defaultCoupon = $defaultCoupon->fresh();
        
        // Debug logging
        Log::info('Loading edit form for default coupon', [
            'coupon_id' => $defaultCoupon->id,
            'coupon_code' => $defaultCoupon->coupon_code,
            'per_user_limit' => $defaultCoupon->per_user_limit,
            'per_user_limit_type' => gettype($defaultCoupon->per_user_limit),
            'per_user_limit_is_null' => $defaultCoupon->per_user_limit === null,
            'max_uses' => $defaultCoupon->max_uses,
        ]);

        return Inertia::render('Admin/DefaultCoupons/Edit', [
            'defaultCoupon' => $defaultCoupon
        ]);
    }

    /**
     * Update the specified default coupon.
     */
    public function update(Request $request, Coupon $defaultCoupon)
    {
        // Ensure this is a default coupon
        if ($defaultCoupon->coupon_type !== Coupon::TYPE_DEFAULT) {
            Log::error('Attempted to update non-default coupon via DefaultCouponController', [
                'coupon_id' => $defaultCoupon->id,
                'coupon_type' => $defaultCoupon->coupon_type,
                'user_id' => auth()->id()
            ]);
            abort(404);
        }

        Log::info('Default coupon update started', [
            'coupon_id' => $defaultCoupon->id,
            'current_code' => $defaultCoupon->coupon_code,
            'current_per_user_limit' => $defaultCoupon->per_user_limit,
            'current_max_uses' => $defaultCoupon->max_uses,
            'request_data' => $request->all(),
            'per_user_limit_raw' => $request->input('per_user_limit'),
            'per_user_limit_type' => gettype($request->input('per_user_limit')),
            'per_user_limit_is_null' => $request->input('per_user_limit') === null,
            'per_user_limit_is_empty_string' => $request->input('per_user_limit') === '',
            'max_uses_raw' => $request->input('max_uses'),
            'max_uses_type' => gettype($request->input('max_uses')),
            'user_id' => auth()->id()
        ]);

        try {
            $validated = $request->validate([
                'coupon_code' => 'required|string|max:50|unique:coupons,coupon_code,' . $defaultCoupon->id,
                'discount_type' => 'required|in:percentage,fixed',
                'discount_amount' => 'required|numeric|min:0',
                'is_active' => 'required|boolean',
                'per_user_limit' => 'nullable|integer|min:0', // Allow null or empty for unlimited, 0 or positive integer
                'max_uses' => 'nullable|integer|min:0', // Allow null or empty for unlimited, 0 or positive integer
            ]);

            Log::info('Default coupon validation passed', [
                'validated_data' => $validated,
                'per_user_limit_after_validation' => $validated['per_user_limit'] ?? 'NOT_IN_VALIDATED',
                'max_uses_after_validation' => $validated['max_uses'] ?? 'NOT_IN_VALIDATED',
            ]);

        } catch (\Illuminate\Validation\ValidationException $e) {
            Log::error('Default coupon validation failed', [
                'coupon_id' => $defaultCoupon->id,
                'errors' => $e->errors(),
                'request_data' => $request->all(),
                'user_id' => auth()->id()
            ]);
            
            return redirect()->back()
                ->withErrors($e->errors())
                ->withInput()
                ->with('error', 'Validation failed. Please check the form and try again.');
        }

        // Handle per_user_limit: explicitly check if the key exists, treat empty string and 0 as null (unlimited)
        if ($request->has('per_user_limit')) {
            $perUserLimit = $request->input('per_user_limit');
            // Convert empty string, null, or 0 to null for unlimited
            if ($perUserLimit === "" || $perUserLimit === null || $perUserLimit === 0 || $perUserLimit === "0") {
                $perUserLimit = null;
            } else {
                $perUserLimit = (int)$perUserLimit;
            }
        } else {
            // If key doesn't exist, keep current value
            $perUserLimit = $defaultCoupon->per_user_limit;
        }

        // Handle max_uses: explicitly check if the key exists, treat empty string and 0 as null (unlimited)
        if ($request->has('max_uses')) {
            $maxUses = $request->input('max_uses');
            // Convert empty string, null, or 0 to null for unlimited
            if ($maxUses === "" || $maxUses === null || $maxUses === 0 || $maxUses === "0") {
                $maxUses = null;
            } else {
                $maxUses = (int)$maxUses;
            }
        } else {
            // If key doesn't exist, keep current value
            $maxUses = $defaultCoupon->max_uses;
        }

        Log::info('Processing usage limits', [
            'per_user_limit_raw' => $request->input('per_user_limit'),
            'per_user_limit_raw_type' => gettype($request->input('per_user_limit')),
            'per_user_limit_processed' => $perUserLimit,
            'per_user_limit_is_null' => $perUserLimit === null,
            'max_uses_raw' => $request->input('max_uses'),
            'max_uses_raw_type' => gettype($request->input('max_uses')),
            'max_uses_processed' => $maxUses,
            'max_uses_is_null' => $maxUses === null,
        ]);

        $updateData = [
            'coupon_code' => strtoupper($validated['coupon_code']),
            'discount_type' => $validated['discount_type'],
            'discount_amount' => $validated['discount_amount'],
            'is_active' => $validated['is_active'],
            'per_user_limit' => $perUserLimit,
            'max_uses' => $maxUses,
            'updated_by' => auth()->id(),
        ];

        Log::info('Attempting to update default coupon', [
            'coupon_id' => $defaultCoupon->id,
            'update_data' => $updateData,
            'per_user_limit_raw' => $validated['per_user_limit'] ?? 'not_provided',
            'per_user_limit_final' => $perUserLimit,
            'max_uses_raw' => $validated['max_uses'] ?? 'not_provided',
            'max_uses_final' => $maxUses,
        ]);

        try {
            $updated = $defaultCoupon->update($updateData);

            if (!$updated) {
                Log::error('Default coupon update failed - update returned false', [
                    'coupon_id' => $defaultCoupon->id,
                    'update_data' => $updateData
                ]);
                
                return redirect()->back()
                    ->withInput()
                    ->with('error', 'Failed to update coupon. Please try again.');
            }

            $freshCoupon = $defaultCoupon->fresh();

            Log::info('Default coupon update successful', [
                'coupon_id' => $defaultCoupon->id,
                'update_result' => $updated,
                'new_code' => $freshCoupon->coupon_code,
                'new_discount_amount' => $freshCoupon->discount_amount,
                'new_is_active' => $freshCoupon->is_active,
                'new_per_user_limit' => $freshCoupon->per_user_limit,
                'new_max_uses' => $freshCoupon->max_uses,
                'updated_by' => auth()->id()
            ]);

            // Build detailed success message
            $successMessage = "Coupon '{$freshCoupon->coupon_code}' updated successfully!";
            $usageInfo = [];
            
            if ($freshCoupon->per_user_limit === null) {
                $usageInfo[] = "unlimited uses per user";
            } else {
                $usageInfo[] = "{$freshCoupon->per_user_limit} use(s) per user";
            }
            
            if ($freshCoupon->max_uses === null) {
                $usageInfo[] = "unlimited total uses";
            } else {
                $usageInfo[] = "{$freshCoupon->max_uses} total uses";
            }

            $detailedMessage = $successMessage . " Settings: " . implode(", ", $usageInfo) . ".";

            return redirect()->back()
                ->with('success', 'Default coupon updated successfully!')
                ->with('message', $detailedMessage);

        } catch (\Exception $e) {
            Log::error('Exception during default coupon update', [
                'coupon_id' => $defaultCoupon->id,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'update_data' => $updateData,
                'user_id' => auth()->id()
            ]);

            return redirect()->back()
                ->withInput()
                ->with('error', 'An error occurred while updating the coupon: ' . $e->getMessage());
        }
    }

    /**
     * Remove the specified default coupon.
     */
    public function destroy(Coupon $defaultCoupon)
    {
        // Ensure this is a default coupon
        if ($defaultCoupon->coupon_type !== Coupon::TYPE_DEFAULT) {
            abort(404);
        }

        // Check if coupon has been used
        if ($defaultCoupon->uses_count > 0) {
            return back()->withErrors(['message' => 'Cannot delete a coupon that has been used.']);
        }

        $couponCode = $defaultCoupon->coupon_code;
        $defaultCoupon->delete();

        Log::info('Default coupon deleted', [
            'coupon_code' => $couponCode,
            'deleted_by' => auth()->id()
        ]);

        return redirect()->route('admin.default-coupons.index')
            ->with('message', 'Default coupon deleted successfully!');
    }

    /**
     * Toggle the active status of a default coupon.
     */
    public function toggleStatus(Coupon $defaultCoupon)
    {
        // Ensure this is a default coupon
        if ($defaultCoupon->coupon_type !== Coupon::TYPE_DEFAULT) {
            abort(404);
        }

        $defaultCoupon->update([
            'is_active' => !$defaultCoupon->is_active,
            'updated_by' => auth()->id(),
        ]);

        $status = $defaultCoupon->is_active ? 'activated' : 'deactivated';
        
        Log::info('Default coupon status toggled', [
            'coupon_id' => $defaultCoupon->id,
            'coupon_code' => $defaultCoupon->coupon_code,
            'new_status' => $status,
            'updated_by' => auth()->id()
        ]);

        return back()->with('message', "Default coupon {$status} successfully!");
    }
} 