<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\EssayTopic;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Inertia\Inertia;

class EssayTopicController extends Controller
{
    /**
     * Display a listing of the essay topics.
     */
    public function index(Request $request)
    {
        // Get search and filter parameters
        $search = $request->input('search', '');
        $category = $request->input('category', 'all');
        $status = $request->input('status', 'all');

        // Build the query
        $query = EssayTopic::with('creator:id,name');

        // Apply search filter if provided
        if (!empty($search)) {
            $query->where(function ($q) use ($search) {
                $q->where('title', 'LIKE', "%{$search}%")
                    ->orWhere('description', 'LIKE', "%{$search}%")
                    ->orWhere('category', 'LIKE', "%{$search}%");
            });
        }

        // Apply category filter if not 'all'
        if ($category !== 'all') {
            $query->where('category', $category);
        }

        // Apply status filter if not 'all'
        if ($status !== 'all') {
            $query->where('is_active', $status === 'active');
        }

        // Get unique categories for filter dropdown
        $categories = EssayTopic::select('category')
            ->whereNotNull('category')
            ->distinct()
            ->orderBy('category')
            ->pluck('category');

        // Get paginated results
        $topics = $query->orderBy('created_at', 'desc')
            ->paginate(10)
            ->withQueryString();

        return Inertia::render('Admin/EssayTopics/Index', [
            'topics' => $topics,
            'categories' => $categories,
            'filters' => [
                'search' => $search,
                'category' => $category,
                'status' => $status,
            ],
        ]);
    }

    /**
     * Show the form for creating a new essay topic.
     */
    public function create()
    {
        // Get unique categories for dropdown
        $categories = EssayTopic::select('category')
            ->whereNotNull('category')
            ->distinct()
            ->orderBy('category')
            ->pluck('category');

        return Inertia::render('Admin/EssayTopics/Create', [
            'categories' => $categories,
        ]);
    }

    /**
     * Store a newly created essay topic in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'required|string',
            'category' => 'nullable|string|max:100',
            'is_active' => 'boolean',
        ]);

        // Add the current user as creator
        $validated['created_by'] = Auth::id();

        EssayTopic::create($validated);

        return redirect()->route('admin.essay-topics.index')
            ->with('success', 'Essay topic created successfully.');
    }

    /**
     * Show the form for editing the specified essay topic.
     */
    public function edit(EssayTopic $essayTopic)
    {
        // Get unique categories for dropdown
        $categories = EssayTopic::select('category')
            ->whereNotNull('category')
            ->distinct()
            ->orderBy('category')
            ->pluck('category');

        return Inertia::render('Admin/EssayTopics/Edit', [
            'topic' => $essayTopic->load('creator:id,name'),
            'categories' => $categories,
        ]);
    }

    /**
     * Update the specified essay topic in storage.
     */
    public function update(Request $request, EssayTopic $essayTopic)
    {
        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'required|string',
            'category' => 'nullable|string|max:100',
            'is_active' => 'boolean',
        ]);

        $essayTopic->update($validated);

        return redirect()->route('admin.essay-topics.index')
            ->with('success', 'Essay topic updated successfully.');
    }

    /**
     * Remove the specified essay topic from storage.
     */
    public function destroy(EssayTopic $essayTopic)
    {
        // Check if the topic has been used in any essays
        $hasEssays = $essayTopic->essays()->exists();

        if ($hasEssays) {
            return redirect()->route('admin.essay-topics.index')
                ->with('error', 'Cannot delete a topic that has been used in essays. Consider deactivating it instead.');
        }

        $essayTopic->delete();

        return redirect()->route('admin.essay-topics.index')
            ->with('success', 'Essay topic deleted successfully.');
    }

    /**
     * Toggle the active status of the specified essay topic.
     */
    public function toggleStatus(EssayTopic $topic)
    {
        $topic->is_active = !$topic->is_active;
        $topic->save();

        $status = $topic->is_active ? 'activated' : 'deactivated';

        return redirect()->back()
            ->with('success', "Topic {$status} successfully.");
    }
}
