<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\FooterSetting;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Cache;
use Inertia\Inertia;

class FooterController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $settings = FooterSetting::orderBy('section_type')
            ->orderBy('order')
            ->get()
            ->groupBy('section_type');

        return Inertia::render('Admin/CMS/Footer/Index', [
            'settings' => $settings,
        ]);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return Inertia::render('Admin/CMS/Footer/Create', [
            'sectionTypes' => [
                'company_info' => 'Company Information',
                'service_link' => 'Service Link',
                'quick_link' => 'Quick Link',
                'contact_info' => 'Contact Information',
                'legal_link' => 'Legal Link',
            ],
        ]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'section_type' => 'required|string|in:company_info,service_link,quick_link,contact_info,legal_link',
            'data' => 'required|array',
            'order' => 'nullable|integer',
            'is_active' => 'boolean',
        ]);

        // Generate a unique key based on section type and data
        $key = $this->generateKey($validated['section_type'], $validated['data']);

        $setting = FooterSetting::create([
            'key' => $key,
            'section_type' => $validated['section_type'],
            'data' => $validated['data'],
            'order' => $validated['order'] ?? 0,
            'is_active' => $validated['is_active'] ?? true,
        ]);

        // Clear footer cache
        Cache::forget('footer_data');

        return redirect()->route('admin.cms.footer.index')
            ->with('success', 'Footer setting created successfully.');
    }

    /**
     * Display the specified resource.
     */
    public function show(FooterSetting $footer)
    {
        return Inertia::render('Admin/CMS/Footer/Show', [
            'setting' => $footer,
        ]);
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(FooterSetting $footer)
    {
        return Inertia::render('Admin/CMS/Footer/Edit', [
            'setting' => $footer,
            'sectionTypes' => [
                'company_info' => 'Company Information',
                'service_link' => 'Service Link',
                'quick_link' => 'Quick Link',
                'contact_info' => 'Contact Information',
                'legal_link' => 'Legal Link',
            ],
        ]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, FooterSetting $footer)
    {
        $validated = $request->validate([
            'section_type' => 'required|string|in:company_info,service_link,quick_link,contact_info,legal_link',
            'data' => 'required|array',
            'order' => 'nullable|integer',
            'is_active' => 'boolean',
        ]);

        $footer->update([
            'section_type' => $validated['section_type'],
            'data' => $validated['data'],
            'order' => $validated['order'] ?? $footer->order,
            'is_active' => $validated['is_active'] ?? $footer->is_active,
        ]);

        // Clear footer cache
        Cache::forget('footer_data');

        return redirect()->route('admin.cms.footer.index')
            ->with('success', 'Footer setting updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(FooterSetting $footer)
    {
        $footer->delete();

        // Clear footer cache
        Cache::forget('footer_data');

        return redirect()->route('admin.cms.footer.index')
            ->with('success', 'Footer setting deleted successfully.');
    }

    /**
     * Update the order of footer settings
     */
    public function updateOrder(Request $request)
    {
        $validated = $request->validate([
            'items' => 'required|array',
            'items.*.id' => 'required|exists:footer_settings,id',
            'items.*.order' => 'required|integer',
        ]);

        foreach ($validated['items'] as $item) {
            FooterSetting::where('id', $item['id'])
                ->update(['order' => $item['order']]);
        }

        // Clear footer cache
        Cache::forget('footer_data');

        return response()->json(['message' => 'Order updated successfully']);
    }

    /**
     * Toggle active status
     */
    public function toggleActive(FooterSetting $footer)
    {
        $footer->update(['is_active' => !$footer->is_active]);

        // Clear footer cache
        Cache::forget('footer_data');

        return redirect()->route('admin.cms.footer.index')
            ->with('success', 'Footer setting status updated successfully.');
    }

    /**
     * Generate a unique key for the footer setting
     */
    private function generateKey(string $sectionType, array $data): string
    {
        $suffix = '';
        
        if (isset($data['name'])) {
            $suffix = Str::slug($data['name'], '_');
        } elseif (isset($data['title'])) {
            $suffix = Str::slug($data['title'], '_');
        } else {
            $suffix = Str::random(8);
        }

        return $sectionType . '_' . $suffix . '_' . time();
    }
}
