<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\PageMedia;
use App\Services\MediaService;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Storage;

class PageMediaController extends Controller
{
    protected MediaService $mediaService;

    public function __construct(MediaService $mediaService)
    {
        $this->mediaService = $mediaService;
    }

    /**
     * Display media files for a page
     */
    public function index(Request $request): JsonResponse
    {
        $pageId = $request->get('page_id');
        $context = $request->get('context');
        $search = $request->get('search');

        if ($search) {
            $media = $this->mediaService->searchMedia($search, $pageId);
        } elseif ($pageId) {
            $media = $this->mediaService->getPageMedia($pageId, $context);
        } else {
            $media = PageMedia::with('page')->orderBy('created_at', 'desc')->paginate(20);
        }

        return response()->json([
            'success' => true,
            'media' => $media
        ]);
    }

    /**
     * Upload media file
     */
    public function store(Request $request): JsonResponse
    {
        $validated = $request->validate([
            'file' => 'required|file|max:10240', // 10MB max
            'page_id' => 'required|exists:pages,id',
            'alt_text' => 'nullable|string|max:255',
            'title' => 'nullable|string|max:255',
            'caption' => 'nullable|string|max:500',
            'usage_context' => 'nullable|string|max:255'
        ]);

        // Validate file
        $fileErrors = $this->mediaService->validateFile($validated['file']);
        if (!empty($fileErrors)) {
            return response()->json([
                'success' => false,
                'message' => 'File validation failed',
                'errors' => $fileErrors
            ], 422);
        }

        try {
            $media = $this->mediaService->uploadMedia(
                $validated['file'],
                $validated['page_id'],
                [
                    'alt_text' => $validated['alt_text'] ?? '',
                    'title' => $validated['title'] ?? '',
                    'caption' => $validated['caption'] ?? '',
                    'usage_context' => $validated['usage_context'] ?? ''
                ]
            );

            return response()->json([
                'success' => true,
                'message' => 'File uploaded successfully',
                'media' => $media
            ], 201);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to upload file: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Display the specified media file
     */
    public function show(PageMedia $media): JsonResponse
    {
        return response()->json([
            'success' => true,
            'media' => $media->load('page')
        ]);
    }

    /**
     * Update media metadata
     */
    public function update(Request $request, PageMedia $media): JsonResponse
    {
        $validated = $request->validate([
            'alt_text' => 'nullable|string|max:255',
            'title' => 'nullable|string|max:255',
            'caption' => 'nullable|string|max:500',
            'usage_context' => 'nullable|string|max:255'
        ]);

        try {
            $updatedMedia = $this->mediaService->updateMediaMetadata($media, $validated);

            return response()->json([
                'success' => true,
                'message' => 'Media updated successfully',
                'media' => $updatedMedia
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to update media: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Remove the specified media file
     */
    public function destroy(PageMedia $media): JsonResponse
    {
        try {
            $this->mediaService->deleteMedia($media);

            return response()->json([
                'success' => true,
                'message' => 'Media deleted successfully'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to delete media: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Bulk delete media files
     */
    public function bulkDelete(Request $request): JsonResponse
    {
        $validated = $request->validate([
            'media_ids' => 'required|array',
            'media_ids.*' => 'exists:page_media,id'
        ]);

        try {
            $deletedCount = 0;
            $errors = [];

            foreach ($validated['media_ids'] as $mediaId) {
                $media = PageMedia::find($mediaId);
                if ($media) {
                    if ($this->mediaService->deleteMedia($media)) {
                        $deletedCount++;
                    } else {
                        $errors[] = 'Failed to delete media ID: ' . $mediaId;
                    }
                }
            }

            return response()->json([
                'success' => true,
                'message' => "Successfully deleted {$deletedCount} media files",
                'deleted_count' => $deletedCount,
                'errors' => $errors
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to delete media files: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get media statistics
     */
    public function stats(Request $request): JsonResponse
    {
        $pageId = $request->get('page_id');
        $stats = $this->mediaService->getMediaStats($pageId);

        return response()->json([
            'success' => true,
            'stats' => $stats
        ]);
    }

    /**
     * Optimize images
     */
    public function optimize(Request $request): JsonResponse
    {
        $validated = $request->validate([
            'page_id' => 'nullable|exists:pages,id',
            'quality' => 'nullable|integer|min:1|max:100'
        ]);

        try {
            $results = $this->mediaService->bulkOptimizeImages(
                $validated['page_id'] ?? null,
                $validated['quality'] ?? 85
            );

            return response()->json([
                'success' => true,
                'message' => 'Image optimization completed',
                'results' => $results
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to optimize images: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get optimized image URL
     */
    public function getOptimizedUrl(Request $request): JsonResponse
    {
        $validated = $request->validate([
            'image_path' => 'required|string',
            'size' => 'nullable|string|in:thumbnail,medium,large,xlarge'
        ]);

        try {
            $url = $this->mediaService->getOptimizedImageUrl(
                $validated['image_path'],
                $validated['size'] ?? 'medium'
            );

            return response()->json([
                'success' => true,
                'url' => $url
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to get optimized URL: ' . $e->getMessage()
            ], 500);
        }
    }
}
