<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\OrderRevision;
use App\Models\WriterPaymentRecord;
use App\Models\AdminSetting;
use App\Services\WriterPaymentCalculationService;
use App\Services\RevisionQualityAssessmentService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Cache;
use Inertia\Inertia;

class PaymentImpactController extends Controller
{
    protected $paymentCalculationService;
    protected $qualityAssessmentService;

    public function __construct(
        WriterPaymentCalculationService $paymentCalculationService,
        RevisionQualityAssessmentService $qualityAssessmentService
    ) {
        $this->paymentCalculationService = $paymentCalculationService;
        $this->qualityAssessmentService = $qualityAssessmentService;
    }

    /**
     * Display payment impact dashboard.
     */
    public function index()
    {
        $stats = $this->getPaymentImpactStats();
        $recentAssessments = OrderRevision::whereNotNull('quality_rating')
            ->with(['order.writer', 'assignedTo'])
            ->latest('updated_at')
            ->take(10)
            ->get();

        return Inertia::render('Admin/PaymentImpacts/Index', compact('stats', 'recentAssessments'));
    }

    /**
     * Show payment impact settings form.
     */
    public function settings()
    {
        $settings = $this->getPaymentImpactSettings();

        return Inertia::render('Admin/PaymentImpacts/Settings', compact('settings'));
    }

    /**
     * Update payment impact settings.
     */
    public function updateSettings(Request $request)
    {
        $validated = $request->validate([
            'quality_bonus' => 'required|numeric|min:0|max:100',
            'quality_penalty' => 'required|numeric|min:0|max:100',
            'delay_penalty_per_hour' => 'required|numeric|min:0|max:10',
        ]);

        try {
            // Update settings in database
            $this->updatePaymentImpactSettings($validated);
            
            // Clear cache to ensure new settings are used
            Cache::forget('payment_impact_settings');

            return back()->with('success', 'Payment impact settings updated successfully.');
        } catch (\Exception $e) {
            Log::error('Failed to update payment impact settings: ' . $e->getMessage());
            return back()->withErrors(['error' => 'Failed to update settings. Please try again.']);
        }
    }

    /**
     * Assess the quality of a completed revision.
     */
    public function assessQuality(Request $request, $revisionId)
    {
        $validated = $request->validate([
            'quality_rating' => 'required|integer|min:1|max:10',
            'quality_notes' => 'nullable|string|max:1000',
        ]);

        try {
            $revision = OrderRevision::findOrFail($revisionId);
            
            if ($revision->status !== 'completed') {
                return back()->withErrors(['error' => 'Only completed revisions can be assessed.']);
            }

            $result = $this->qualityAssessmentService->assessQuality(
                $revision,
                $validated['quality_rating'],
                $validated['quality_notes'] ?? null
            );

            if ($result['success']) {
                return back()->with('success', 'Quality assessment completed successfully.');
            } else {
                return back()->withErrors(['error' => $result['message']]);
            }

        } catch (\Exception $e) {
            Log::error('Failed to assess revision quality: ' . $e->getMessage());
            return back()->withErrors(['error' => 'Failed to assess quality. Please try again.']);
        }
    }

    /**
     * Reject a revision and apply penalties.
     */
    public function rejectRevision(Request $request, $revisionId)
    {
        $validated = $request->validate([
            'rejection_reason' => 'required|string|max:1000',
            'admin_notes' => 'nullable|string|max:1000',
        ]);

        try {
            $revision = OrderRevision::findOrFail($revisionId);
            
            if (!in_array($revision->status, ['requested', 'in_progress', 'completed'])) {
                return back()->withErrors(['error' => 'This revision cannot be rejected.']);
            }

            $result = $this->qualityAssessmentService->rejectRevision(
                $revision,
                $validated['rejection_reason'],
                $validated['admin_notes'] ?? null
            );

            if ($result['success']) {
                return back()->with('success', 'Revision rejected successfully.');
            } else {
                return back()->withErrors(['error' => $result['message']]);
            }

        } catch (\Exception $e) {
            Log::error('Failed to reject revision: ' . $e->getMessage());
            return back()->withErrors(['error' => 'Failed to reject revision. Please try again.']);
        }
    }

    /**
     * Show payment impact analysis for a specific writer.
     */
    public function writerAnalysis($writerId)
    {
        $writer = \App\Models\User::findOrFail($writerId);
        
        $revisions = OrderRevision::where('assigned_to', $writerId)
            ->whereNotNull('quality_rating')
            ->with(['order'])
            ->get();

        $paymentSummary = $this->paymentCalculationService->getWriterPaymentSummary($writerId);
        
        $qualityStats = $this->calculateWriterQualityStats($revisions);
        $paymentImpactHistory = $this->getPaymentImpactHistory($writerId);

        return view('admin.payment-impacts.writer-analysis', compact(
            'writer', 
            'revisions', 
            'paymentSummary', 
            'qualityStats', 
            'paymentImpactHistory'
        ));
    }

    /**
     * Show payment impact analysis for a specific order.
     */
    public function orderAnalysis($orderId)
    {
        $order = \App\Models\Order::with(['revisions.assignedTo', 'writer'])->findOrFail($orderId);
        
        $revisionImpacts = $order->revisions->map(function ($revision) {
            return [
                'revision_id' => $revision->id,
                'type' => $revision->revision_type,
                'quality_rating' => $revision->quality_rating,
                'payment_impact' => $revision->payment_impact,
                'quality_bonus' => $revision->quality_bonus,
                'quality_penalty' => $revision->quality_penalty,
                'assigned_to' => $revision->assignedTo,
                'completed_at' => $revision->completed_at,
            ];
        });

        $totalPaymentImpact = $revisionImpacts->sum('payment_impact');
        $finalPaymentCalculation = $this->paymentCalculationService->calculateFinalPayment($order);

        return view('admin.payment-impacts.order-analysis', compact(
            'order', 
            'revisionImpacts', 
            'totalPaymentImpact', 
            'finalPaymentCalculation'
        ));
    }

    /**
     * Recalculate payment impacts for a specific order.
     */
    public function recalculateOrder($orderId)
    {
        try {
            DB::beginTransaction();

            $order = \App\Models\Order::findOrFail($orderId);
            
            // Recalculate payment for the order
            $paymentRecord = $this->paymentCalculationService->processPayment($order);

            DB::commit();

            return back()->with('success', 'Payment impacts recalculated successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Failed to recalculate payment impacts: ' . $e->getMessage());
            
            return back()->withErrors(['error' => 'Failed to recalculate payment impacts. Please try again.']);
        }
    }

    /**
     * Get payment impact statistics.
     */
    private function getPaymentImpactStats(): array
    {
        $totalRevisions = OrderRevision::whereNotNull('quality_rating')->count();
        $totalBonuses = OrderRevision::whereNotNull('quality_bonus')->sum('quality_bonus');
        $totalPenalties = OrderRevision::whereNotNull('quality_penalty')->sum('quality_penalty');
        $netImpact = $totalBonuses - $totalPenalties;

        $qualityDistribution = OrderRevision::whereNotNull('quality_rating')
            ->select('quality_rating', DB::raw('count(*) as count'))
            ->groupBy('quality_rating')
            ->orderBy('quality_rating')
            ->pluck('count', 'quality_rating')
            ->toArray();

        $monthlyTrends = OrderRevision::whereNotNull('quality_rating')
            ->select(
                DB::raw('DATE_FORMAT(updated_at, "%Y-%m") as month'),
                DB::raw('count(*) as count'),
                DB::raw('avg(quality_rating) as avg_rating'),
                DB::raw('sum(quality_bonus) as total_bonus'),
                DB::raw('sum(quality_penalty) as total_penalty')
            )
            ->groupBy('month')
            ->orderBy('month', 'desc')
            ->take(12)
            ->get();

        return [
            'total_revisions' => $totalRevisions,
            'total_bonuses' => $totalBonuses,
            'total_penalties' => $totalPenalties,
            'net_impact' => $netImpact,
            'quality_distribution' => $qualityDistribution,
            'monthly_trends' => $monthlyTrends,
        ];
    }

    /**
     * Calculate quality statistics for a writer.
     */
    private function calculateWriterQualityStats($revisions): array
    {
        if ($revisions->isEmpty()) {
            return [
                'total_revisions' => 0,
                'average_rating' => 0,
                'total_bonus' => 0,
                'total_penalty' => 0,
                'net_impact' => 0,
            ];
        }

        $totalRevisions = $revisions->count();
        $averageRating = $revisions->avg('quality_rating');
        $totalBonus = $revisions->sum('quality_bonus');
        $totalPenalty = $revisions->sum('quality_penalty');
        $netImpact = $totalBonus - $totalPenalty;

        return [
            'total_revisions' => $totalRevisions,
            'average_rating' => round($averageRating, 2),
            'total_bonus' => $totalBonus,
            'total_penalty' => $totalPenalty,
            'net_impact' => $netImpact,
        ];
    }

    /**
     * Get payment impact history for a writer.
     */
    private function getPaymentImpactHistory($writerId): array
    {
        return OrderRevision::where('assigned_to', $writerId)
            ->whereNotNull('quality_rating')
            ->with(['order'])
            ->orderBy('updated_at', 'desc')
            ->take(20)
            ->get()
            ->map(function ($revision) {
                return [
                    'revision_id' => $revision->id,
                    'order_id' => $revision->order_id,
                    'order_number' => $revision->order->order_number ?? 'N/A',
                    'quality_rating' => $revision->quality_rating,
                    'payment_impact' => $revision->payment_impact,
                    'quality_bonus' => $revision->quality_bonus,
                    'quality_penalty' => $revision->quality_penalty,
                    'assessed_at' => $revision->updated_at,
                ];
            })
            ->toArray();
    }

    /**
     * Update configuration values.
     */
    private function updateConfigValues(array $values): void
    {
        // This is a simplified approach - in production, you might want to use a database table
        // or environment file management system
        foreach ($values as $key => $value) {
            // For now, we'll just log the changes
            Log::info("Payment impact setting updated: {$key} = {$value}");
        }
    }

    /**
     * Get payment impact settings from database or config.
     */
    private function getPaymentImpactSettings(): array
    {
        // Try to get from cache first
        $settings = Cache::remember('payment_impact_settings', 3600, function () {
            // Check if we have settings in admin_settings table
            $adminSettings = AdminSetting::first();
            
            if ($adminSettings && isset($adminSettings->quality_bonus)) {
                return [
                    'quality_bonus' => $adminSettings->quality_bonus,
                    'quality_penalty' => $adminSettings->quality_penalty,
                    'delay_penalty_per_hour' => $adminSettings->delay_penalty_per_hour,
                ];
            }
            
            // Fallback to config values
            return [
                'quality_bonus' => config('revisions.payment_impact.quality_bonus', 5.00),
                'quality_penalty' => config('revisions.payment_impact.quality_penalty', 10.00),
                'delay_penalty_per_hour' => config('revisions.payment_impact.delay_penalty_per_hour', 0.50),
            ];
        });

        return $settings;
    }

    /**
     * Update payment impact settings in database.
     */
    private function updatePaymentImpactSettings(array $values): void
    {
        $adminSettings = AdminSetting::first();
        
        if (!$adminSettings) {
            // Create new admin settings record
            $adminSettings = new AdminSetting();
        }
        
        // Update the payment impact fields
        $adminSettings->quality_bonus = $values['quality_bonus'];
        $adminSettings->quality_penalty = $values['quality_penalty'];
        $adminSettings->delay_penalty_per_hour = $values['delay_penalty_per_hour'];
        
        $adminSettings->save();
        
        // Log the changes
        Log::info('Payment impact settings updated', $values);
    }
} 