<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\SeasonalType;
use App\Models\Coupon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Validation\Rule;
use Inertia\Inertia;
use Carbon\Carbon;

class SeasonalTypeController extends Controller
{
    /**
     * Display a listing of the seasonal types.
     *
     * @param Request $request
     * @return \Inertia\Response
     */
    public function index(Request $request)
    {
        $query = SeasonalType::query();

        // Search filter
        if ($request->has('search') && !empty($request->search)) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                    ->orWhere('description', 'like', "%{$search}%");
            });
        }

        // Status filter
        if ($request->has('status') && $request->status !== 'all') {
            $status = $request->status === 'active';
            $query->where('is_active', $status);
        }

        // Date filter
        if ($request->has('date_range') && !empty($request->date_range)) {
            try {
                $dates = explode(' to ', $request->date_range);
                if (count($dates) === 2) {
                    $startDate = Carbon::parse($dates[0])->startOfDay();
                    $endDate = Carbon::parse($dates[1])->endOfDay();

                    $query->where(function ($q) use ($startDate, $endDate) {
                        $q->whereBetween('default_start_date', [$startDate, $endDate])
                            ->orWhereBetween('default_end_date', [$startDate, $endDate]);
                    });
                }
            } catch (\Exception $e) {
                Log::error('Error parsing date range in SeasonalTypeController: ' . $e->getMessage());
            }
        }

        // Current period filter
        if ($request->has('period') && $request->period === 'current') {
            $now = Carbon::now();
            $currentYear = $now->year;
            $query->where(function ($q) use ($now, $currentYear) {
                // For standard seasonal periods (start date before end date)
                $q->whereRaw("
                    (MONTH(default_start_date) < MONTH(default_end_date) AND
                    (MONTH(?) BETWEEN MONTH(default_start_date) AND MONTH(default_end_date) AND
                    DAY(?) BETWEEN DAY(default_start_date) AND DAY(default_end_date)))
                ", [$now, $now]);

                // For seasonal periods that cross year boundary (e.g., winter: Dec-Jan)
                $q->orWhereRaw("
                    (MONTH(default_start_date) > MONTH(default_end_date) AND
                    (MONTH(?) >= MONTH(default_start_date) OR
                    MONTH(?) <= MONTH(default_end_date)))
                ", [$now, $now]);
            });
        }

        // Sort by
        $sortField = $request->input('sort_field', 'name');
        $sortDirection = $request->input('sort_direction', 'asc');

        // Validate sort field to prevent SQL injection
        $allowedSortFields = ['name', 'default_start_date', 'default_end_date', 'is_active', 'created_at', 'updated_at'];
        if (!in_array($sortField, $allowedSortFields)) {
            $sortField = 'name';
        }

        $query->orderBy($sortField, $sortDirection);

        // Get paginated results with counts
        $seasonalTypes = $query->withCount('coupons')->paginate(10)->appends(request()->query());

        // Get counts for filter badges
        $counts = [
            'all' => SeasonalType::count(),
            'active' => SeasonalType::where('is_active', true)->count(),
            'inactive' => SeasonalType::where('is_active', false)->count(),
            'current' => SeasonalType::active()->get()->filter(function ($type) {
                return $type->isCurrentlyActive();
            })->count()
        ];

        return Inertia::render('Admin/SeasonalTypes/Index', [
            'seasonalTypes' => $seasonalTypes,
            'counts' => $counts,
            'filters' => [
                'search' => $request->search ?? '',
                'status' => $request->status ?? 'all',
                'date_range' => $request->date_range ?? '',
                'period' => $request->period ?? 'all',
                'sort_field' => $sortField,
                'sort_direction' => $sortDirection,
            ],
        ]);
    }

    /**
     * Show the form for creating a new seasonal type.
     *
     * @return \Inertia\Response
     */
    public function create()
    {
        $currentMonth = Carbon::now()->month;

        // Default to next month if we're past the 15th of the current month
        $defaultStartMonth = Carbon::now()->day > 15
            ? ($currentMonth % 12) + 1
            : $currentMonth;

        $defaultEndMonth = ($defaultStartMonth % 12) + 1;

        $defaultStartDate = Carbon::create(null, $defaultStartMonth, 1)->format('Y-m-d');
        $defaultEndDate = Carbon::create(null, $defaultEndMonth, 1)->subDay(1)->format('Y-m-d');

        return Inertia::render('Admin/SeasonalTypes/Create', [
            'defaultDates' => [
                'start' => $defaultStartDate,
                'end' => $defaultEndDate,
            ]
        ]);
    }

    /**
     * Store a newly created seasonal type in storage.
     *
     * @param Request $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255|unique:seasonal_types,name',
            'description' => 'nullable|string|max:500',
            'default_start_date' => 'required|date',
            'default_end_date' => 'required|date',
            'is_active' => 'boolean',
            'color_code' => 'nullable|string|max:7',
        ]);

        try {
            DB::beginTransaction();

            // Format dates to store only month and day
            if (isset($validated['default_start_date'])) {
                $validated['default_start_date'] = Carbon::parse($validated['default_start_date']);
            }

            if (isset($validated['default_end_date'])) {
                $validated['default_end_date'] = Carbon::parse($validated['default_end_date']);
            }

            $seasonalType = SeasonalType::create($validated);

            DB::commit();

            return redirect()->route('admin.seasonal-types.index')
                ->with('success', "Seasonal type '{$seasonalType->name}' created successfully.");
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Failed to create seasonal type: ' . $e->getMessage(), [
                'data' => $validated,
                'exception' => $e,
            ]);

            return back()->withErrors(['general' => 'Failed to create seasonal type. Please try again.'])
                ->withInput();
        }
    }

    /**
     * Display the specified seasonal type.
     *
     * @param SeasonalType $seasonalType
     * @return \Inertia\Response
     */
    public function show(SeasonalType $seasonalType)
    {
        // Load active coupons for this seasonal type with counts
        $activeCoupons = Coupon::where('seasonal_type_id', $seasonalType->id)
            ->where('is_active', true)
            ->where(function ($query) {
                $now = now();
                $query->whereNull('start_date')
                    ->orWhere('start_date', '<=', $now);
            })
            ->where(function ($query) {
                $now = now();
                $query->whereNull('expiry_date')
                    ->orWhere('expiry_date', '>=', $now);
            })
            ->get();

        // Get seasonal type usage statistics
        $statistics = [
            'total_coupons' => Coupon::where('seasonal_type_id', $seasonalType->id)->count(),
            'active_coupons' => $activeCoupons->count(),
            'total_uses' => Coupon::where('seasonal_type_id', $seasonalType->id)->sum('uses_count'),
            'total_discount' => DB::table('user_coupons')
                ->join('coupons', 'user_coupons.coupon_id', '=', 'coupons.id')
                ->where('coupons.seasonal_type_id', $seasonalType->id)
                ->sum('user_coupons.discount_amount'),
        ];

        // Get all coupons associated with this seasonal type
        $coupons = Coupon::where('seasonal_type_id', $seasonalType->id)
            ->withCount('userUsages')
            ->orderBy('created_at', 'desc')
            ->paginate(10);

        // Calculate if this season is currently active
        $isCurrentlyActive = $seasonalType->isCurrentlyActive();

        // Calculate next occurrence
        $nextOccurrence = null;
        try {
            if ($seasonalType->default_start_date && $seasonalType->default_end_date) {
                $now = Carbon::now();
                $year = $now->year;

                // Create dates for the current year
                $start = Carbon::createFromFormat(
                    'Y-m-d',
                    $year . '-' . $seasonalType->default_start_date->format('m-d')
                );

                $end = Carbon::createFromFormat(
                    'Y-m-d',
                    $year . '-' . $seasonalType->default_end_date->format('m-d')
                );

                // If end date is before start date, it spans across years (e.g., winter break Dec-Jan)
                if ($end->lt($start)) {
                    $end->addYear();
                }

                // If the season has passed for this year, calculate for next year
                if ($now->gt($end)) {
                    $start->addYear();
                    $end->addYear();
                }

                // If we're in the middle of the season, show current dates
                if ($now->between($start, $end)) {
                    $nextOccurrence = [
                        'start' => $start->format('Y-m-d'),
                        'end' => $end->format('Y-m-d'),
                        'status' => 'current'
                    ];
                } else {
                    $nextOccurrence = [
                        'start' => $start->format('Y-m-d'),
                        'end' => $end->format('Y-m-d'),
                        'status' => 'upcoming'
                    ];
                }
            }
        } catch (\Exception $e) {
            Log::error('Error calculating next occurrence: ' . $e->getMessage(), [
                'seasonal_type_id' => $seasonalType->id,
                'exception' => $e,
            ]);
        }

        return Inertia::render('Admin/SeasonalTypes/Show', [
            'seasonalType' => $seasonalType,
            'statistics' => $statistics,
            'coupons' => $coupons,
            'activeCoupons' => $activeCoupons,
            'isCurrentlyActive' => $isCurrentlyActive,
            'nextOccurrence' => $nextOccurrence,
        ]);
    }

    /**
     * Show the form for editing the specified seasonal type.
     *
     * @param SeasonalType $seasonalType
     * @return \Inertia\Response
     */
    public function edit(SeasonalType $seasonalType)
    {
        $couponCount = Coupon::where('seasonal_type_id', $seasonalType->id)->count();

        // Format dates for the frontend
        $formattedStartDate = null;
        $formattedEndDate = null;

        if ($seasonalType->default_start_date) {
            $formattedStartDate = Carbon::parse($seasonalType->default_start_date)
                ->format('Y-m-d');
        }

        if ($seasonalType->default_end_date) {
            $formattedEndDate = Carbon::parse($seasonalType->default_end_date)
                ->format('Y-m-d');
        }

        return Inertia::render('Admin/SeasonalTypes/Edit', [
            'seasonalType' => array_merge($seasonalType->toArray(), [
                'formatted_start_date' => $formattedStartDate,
                'formatted_end_date' => $formattedEndDate,
            ]),
            'couponCount' => $couponCount,
        ]);
    }

    /**
     * Update the specified seasonal type in storage.
     *
     * @param Request $request
     * @param SeasonalType $seasonalType
     * @return \Illuminate\Http\RedirectResponse
     */
    public function update(Request $request, SeasonalType $seasonalType)
    {
        $validated = $request->validate([
            'name' => [
                'required',
                'string',
                'max:255',
                Rule::unique('seasonal_types')->ignore($seasonalType->id),
            ],
            'description' => 'nullable|string|max:500',
            'default_start_date' => 'required|date',
            'default_end_date' => 'required|date',
            'is_active' => 'boolean',
            'color_code' => 'nullable|string|max:7',
        ]);

        try {
            DB::beginTransaction();

            // Format dates to store only month and day
            if (isset($validated['default_start_date'])) {
                $validated['default_start_date'] = Carbon::parse($validated['default_start_date']);
            }

            if (isset($validated['default_end_date'])) {
                $validated['default_end_date'] = Carbon::parse($validated['default_end_date']);
            }

            $seasonalType->update($validated);

            DB::commit();

            return redirect()->route('admin.seasonal-types.index')
                ->with('success', "Seasonal type '{$seasonalType->name}' updated successfully.");
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Failed to update seasonal type: ' . $e->getMessage(), [
                'seasonal_type_id' => $seasonalType->id,
                'data' => $validated,
                'exception' => $e,
            ]);

            return back()->withErrors(['general' => 'Failed to update seasonal type. Please try again.'])
                ->withInput();
        }
    }

    /**
     * Remove the specified seasonal type from storage.
     *
     * @param SeasonalType $seasonalType
     * @return \Illuminate\Http\RedirectResponse
     */
    public function destroy(SeasonalType $seasonalType)
    {
        try {
            DB::beginTransaction();

            // Check if there are coupons using this seasonal type
            $couponCount = Coupon::where('seasonal_type_id', $seasonalType->id)->count();
            if ($couponCount > 0) {
                return back()->withErrors([
                    'general' => "Cannot delete this seasonal type. It's being used by {$couponCount} coupons."
                ]);
            }

            $typeName = $seasonalType->name;
            $seasonalType->delete();

            DB::commit();

            return redirect()->route('admin.seasonal-types.index')
                ->with('success', "Seasonal type '{$typeName}' deleted successfully.");
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Failed to delete seasonal type: ' . $e->getMessage(), [
                'seasonal_type_id' => $seasonalType->id,
                'exception' => $e,
            ]);

            return back()->withErrors(['general' => 'Failed to delete seasonal type. Please try again.']);
        }
    }

    /**
     * Toggle the status of the specified seasonal type.
     *
     * @param SeasonalType $seasonalType
     * @return \Illuminate\Http\RedirectResponse
     */
    public function toggleStatus(SeasonalType $seasonalType)
    {
        try {
            $seasonalType->is_active = !$seasonalType->is_active;
            $seasonalType->save();

            $status = $seasonalType->is_active ? 'activated' : 'deactivated';
            return back()->with('success', "Seasonal type '{$seasonalType->name}' {$status} successfully.");
        } catch (\Exception $e) {
            Log::error('Failed to toggle seasonal type status: ' . $e->getMessage(), [
                'seasonal_type_id' => $seasonalType->id,
                'exception' => $e,
            ]);

            return back()->withErrors(['general' => 'Failed to update seasonal type status. Please try again.']);
        }
    }

    /**
     * Bulk toggle status for multiple seasonal types.
     *
     * @param Request $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function bulkToggleStatus(Request $request)
    {
        $validated = $request->validate([
            'ids' => 'required|array',
            'ids.*' => 'exists:seasonal_types,id',
            'status' => 'required|boolean',
        ]);

        try {
            DB::beginTransaction();

            $count = SeasonalType::whereIn('id', $validated['ids'])
                ->update(['is_active' => $validated['status']]);

            DB::commit();

            $status = $validated['status'] ? 'activated' : 'deactivated';
            return back()->with('success', "{$count} seasonal types {$status} successfully.");
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Failed to bulk toggle seasonal type status: ' . $e->getMessage(), [
                'ids' => $validated['ids'],
                'status' => $validated['status'],
                'exception' => $e,
            ]);

            return back()->withErrors(['general' => 'Failed to update seasonal type statuses. Please try again.']);
        }
    }

    /**
     * Bulk delete multiple seasonal types.
     *
     * @param Request $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function bulkDelete(Request $request)
    {
        $validated = $request->validate([
            'ids' => 'required|array',
            'ids.*' => 'exists:seasonal_types,id',
        ]);

        try {
            DB::beginTransaction();

            // Check for coupons using these seasonal types
            $couponCounts = Coupon::whereIn('seasonal_type_id', $validated['ids'])
                ->selectRaw('seasonal_type_id, count(*) as count')
                ->groupBy('seasonal_type_id')
                ->get()
                ->keyBy('seasonal_type_id');

            if ($couponCounts->isNotEmpty()) {
                $typeWithCoupons = SeasonalType::whereIn('id', $couponCounts->keys())->pluck('name');
                return back()->withErrors([
                    'general' => "Cannot delete seasonal types that are being used by coupons: " . $typeWithCoupons->implode(', ')
                ]);
            }

            $count = SeasonalType::whereIn('id', $validated['ids'])->delete();

            DB::commit();

            return back()->with('success', "{$count} seasonal types deleted successfully.");
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Failed to bulk delete seasonal types: ' . $e->getMessage(), [
                'ids' => $validated['ids'],
                'exception' => $e,
            ]);

            return back()->withErrors(['general' => 'Failed to delete seasonal types. Please try again.']);
        }
    }

    /**
     * Get seasonal type usage statistics.
     *
     * @return \Inertia\Response
     */
    public function showStatistics()
    {
        // Get all seasonal types with associated data
        $seasonalTypes = SeasonalType::withCount('coupons')
            ->withSum('coupons', 'uses_count')
            ->orderBy('name')
            ->get();

        // Add calculated fields to each seasonal type
        foreach ($seasonalTypes as $type) {
            $type->is_currently_active = $type->isCurrentlyActive();

            $type->total_discount = DB::table('user_coupons')
                ->join('coupons', 'user_coupons.coupon_id', '=', 'coupons.id')
                ->where('coupons.seasonal_type_id', $type->id)
                ->sum('user_coupons.discount_amount');
        }

        // Get overall statistics
        $statistics = [
            'total_types' => $seasonalTypes->count(),
            'active_types' => $seasonalTypes->where('is_active', true)->count(),
            'currently_active' => $seasonalTypes->where('is_currently_active', true)->count(),
            'total_coupons' => $seasonalTypes->sum('coupons_count'),
            'total_redemptions' => $seasonalTypes->sum('coupons_sum_uses_count') ?? 0,
            'total_discount' => $seasonalTypes->sum('total_discount') ?? 0,
        ];

        // Get monthly distribution of usage
        $monthlyUsage = DB::table('user_coupons')
            ->join('coupons', 'user_coupons.coupon_id', '=', 'coupons.id')
            ->join('seasonal_types', 'coupons.seasonal_type_id', '=', 'seasonal_types.id')
            ->selectRaw('MONTH(user_coupons.used_at) as month, seasonal_types.name, COUNT(*) as usage_count, SUM(user_coupons.discount_amount) as discount_amount')
            ->whereNotNull('coupons.seasonal_type_id')
            ->whereYear('user_coupons.used_at', Carbon::now()->year)
            ->groupBy('month', 'seasonal_types.name')
            ->orderBy('month')
            ->get();

        return Inertia::render('Admin/SeasonalTypes/Statistics', [
            'seasonalTypes' => $seasonalTypes,
            'statistics' => $statistics,
            'monthlyUsage' => $monthlyUsage,
        ]);
    }
}
