<?php



namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\WriterTestQuestion;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Inertia\Inertia;

class TestQuestionController extends Controller
{
    /**
     * Display a listing of the test questions.
     */
    public function index(Request $request)
    {
        // Get search and filter parameters
        $search = $request->input('search', '');
        $category = $request->input('category', 'all');
        $status = $request->input('status', 'all');

        // Build the query
        $query = WriterTestQuestion::query();

        // Apply search filter if provided
        if (!empty($search)) {
            $query->where(function ($q) use ($search) {
                $q->where('question', 'LIKE', "%{$search}%")
                    ->orWhere('category', 'LIKE', "%{$search}%");
            });
        }

        // Apply category filter if not 'all'
        if ($category !== 'all') {
            $query->where('category', $category);
        }

        // Apply status filter if not 'all'
        if ($status !== 'all') {
            $query->where('is_active', $status === 'active');
        }

        // Get unique categories for filter dropdown
        $categories = WriterTestQuestion::select('category')
            ->distinct()
            ->orderBy('category')
            ->pluck('category');

        // Get paginated results
        $questions = $query->orderBy('created_at', 'desc')
            ->paginate(10)
            ->appends($request->query());
        // ->withQueryString();

        return Inertia::render('Admin/TestQuestions/Index', [
            'questions' => $questions,
            'categories' => $categories,
            'filters' => [
                'search' => $search,
                'category' => $category,
                'status' => $status,
            ],
        ]);
    }

    /**
     * Show the form for creating a new test question.
     */
    public function create()
    {
        // Get unique categories for dropdown
        $categories = WriterTestQuestion::select('category')
            ->distinct()
            ->orderBy('category')
            ->pluck('category');

        return Inertia::render('Admin/TestQuestions/Create', [
            'categories' => $categories,
        ]);
    }

    /**
     * Store a newly created test question in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'question' => 'required|string|max:1000',
            'category' => 'required|string|max:100',
            'option_a' => 'required|string|max:500',
            'option_b' => 'required|string|max:500',
            'option_c' => 'required|string|max:500',
            'option_d' => 'required|string|max:500',
            'correct_answer' => 'required|in:a,b,c,d',
            'explanation' => 'nullable|string|max:1000',
            'is_active' => 'boolean',
        ]);

        WriterTestQuestion::create($validated);

        return redirect()->route('admin.test-questions.index')
            ->with('success', 'Test question created successfully.');
    }

    /**
     * Show the form for editing the specified test question.
     */
    public function edit(WriterTestQuestion $testQuestion)
    {
        // Get unique categories for dropdown
        $categories = WriterTestQuestion::select('category')
            ->distinct()
            ->orderBy('category')
            ->pluck('category');

        return Inertia::render('Admin/TestQuestions/Edit', [
            'question' => $testQuestion,
            'categories' => $categories,
        ]);
    }

    /**
     * Update the specified test question in storage.
     */
    public function update(Request $request, WriterTestQuestion $testQuestion)
    {
        $validated = $request->validate([
            'question' => 'required|string|max:1000',
            'category' => 'required|string|max:100',
            'option_a' => 'required|string|max:500',
            'option_b' => 'required|string|max:500',
            'option_c' => 'required|string|max:500',
            'option_d' => 'required|string|max:500',
            'correct_answer' => 'required|in:a,b,c,d',
            'explanation' => 'nullable|string|max:1000',
            'is_active' => 'boolean',
        ]);

        $testQuestion->update($validated);

        return redirect()->route('admin.test-questions.index')
            ->with('success', 'Test question updated successfully.');
    }

    /**
     * Remove the specified test question from storage.
     */
    public function destroy(WriterTestQuestion $testQuestion)
    {
        // Check if the question has been used in any tests
        $hasResponses = $testQuestion->responses()->exists();

        if ($hasResponses) {
            return redirect()->route('admin.test-questions.index')
                ->with('error', 'Cannot delete a question that has been used in tests. Consider deactivating it instead.');
        }

        $testQuestion->delete();

        return redirect()->route('admin.test-questions.index')
            ->with('success', 'Test question deleted successfully.');
    }

    /**
     * Toggle the active status of the specified test question.
     */
    public function toggleStatus(WriterTestQuestion $question)
    {
        $question->is_active = !$question->is_active;
        $question->save();

        $status = $question->is_active ? 'activated' : 'deactivated';

        return redirect()->back()
            ->with('success', "Question {$status} successfully.");
    }
}
