<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\WriterFine;
use App\Models\WriterInfraction;
use App\Models\User;
use App\Models\Order;
use App\Services\FineCalculationService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Validation\Rule;
use Illuminate\Support\Facades\Notification;
use App\Notifications\WriterFineApplied;

class WriterFineController extends Controller
{
    protected $fineCalculationService;

    public function __construct(FineCalculationService $fineCalculationService)
    {
        $this->fineCalculationService = $fineCalculationService;
    }

    /**
     * Display a listing of writer fines
     */
    public function index(Request $request)
    {
        $query = WriterFine::with(['writer', 'infraction', 'order', 'appliedBy', 'reviewedBy'])
            ->orderBy('created_at', 'desc');

        // Apply filters
        if ($request->filled('writer_id')) {
            $query->where('writer_id', $request->writer_id);
        }

        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        if ($request->filled('type')) {
            $query->where('fine_type', $request->type);
        }

        if ($request->filled('application_method')) {
            $query->where('application_method', $request->application_method);
        }

        if ($request->filled('date_from')) {
            $query->whereDate('applied_at', '>=', $request->date_from);
        }

        if ($request->filled('date_to')) {
            $query->whereDate('applied_at', '<=', $request->date_to);
        }

        if ($request->filled('is_automatic')) {
            $query->where('is_automatic', $request->is_automatic === 'true');
        }

        $fines = $query->paginate(25);

        // Get filter options
        $writers = User::where('role', 'writer')->orderBy('name')->get();
        $statuses = WriterFine::distinct()->pluck('status');
        $types = WriterFine::distinct()->pluck('fine_type');
        $applicationMethods = WriterFine::distinct()->pluck('application_method');

        return view('admin.writer-fines.index', compact(
            'fines', 
            'writers', 
            'statuses', 
            'types', 
            'applicationMethods'
        ));
    }

    /**
     * Show the form for creating a new fine
     */
    public function create()
    {
        $writers = User::where('role', 'writer')->orderBy('name')->get();
        $infractions = WriterInfraction::where('status', 'confirmed')->with('writer')->get();
        $orders = Order::where('status', '!=', 'cancelled')->orderBy('created_at', 'desc')->get();
        
        $fineTypes = [
            'fixed_amount' => 'Fixed Amount',
            'percentage_based' => 'Percentage Based',
            'progressive' => 'Progressive',
            'opportunity_cost' => 'Opportunity Cost',
            'quality_penalty' => 'Quality Penalty',
            'deadline_violation' => 'Deadline Violation',
            'other' => 'Other'
        ];

        $applicationMethods = [
            'immediate_deduction' => 'Immediate Deduction',
            'future_earning_deduction' => 'Future Earning Deduction',
            'withdrawal_deduction' => 'Withdrawal Deduction',
            'bonus_cancellation' => 'Bonus Cancellation',
            'other' => 'Other'
        ];

        return view('admin.writer-fines.create', compact(
            'writers', 
            'infractions', 
            'orders', 
            'fineTypes', 
            'applicationMethods'
        ));
    }

    /**
     * Store a newly created fine
     */
    public function store(Request $request)
    {
        $request->validate([
            'writer_id' => 'required|exists:users,id',
            'infraction_id' => 'nullable|exists:writer_infractions,id',
            'order_id' => 'nullable|exists:orders,id',
            'fine_type' => ['required', Rule::in([
                'fixed_amount', 'percentage_based', 'progressive', 'opportunity_cost',
                'quality_penalty', 'deadline_violation', 'other'
            ])],
            'fine_amount' => 'required|numeric|min:0.01',
            'currency' => 'required|string|size:3',
            'fine_reason' => 'required|string|max:2000',
            'calculation_details' => 'nullable|string|max:2000',
            'application_method' => ['required', Rule::in([
                'immediate_deduction', 'future_earning_deduction', 'withdrawal_deduction',
                'bonus_cancellation', 'other'
            ])],
            'effective_from' => 'nullable|date|after_or_equal:today',
            'fine_configuration' => 'nullable|array',
            'manual_adjustment_notes' => 'nullable|string|max:1000',
        ]);

        try {
            DB::beginTransaction();

            $fine = WriterFine::create([
                'writer_id' => $request->writer_id,
                'infraction_id' => $request->infraction_id,
                'order_id' => $request->order_id,
                'applied_by' => auth()->id(),
                'fine_type' => $request->fine_type,
                'fine_amount' => $request->fine_amount,
                'currency' => $request->currency,
                'fine_reason' => $request->fine_reason,
                'calculation_details' => $request->calculation_details,
                'application_method' => $request->application_method,
                'status' => 'pending',
                'applied_at' => now(),
                'effective_from' => $request->effective_from,
                'remaining_amount' => $request->fine_amount,
                'fine_configuration' => $request->fine_configuration,
                'is_automatic' => false,
                'manual_adjustment_notes' => $request->manual_adjustment_notes,
            ]);

            // If this is related to an infraction, update the infraction
            if ($request->infraction_id) {
                $infraction = WriterInfraction::find($request->infraction_id);
                if ($infraction) {
                    $infraction->update([
                        'status' => 'resolved',
                        'resolved_at' => now()
                    ]);
                }
            }

            // Notify the writer
            $writer = User::find($request->writer_id);
            if ($writer) {
                $writer->notify(new WriterFineApplied($fine));
            }

            DB::commit();

            Log::info('Writer fine created', [
                'fine_id' => $fine->id,
                'writer_id' => $request->writer_id,
                'amount' => $request->fine_amount,
                'created_by' => auth()->id()
            ]);

            return redirect()->route('admin.writer-fines.show', $fine)
                ->with('success', 'Fine applied successfully.');

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Failed to create writer fine', [
                'error' => $e->getMessage(),
                'request_data' => $request->all()
            ]);

            return back()->with('error', 'Failed to apply fine. Please try again.')
                ->withInput();
        }
    }

    /**
     * Display the specified fine
     */
    public function show(WriterFine $fine)
    {
        $fine->load(['writer', 'infraction', 'order', 'appliedBy', 'reviewedBy', 'paymentRecord']);
        
        $relatedFines = WriterFine::where('writer_id', $fine->writer_id)
            ->where('id', '!=', $fine->id)
            ->orderBy('created_at', 'desc')
            ->get();

        $fineStatistics = $this->fineCalculationService->getWriterFineStatistics($fine->writer_id);

        return view('admin.writer-fines.show', compact(
            'fine', 
            'relatedFines', 
            'fineStatistics'
        ));
    }

    /**
     * Show the form for editing the specified fine
     */
    public function edit(WriterFine $fine)
    {
        $writers = User::where('role', 'writer')->orderBy('name')->get();
        $infractions = WriterInfraction::where('status', 'confirmed')->with('writer')->get();
        $orders = Order::where('status', '!=', 'cancelled')->orderBy('created_at', 'desc')->get();
        
        $fineTypes = [
            'fixed_amount' => 'Fixed Amount',
            'percentage_based' => 'Percentage Based',
            'progressive' => 'Progressive',
            'opportunity_cost' => 'Opportunity Cost',
            'quality_penalty' => 'Quality Penalty',
            'deadline_violation' => 'Deadline Violation',
            'other' => 'Other'
        ];

        $applicationMethods = [
            'immediate_deduction' => 'Immediate Deduction',
            'future_earning_deduction' => 'Future Earning Deduction',
            'withdrawal_deduction' => 'Withdrawal Deduction',
            'bonus_cancellation' => 'Bonus Cancellation',
            'other' => 'Other'
        ];

        return view('admin.writer-fines.edit', compact(
            'fine', 
            'writers', 
            'infractions', 
            'orders', 
            'fineTypes', 
            'applicationMethods'
        ));
    }

    /**
     * Update the specified fine
     */
    public function update(Request $request, WriterFine $fine)
    {
        $request->validate([
            'fine_type' => ['required', Rule::in([
                'fixed_amount', 'percentage_based', 'progressive', 'opportunity_cost',
                'quality_penalty', 'deadline_violation', 'other'
            ])],
            'fine_amount' => 'required|numeric|min:0.01',
            'fine_reason' => 'required|string|max:2000',
            'calculation_details' => 'nullable|string|max:2000',
            'application_method' => ['required', Rule::in([
                'immediate_deduction', 'future_earning_deduction', 'withdrawal_deduction',
                'bonus_cancellation', 'other'
            ])],
            'effective_from' => 'nullable|date',
            'fine_configuration' => 'nullable|array',
            'manual_adjustment_notes' => 'nullable|string|max:1000',
        ]);

        try {
            $oldAmount = $fine->fine_amount;
            
            $fine->update([
                'fine_type' => $request->fine_type,
                'fine_amount' => $request->fine_amount,
                'fine_reason' => $request->fine_reason,
                'calculation_details' => $request->calculation_details,
                'application_method' => $request->application_method,
                'effective_from' => $request->effective_from,
                'fine_configuration' => $request->fine_configuration,
                'manual_adjustment_notes' => $request->manual_adjustment_notes,
            ]);

            // Adjust remaining amount if fine amount changed
            if ($oldAmount != $request->fine_amount) {
                $difference = $request->fine_amount - $oldAmount;
                $fine->remaining_amount = max(0, $fine->remaining_amount + $difference);
                $fine->save();
            }

            Log::info('Writer fine updated', [
                'fine_id' => $fine->id,
                'updated_by' => auth()->id(),
                'old_amount' => $oldAmount,
                'new_amount' => $request->fine_amount
            ]);

            return redirect()->route('admin.writer-fines.show', $fine)
                ->with('success', 'Fine updated successfully.');

        } catch (\Exception $e) {
            Log::error('Failed to update writer fine', [
                'error' => $e->getMessage(),
                'fine_id' => $fine->id
            ]);

            return back()->with('error', 'Failed to update fine. Please try again.')
                ->withInput();
        }
    }

    /**
     * Remove the specified fine
     */
    public function destroy(WriterFine $fine)
    {
        try {
            // Check if fine has been applied
            if ($fine->status === 'applied' && $fine->amount_deducted > 0) {
                return back()->with('error', 'Cannot delete fine that has been applied.');
            }

            $fine->delete();

            Log::info('Writer fine deleted', [
                'fine_id' => $fine->id,
                'deleted_by' => auth()->id()
            ]);

            return redirect()->route('admin.writer-fines.index')
                ->with('success', 'Fine deleted successfully.');

        } catch (\Exception $e) {
            Log::error('Failed to delete writer fine', [
                'error' => $e->getMessage(),
                'fine_id' => $fine->id
            ]);

            return back()->with('error', 'Failed to delete fine. Please try again.');
        }
    }

    /**
     * Change fine status
     */
    public function changeStatus(Request $request, WriterFine $fine)
    {
        $request->validate([
            'status' => ['required', Rule::in([
                'pending', 'applied', 'disputed', 'reduced', 'waived', 'reversed'
            ])],
            'review_notes' => 'nullable|string|max:1000',
        ]);

        try {
            $oldStatus = $fine->status;
            
            $updateData = [
                'status' => $request->status,
                'reviewed_by' => auth()->id(),
                'review_notes' => $request->review_notes,
            ];

            if ($request->status === 'applied') {
                $updateData['deducted_at'] = now();
            }

            if ($request->status === 'reduced') {
                $request->validate([
                    'reduced_amount' => 'required|numeric|min:0|max:' . $fine->fine_amount
                ]);
                $updateData['fine_amount'] = $request->reduced_amount;
                $updateData['remaining_amount'] = $request->reduced_amount;
            }

            if ($request->status === 'waived') {
                $updateData['remaining_amount'] = 0;
            }

            $fine->update($updateData);

            Log::info('Writer fine status changed', [
                'fine_id' => $fine->id,
                'old_status' => $oldStatus,
                'new_status' => $request->status,
                'changed_by' => auth()->id()
            ]);

            return back()->with('success', 'Fine status updated successfully.');

        } catch (\Exception $e) {
            Log::error('Failed to change fine status', [
                'error' => $e->getMessage(),
                'fine_id' => $fine->id
            ]);

            return back()->with('error', 'Failed to update status. Please try again.');
        }
    }

    /**
     * Apply fine to payment
     */
    public function applyToPayment(Request $request, WriterFine $fine)
    {
        $request->validate([
            'payment_record_id' => 'required|exists:writer_payment_records,id',
            'deduction_amount' => 'required|numeric|min:0.01|max:' . $fine->remaining_amount,
        ]);

        try {
            DB::beginTransaction();

            $paymentRecord = \App\Models\WriterPaymentRecord::find($request->payment_record_id);
            
            if (!$paymentRecord) {
                throw new \Exception('Payment record not found.');
            }

            if ($paymentRecord->payment_amount < $request->deduction_amount) {
                throw new \Exception('Deduction amount exceeds payment amount.');
            }

            // Apply the deduction
            $fine->amount_deducted += $request->deduction_amount;
            $fine->remaining_amount = max(0, $fine->remaining_amount - $request->deduction_amount);
            $fine->payment_record_id = $paymentRecord->id;
            $fine->deducted_at = now();

            if ($fine->remaining_amount <= 0) {
                $fine->status = 'applied';
            }

            $fine->save();

            DB::commit();

            Log::info('Fine applied to payment', [
                'fine_id' => $fine->id,
                'payment_record_id' => $request->payment_record_id,
                'deduction_amount' => $request->deduction_amount,
                'applied_by' => auth()->id()
            ]);

            return back()->with('success', 'Fine applied to payment successfully.');

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Failed to apply fine to payment', [
                'error' => $e->getMessage(),
                'fine_id' => $fine->id
            ]);

            return back()->with('error', 'Failed to apply fine to payment: ' . $e->getMessage());
        }
    }

    /**
     * Export fines data
     */
    public function export(Request $request)
    {
        $query = WriterFine::with(['writer', 'infraction', 'order']);

        // Apply filters
        if ($request->filled('writer_id')) {
            $query->where('writer_id', $request->writer_id);
        }

        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        if ($request->filled('type')) {
            $query->where('fine_type', $request->type);
        }

        $fines = $query->get();

        // Generate CSV
        $filename = 'writer_fines_' . now()->format('Y-m-d_H-i-s') . '.csv';
        
        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => 'attachment; filename="' . $filename . '"',
        ];

        $callback = function() use ($fines) {
            $file = fopen('php://output', 'w');
            
            // CSV headers
            fputcsv($file, [
                'ID', 'Writer', 'Type', 'Amount', 'Status', 'Reason', 
                'Applied At', 'Effective From', 'Remaining Amount'
            ]);

            foreach ($fines as $fine) {
                fputcsv($file, [
                    $fine->id,
                    $fine->writer->name ?? 'N/A',
                    $fine->fine_type,
                    $fine->fine_amount,
                    $fine->status,
                    $fine->fine_reason,
                    $fine->applied_at->format('Y-m-d H:i:s'),
                    $fine->effective_from ? $fine->effective_from->format('Y-m-d H:i:s') : 'N/A',
                    $fine->remaining_amount
                ]);
            }

            fclose($file);
        };

        return response()->stream($callback, 200, $headers);
    }

    /**
     * Get fine statistics
     */
    public function statistics()
    {
        $totalFines = WriterFine::count();
        $pendingFines = WriterFine::pending()->count();
        $appliedFines = WriterFine::applied()->count();
        $totalAmount = WriterFine::sum('fine_amount');
        $totalDeducted = WriterFine::sum('amount_deducted');
        $totalRemaining = WriterFine::sum('remaining_amount');

        $finesByType = WriterFine::selectRaw('fine_type, COUNT(*) as count, SUM(fine_amount) as total_amount')
            ->groupBy('fine_type')
            ->orderBy('total_amount', 'desc')
            ->get();

        $finesByStatus = WriterFine::selectRaw('status, COUNT(*) as count, SUM(fine_amount) as total_amount')
            ->groupBy('status')
            ->orderBy('total_amount', 'desc')
            ->get();

        $monthlyFines = WriterFine::selectRaw('DATE_FORMAT(applied_at, "%Y-%m") as month, COUNT(*) as count, SUM(fine_amount) as total_amount')
            ->groupBy('month')
            ->orderBy('month', 'desc')
            ->limit(12)
            ->get();

        return view('admin.writer-fines.statistics', compact(
            'totalFines',
            'pendingFines',
            'appliedFines',
            'totalAmount',
            'totalDeducted',
            'totalRemaining',
            'finesByType',
            'finesByStatus',
            'monthlyFines'
        ));
    }
} 