<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\WriterInfraction;
use App\Models\WriterFine;
use App\Models\User;
use App\Models\Order;
use App\Services\FineCalculationService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Validation\Rule;
use Illuminate\Support\Facades\Notification;
use App\Notifications\WriterInfractionReported;
use Inertia\Inertia;

class WriterInfractionController extends Controller
{
    protected $fineCalculationService;

    public function __construct(FineCalculationService $fineCalculationService)
    {
        $this->fineCalculationService = $fineCalculationService;
    }

    /**
     * Display a listing of writer infractions
     */
    public function index(Request $request)
    {
        $query = WriterInfraction::with(['writer', 'order', 'reportedBy', 'reviewedBy'])
            ->orderBy('created_at', 'desc');

        // Apply filters
        if ($request->filled('writer_id')) {
            $query->where('writer_id', $request->writer_id);
        }

        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        if ($request->filled('severity')) {
            $query->where('severity_level', $request->severity);
        }

        if ($request->filled('type')) {
            $query->where('infraction_type', $request->type);
        }

        if ($request->filled('date_from')) {
            $query->whereDate('reported_at', '>=', $request->date_from);
        }

        if ($request->filled('date_to')) {
            $query->whereDate('reported_at', '<=', $request->date_to);
        }

        if ($request->filled('escalated')) {
            $query->where('escalated', $request->escalated === 'true');
        }

        $infractions = $query->paginate(25);

        // Get filter options
        $writers = User::where('user_type', 'writer')->orderBy('name')->get();
        
        // Handle empty table case for filter options
        $statuses = WriterInfraction::count() > 0 ? WriterInfraction::distinct()->pluck('status') : collect();
        $severities = WriterInfraction::count() > 0 ? WriterInfraction::distinct()->pluck('severity_level') : collect();
        $types = WriterInfraction::count() > 0 ? WriterInfraction::distinct()->pluck('infraction_type') : collect();

        return inertia('Admin/WriterInfractions/Index', compact(
            'infractions', 
            'writers', 
            'statuses', 
            'severities', 
            'types'
        ));
    }

    /**
     * Show the form for creating a new infraction
     */
    public function create()
    {
        $writers = User::where('user_type', 'writer')->orderBy('name')->get();
        $orders = Order::where('order_status', '!=', 'cancelled')->orderBy('dateposted', 'desc')->get();
        
        $infractionTypes = [
            'deadline_violation' => 'Deadline Violation',
            'quality_issues' => 'Quality Issues',
            'plagiarism' => 'Plagiarism',
            'communication_failure' => 'Communication Failure',
            'policy_violation' => 'Policy Violation',
            'unprofessional_behavior' => 'Unprofessional Behavior',
            'client_complaint' => 'Client Complaint',
            'other' => 'Other'
        ];

        $severityLevels = [
            'warning' => 'Warning',
            'minor' => 'Minor',
            'major' => 'Major',
            'critical' => 'Critical'
        ];

        return inertia('Admin/WriterInfractions/Create', compact(
            'writers', 
            'orders', 
            'infractionTypes', 
            'severityLevels'
        ));
    }

    /**
     * Store a newly created infraction
     */
    public function store(Request $request)
    {
        $request->validate([
            'writer_id' => 'required|exists:users,id',
            'order_id' => 'nullable|exists:orders,id',
            'infraction_type' => ['required', Rule::in([
                'deadline_violation', 'quality_issues', 'plagiarism', 
                'communication_failure', 'policy_violation', 'unprofessional_behavior',
                'client_complaint', 'other'
            ])],
            'severity_level' => ['required', Rule::in(['warning', 'minor', 'major', 'critical'])],
            'description' => 'required|string|max:2000',
            'evidence' => 'nullable|string|max:2000',
            'admin_notes' => 'nullable|string|max:1000',
            'occurred_at' => 'nullable|date',
            'estimated_cost_impact' => 'nullable|numeric|min:0',
            'client_satisfaction_impact' => 'nullable|integer|min:1|max:5',
            'impact_assessment' => 'nullable|string|max:1000',
        ]);

        try {
            DB::beginTransaction();

            // Check for repeat offenses
            $repeatCount = WriterInfraction::where('writer_id', $request->writer_id)
                ->where('infraction_type', $request->infraction_type)
                ->where('status', '!=', 'dismissed')
                ->count() + 1;

            $infraction = WriterInfraction::create([
                'writer_id' => $request->writer_id,
                'order_id' => $request->order_id,
                'reported_by' => auth()->id(),
                'infraction_type' => $request->infraction_type,
                'severity_level' => $request->severity_level,
                'description' => $request->description,
                'evidence' => $request->evidence,
                'admin_notes' => $request->admin_notes,
                'occurred_at' => $request->occurred_at,
                'reported_at' => now(),
                'estimated_cost_impact' => $request->estimated_cost_impact ?? 0,
                'client_satisfaction_impact' => $request->client_satisfaction_impact,
                'impact_assessment' => $request->impact_assessment,
                'repeat_count' => $repeatCount,
                'status' => 'reported',
                'appeal_deadline' => now()->addDays(7), // 7 days to appeal
            ]);

            // Check if this should trigger escalation
            if ($infraction->shouldTriggerEscalation()) {
                $infraction->update([
                    'escalated' => true,
                    'escalation_notes' => 'Automatically escalated due to repeat offense or critical severity'
                ]);
            }

            // Notify the writer
            $writer = User::find($request->writer_id);
            if ($writer) {
                $writer->notify(new WriterInfractionReported($infraction));
            }

            DB::commit();

            Log::info('Writer infraction created', [
                'infraction_id' => $infraction->id,
                'writer_id' => $request->writer_id,
                'created_by' => auth()->id()
            ]);

            return redirect()->route('admin.writer-infractions.show', $infraction)
                ->with('success', 'Infraction reported successfully.');

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Failed to create writer infraction', [
                'error' => $e->getMessage(),
                'request_data' => $request->all()
            ]);

            return back()->with('error', 'Failed to report infraction. Please try again.')
                ->withInput();
        }
    }

    /**
     * Display the specified infraction
     */
    public function show(WriterInfraction $infraction)
    {
        $infraction->load(['writer', 'order', 'reportedBy', 'reviewedBy', 'fines']);
        
        $relatedInfractions = WriterInfraction::where('writer_id', $infraction->writer_id)
            ->where('id', '!=', $infraction->id)
            ->orderBy('created_at', 'desc')
            ->get();

        $fineStatistics = $this->fineCalculationService->getWriterFineStatistics($infraction->writer_id);

        return inertia('Admin/WriterInfractions/Show', compact(
            'infraction', 
            'relatedInfractions', 
            'fineStatistics'
        ));
    }

    /**
     * Show the form for editing the specified infraction
     */
    public function edit(WriterInfraction $infraction)
    {
        $writers = User::where('user_type', 'writer')->orderBy('name')->get();
        $orders = Order::where('order_status', '!=', 'cancelled')->orderBy('dateposted', 'desc')->get();
        
        $infractionTypes = [
            'deadline_violation' => 'Deadline Violation',
            'quality_issues' => 'Quality Issues',
            'plagiarism' => 'Plagiarism',
            'communication_failure' => 'Communication Failure',
            'policy_violation' => 'Policy Violation',
            'unprofessional_behavior' => 'Unprofessional Behavior',
            'client_complaint' => 'Client Complaint',
            'other' => 'Other'
        ];

        $severityLevels = [
            'warning' => 'Warning',
            'minor' => 'Minor',
            'major' => 'Major',
            'critical' => 'Critical'
        ];

        return inertia('Admin/WriterInfractions/Edit', compact(
            'infraction', 
            'writers', 
            'orders', 
            'infractionTypes', 
            'severityLevels'
        ));
    }

    /**
     * Update the specified infraction
     */
    public function update(Request $request, WriterInfraction $infraction)
    {
        $request->validate([
            'infraction_type' => ['required', Rule::in([
                'deadline_violation', 'quality_issues', 'plagiarism', 
                'communication_failure', 'policy_violation', 'unprofessional_behavior',
                'client_complaint', 'other'
            ])],
            'severity_level' => ['required', Rule::in(['warning', 'minor', 'major', 'critical'])],
            'description' => 'required|string|max:2000',
            'evidence' => 'nullable|string|max:2000',
            'admin_notes' => 'nullable|string|max:1000',
            'occurred_at' => 'nullable|date',
            'estimated_cost_impact' => 'nullable|numeric|min:0',
            'client_satisfaction_impact' => 'nullable|integer|min:1|max:5',
            'impact_assessment' => 'nullable|string|max:1000',
        ]);

        try {
            $infraction->update([
                'infraction_type' => $request->infraction_type,
                'severity_level' => $request->severity_level,
                'description' => $request->description,
                'evidence' => $request->evidence,
                'admin_notes' => $request->admin_notes,
                'occurred_at' => $request->occurred_at,
                'estimated_cost_impact' => $request->estimated_cost_impact ?? 0,
                'client_satisfaction_impact' => $request->client_satisfaction_impact,
                'impact_assessment' => $request->impact_assessment,
            ]);

            Log::info('Writer infraction updated', [
                'infraction_id' => $infraction->id,
                'updated_by' => auth()->id()
            ]);

            return redirect()->route('admin.writer-infractions.show', $infraction)
                ->with('success', 'Infraction updated successfully.');

        } catch (\Exception $e) {
            Log::error('Failed to update writer infraction', [
                'error' => $e->getMessage(),
                'infraction_id' => $infraction->id
            ]);

            return back()->with('error', 'Failed to update infraction. Please try again.')
                ->withInput();
        }
    }

    /**
     * Remove the specified infraction
     */
    public function destroy(WriterInfraction $infraction)
    {
        try {
            // Check if infraction has associated fines
            if ($infraction->fines()->exists()) {
                return back()->with('error', 'Cannot delete infraction with associated fines.');
            }

            $infraction->delete();

            Log::info('Writer infraction deleted', [
                'infraction_id' => $infraction->id,
                'deleted_by' => auth()->id()
            ]);

            return redirect()->route('admin.writer-infractions.index')
                ->with('success', 'Infraction deleted successfully.');

        } catch (\Exception $e) {
            Log::error('Failed to delete writer infraction', [
                'error' => $e->getMessage(),
                'infraction_id' => $infraction->id
            ]);

            return back()->with('error', 'Failed to delete infraction. Please try again.');
        }
    }

    /**
     * Change infraction status
     */
    public function changeStatus(Request $request, WriterInfraction $infraction)
    {
        $request->validate([
            'status' => ['required', Rule::in([
                'reported', 'under_review', 'confirmed', 'dismissed', 
                'resolved', 'appealed', 'appeal_approved', 'appeal_rejected'
            ])],
            'admin_notes' => 'nullable|string|max:1000',
        ]);

        try {
            $oldStatus = $infraction->status;
            
            $updateData = [
                'status' => $request->status,
                'admin_notes' => $request->admin_notes,
            ];

            if ($request->status === 'confirmed') {
                $updateData['reviewed_by'] = auth()->id();
                $updateData['reviewed_at'] = now();
            }

            if ($request->status === 'resolved') {
                $updateData['resolved_at'] = now();
            }

            $infraction->update($updateData);

            Log::info('Writer infraction status changed', [
                'infraction_id' => $infraction->id,
                'old_status' => $oldStatus,
                'new_status' => $request->status,
                'changed_by' => auth()->id()
            ]);

            return back()->with('success', 'Infraction status updated successfully.');

        } catch (\Exception $e) {
            Log::error('Failed to change infraction status', [
                'error' => $e->getMessage(),
                'infraction_id' => $infraction->id
            ]);

            return back()->with('error', 'Failed to update status. Please try again.');
        }
    }

    /**
     * Export infractions data
     */
    public function export(Request $request)
    {
        $query = WriterInfraction::with(['writer', 'order', 'reportedBy']);

        // Apply filters
        if ($request->filled('writer_id')) {
            $query->where('writer_id', $request->writer_id);
        }

        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        if ($request->filled('severity')) {
            $query->where('severity_level', $request->severity);
        }

        if ($request->filled('type')) {
            $query->where('infraction_type', $request->type);
        }

        $infractions = $query->get();

        // Generate CSV
        $filename = 'writer_infractions_' . now()->format('Y-m-d_H-i-s') . '.csv';
        
        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => 'attachment; filename="' . $filename . '"',
        ];

        $callback = function() use ($infractions) {
            $file = fopen('php://output', 'w');
            
            // CSV headers
            fputcsv($file, [
                'ID', 'Writer', 'Type', 'Severity', 'Status', 'Description', 
                'Reported At', 'Occurred At', 'Cost Impact', 'Repeat Count'
            ]);

            foreach ($infractions as $infraction) {
                fputcsv($file, [
                    $infraction->id,
                    $infraction->writer->name ?? 'N/A',
                    $infraction->infraction_type,
                    $infraction->severity_level,
                    $infraction->status,
                    $infraction->description,
                    $infraction->reported_at->format('Y-m-d H:i:s'),
                    $infraction->occurred_at ? $infraction->occurred_at->format('Y-m-d H:i:s') : 'N/A',
                    $infraction->estimated_cost_impact,
                    $infraction->repeat_count
                ]);
            }

            fclose($file);
        };

        return response()->stream($callback, 200, $headers);
    }

    /**
     * Get infraction statistics
     */
    public function statistics()
    {
        $totalInfractions = WriterInfraction::count();
        $activeInfractions = WriterInfraction::active()->count();
        $resolvedInfractions = WriterInfraction::resolved()->count();
        $escalatedInfractions = WriterInfraction::escalated()->count();

        $infractionsByType = WriterInfraction::selectRaw('infraction_type, COUNT(*) as count')
            ->groupBy('infraction_type')
            ->orderBy('count', 'desc')
            ->get();

        $infractionsBySeverity = WriterInfraction::selectRaw('severity_level, COUNT(*) as count')
            ->groupBy('severity_level')
            ->orderBy('count', 'desc')
            ->get();

        $infractionsByStatus = WriterInfraction::selectRaw('status, COUNT(*) as count')
            ->groupBy('status')
            ->orderBy('count', 'desc')
            ->get();

        $monthlyInfractions = WriterInfraction::selectRaw('DATE_FORMAT(reported_at, "%Y-%m") as month, COUNT(*) as count')
            ->groupBy('month')
            ->orderBy('month', 'desc')
            ->limit(12)
            ->get();

        return inertia('Admin/WriterInfractions/Statistics', compact(
            'totalInfractions',
            'activeInfractions',
            'resolvedInfractions',
            'escalatedInfractions',
            'infractionsByType',
            'infractionsBySeverity',
            'infractionsByStatus',
            'monthlyInfractions'
        ));
    }
} 