<?php

namespace App\Http\Controllers;

use App\Models\User;
use Inertia\Inertia;
use App\Models\Order;
use Inertia\Response;
use App\Models\Config;
use App\Models\Coupon;
use App\Models\Pricing;
use App\Models\Subject;
use App\Models\PaperType;
use Illuminate\Http\Request;
use App\Models\AcademicLevel;
use App\Models\AdminSetting;
use Illuminate\Support\Facades\Auth;
use App\Notifications\OrderStatusChange;
use App\Notifications\OrderDeadlineChange;
use Illuminate\Support\Facades\Notification;
use App\Notifications\AdminOrderPlacedNotification;
use App\Notifications\ClientOrderPlacedNotification;

class AdminOrdersController extends Controller
{
    public function index()
    {
        //total order count
        $totalOrders = Order::count();
        //assigned orders count
        $assignedOrders = Order::where('order_status', 'assigned')->count();

        //pending orders count
        $pendingOrders = Order::where('order_status', 'bidding')
            ->where('payment_status', 'unpaid')
            ->count();
        // Fetch all orders from the database
        $orders = Order::orderBy('id', 'desc')->paginate(5);
        return Inertia::render('Admin/Orders/Index', ['orders' => $orders, 'totalOrders' => $totalOrders, 'assignedOrders' => $assignedOrders, 'pendingOrders' => $pendingOrders]);
    }

    public function search(Request $request): Response
    {
        $query = $request->input('query');
        $paymentStatus = $request->input('paymentStatus');
        $orderStatus = $request->input('orderStatus');

        $orders = Order::query()
            ->search($query)
            ->paymentStatus($paymentStatus)
            ->orderStatus($orderStatus)
            ->paginate(5)
            ->appends($request->query()); // Preserve query parameters in pagination links

        return Inertia::render('Admin/Orders/Index', [
            'orders' => $orders,
            'query' => $query,
            'filters' => [
                'paymentStatus' => $paymentStatus,
                'orderStatus' => $orderStatus,
            ],
        ]);
    }


    public function create()
    {
        $academicLevels = AcademicLevel::all();
        $deadlines = Pricing::all();
        $paperTypes = PaperType::all();
        $subjects = Subject::all();
        $clients = User::select('id', 'name')->get();
        $couponCode = \App\Models\Coupon::where('id', 1)->value('coupon_code');
        return Inertia::render('Admin/Orders/Create', [
            'academicLevels' => $academicLevels,
            'deadlines' => $deadlines,
            'paperTypes' => $paperTypes,
            'subjects' => $subjects,
            'couponEnabled' => AdminSetting::where('enable_coupon', 'yes')->exists(),
            'couponCode' => $couponCode,
            'clients' => $clients
        ]);
    }

    public function store(Request $request)
    {
        $timezone = $request->input('clientTimezone');



        // Validate the request data
        $validated = $request->validate([
            'title' => 'required|string|max:250',
            'instructions' => 'required|string',
            'pages' => 'required|integer|min:1',
            'subject' => 'required|string',
            'aclevel' => 'required|string',
            'papertype' => 'required|string',
            'ppslides' => 'nullable|integer|min:0',
            'deadline' => 'required|string',
            'spacing' => 'required|string',
            'totalAmount' => 'required|numeric|min:0',
            'styles' => 'required|string',
            'sources' => 'required|integer|min:0',
            'language' => 'required|string',
            'files.*' => 'nullable|file|max:10240', // 10MB max per file
        ]);
        // $maxOrderId = Order::getMaxOrderId();

        // Generate order number
        $maxOrderId = Order::max('order_number') ?? 59700;
        $orderNumber = $maxOrderId + 1;
        // Get client timezone from request
        $clientTimezone = request()->input('clientTimezone');

        // Set timezone for deadline calculation
        $deadline = now()->setTimezone($clientTimezone);



        // Parse deadline string to get actual deadline date
        $deadlineParts = explode('#', $validated['deadline']);
        $deadlineValue = (int)$deadlineParts[1];
        $deadlineUnit = $deadlineParts[2] ?? 'Hours';

        $deadline = $deadlineUnit === 'Days'
            ? $deadline->addDays($deadlineValue)
            : $deadline->addHours($deadlineValue);

        $academicLevel = optional(explode('#', $request->aclevel))[1] ?? null;
        $subject = optional(explode('#', $request->subject))[1] ?? null;
        $paperType = optional(explode('#', $request->papertype))[1] ?? null;
        $spacing  = optional(explode('#', $request->spacing))[1] ?? null;

        // Create the order
        $order = new Order();

        $writerDeadline = $order->calculateWriterDeadline($deadline);
        $order->writer_deadline = $writerDeadline;

        $netAmount = request()->input('netAmount');
        $writerAmount = $order->calculateWriterAmount($netAmount);
        $order->writer_amount = $writerAmount;
        $order->order_number = $orderNumber;
        $order->user_id = request()->input('user_id');
        $order->title = $validated['title'];
        $order->type_of_paper = $paperType;
        $order->subject = $subject;
        $order->academic_level = $academicLevel;
        $order->urgency = $request->input('urgency');
        $order->pages = $validated['pages'];
        $order->instructions = $validated['instructions'];
        $order->styles = $validated['styles'];
        $order->dateposted = now()->setTimezone($timezone);
        $order->deadline = $deadline;
        $order->client_timezone = $timezone;
        $order->sources = $validated['sources'];
        $order->order_amount = $validated['totalAmount'];
        $order->net_amount = $netAmount;
        $order->discount = request()->input('discount');
        $order->payment_status = 'unpaid';
        $order->order_status = 'bidding';
        $order->powerpoint_slides = $validated['ppslides'];
        $order->spacing = $spacing;

        $order->save();



        // Fetch the admin email from the .env file
        $adminEmail = env('ADMIN_EMAIL');

        // Fetch the client details
        $client = User::find($order->user_id);

        // Notify the admin
        Notification::route('mail', $adminEmail)->notify(new AdminOrderPlacedNotification($order, $client->name));

        // Notify the client
        $client->notify(new ClientOrderPlacedNotification($order));

        return redirect()->route('admin.orders.show', $order->id)->with('message', 'Order created successfully!');
    }


    // Add a new method to validate coupons
    public function couponValidate(Request $request)
    {
        $coupon = Coupon::where('coupon_code', $request->code)->first();

        if (!$coupon) {
            return response()->json(['valid' => false]);
        }

        return response()->json([
            'valid' => true,
            'type' => $coupon->discount_type,
            'amount' => $coupon->discount_amount
        ]);
    }


    public function show($id)
    {
        $isAdmin = in_array(Auth::user()->user_type, ['admin', 'super_admin']);
        $order = Order::with([
            'user:id,name,email,created_at,status', 
            'orderFiles',
            'writer:id,name,email',
            'writer.writerProfile:id,user_id,nickname'
        ])->find($id);
        return Inertia::render('Admin/Orders/Show', ['order' => $order, 'isAdmin' => $isAdmin]);
    }

    public function updateDeadline(Request $request, $id)
    {
        $request->validate([
            'deadline' => 'required|string',
        ]);

        $deadline = $request->deadline;

        \Log::info('Updating deadline', [
            'order_id' => $id,
            'requested_deadline' => $deadline,
        ]);

        try {
            $date = new \DateTime($deadline);
            $deadline = $date->format('Y-m-d H:i:s');
        } catch (\Exception $e) {
            \Log::error('Invalid date format', ['error' => $e->getMessage()]);
            return response()->json([
                'message' => 'Invalid date format'
            ], 422);
        }

        $order = Order::findOrFail($id);

        \Log::info('Order found', [
            'order_dateposted' => $order->dateposted,
            'new_deadline' => $deadline,
            'comparison' => [
                'deadline_timestamp' => strtotime($deadline),
                'dateposted_timestamp' => strtotime($order->dateposted),
            ]
        ]);

        // Check if new deadline is after the posted date
        if (strtotime($deadline) <= strtotime($order->dateposted)) {
            \Log::warning('Deadline validation failed - deadline not after posted date');
            return response()->json([
                'message' => 'New deadline must be after the order posted date',
                'debug' => [
                    'deadline' => $deadline,
                    'dateposted' => $order->dateposted,
                ]
            ], 422);
        }
        $client = $order->user;
        
        if (!$client) {
            \Log::error('Order has no associated client', ['order_id' => $id]);
            return response()->json([
                'message' => 'Order has no associated client'
            ], 422);
        }
        
        // Notify the client
        $client->notify(new OrderDeadlineChange($order));
        $order->deadline = $deadline;
        $order->save();

        \Log::info('Deadline updated successfully', ['order_id' => $id]);

        return response()->json([
            'message' => 'Deadline updated successfully',
            'order' => $order
        ]);
    }

    public function updateWriterDeadline(Request $request, $id)
    {
        $request->validate([
            'writer_deadline' => 'required|string',
        ]);

        $writerDeadline = $request->writer_deadline;

        \Log::info('Updating writer deadline', [
            'order_id' => $id,
            'requested_writer_deadline' => $writerDeadline,
        ]);

        try {
            $date = new \DateTime($writerDeadline);
            $writerDeadline = $date->format('Y-m-d H:i:s');
        } catch (\Exception $e) {
            \Log::error('Invalid date format', ['error' => $e->getMessage()]);
            return response()->json([
                'message' => 'Invalid date format'
            ], 422);
        }

        $order = Order::findOrFail($id);

        \Log::info('Order found', [
            'order_dateposted' => $order->dateposted,
            'new_writer_deadline' => $writerDeadline,
            'comparison' => [
                'writer_deadline_timestamp' => strtotime($writerDeadline),
                'dateposted_timestamp' => strtotime($order->dateposted),
            ]
        ]);

        // Check if new deadline is after the posted date
        if (strtotime($writerDeadline) <= strtotime($order->dateposted)) {
            \Log::warning('Writer deadline validation failed - deadline not after posted date');
            return response()->json([
                'message' => 'New Writer deadline must be after the order posted date',
                'debug' => [
                    'writer_deadline' => $writerDeadline,
                    'dateposted' => $order->dateposted,
                ]
            ], 422);
        }
        $client = $order->user;
        
        if (!$client) {
            \Log::error('Order has no associated client', ['order_id' => $id]);
            return response()->json([
                'message' => 'Order has no associated client'
            ], 422);
        }
        
        // Notify the client
        $client->notify(new OrderDeadlineChange($order));
        $order->writer_deadline = $writerDeadline;
        $order->save();

        \Log::info('Writer deadline updated successfully', ['order_id' => $id]);

        return response()->json([
            'message' => 'Writer Deadline updated successfully',
            'order' => $order
        ]);
    }

    public function updatePaymentStatus(Request $request, $id)
    {
        $request->validate([
            'payment_status' => 'required|string|in:paid,unpaid,refunded,partially_refunded',
        ]);

        $order = Order::findOrFail($id);
        $order->payment_status = $request->payment_status;
        $order->save();

        return response()->json([
            'message' => 'Payment status updated successfully',
            'order' => $order
        ]);
    }

    public function updateOrderStatus(Request $request, $id)
    {
        $request->validate([
            'order_status' => 'required|string|in:bidding,assigned,completed,revision,editing,cancelled,approved',
        ]);

        $order = Order::findOrFail($id);
        $order->order_status = $request->order_status;
        $order->save();

        $client = User::find($order->user_id);
        $client->notify(new OrderStatusChange($order));

        return response()->json([
            'message' => 'Order status updated successfully',
            'order' => $order
        ]);
    }

    public function destroy($id)
    {
        $order = Order::findOrFail($id);
        $order->delete();
        return redirect()->back()->with('success', 'Order deleted successfully');
    }
}
