<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Services\UrgentOrderNotificationService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class UrgentOrdersController extends Controller
{
    protected UrgentOrderNotificationService $urgentOrderService;

    public function __construct(UrgentOrderNotificationService $urgentOrderService)
    {
        $this->middleware('auth');
        $this->urgentOrderService = $urgentOrderService;
    }

    /**
     * Get urgent orders with filtering options
     */
    public function index(Request $request)
    {
        $user = Auth::user();
        
        // Get urgent orders for the authenticated user
        $urgentOrders = $this->urgentOrderService->getUrgentOrdersForUser($user);
        
        // Apply filters if provided
        $filteredOrders = $this->applyFilters($urgentOrders, $request);
        
        return response()->json([
            'success' => true,
            'data' => $filteredOrders,
            'counts' => $this->urgentOrderService->getUrgentOrdersCount($user),
            'filters' => $request->only(['urgency_level', 'order_status', 'search'])
        ]);
    }

    /**
     * Get urgent orders count for dashboard
     */
    public function counts()
    {
        $user = Auth::user();
        $counts = $this->urgentOrderService->getUrgentOrdersCount($user);
        
        return response()->json([
            'success' => true,
            'data' => $counts
        ]);
    }

    /**
     * Apply filters to urgent orders
     */
    private function applyFilters(array $urgentOrders, Request $request): array
    {
        $filteredOrders = $urgentOrders;
        
        // Filter by urgency level
        if ($request->filled('urgency_level')) {
            $urgencyLevel = $request->input('urgency_level');
            $filteredOrders = collect($urgentOrders)
                ->only([$urgencyLevel])
                ->toArray();
        }
        
        // Filter by order status
        if ($request->filled('order_status')) {
            $orderStatus = $request->input('order_status');
            $filteredOrders = collect($filteredOrders)
                ->map(function ($category) use ($orderStatus) {
                    return collect($category)
                        ->where('order_status', $orderStatus)
                        ->toArray();
                })
                ->toArray();
        }
        
        // Search by order title or ID
        if ($request->filled('search')) {
            $search = $request->input('search');
            $filteredOrders = collect($filteredOrders)
                ->map(function ($category) use ($search) {
                    return collect($category)
                        ->filter(function ($order) use ($search) {
                            return stripos($order['title'], $search) !== false ||
                                   stripos($order['order_number'], $search) !== false;
                        })
                        ->toArray();
                })
                ->toArray();
        }
        
        return $filteredOrders;
    }

    /**
     * Get available filter options
     */
    public function filterOptions()
    {
        $user = Auth::user();
        $urgentOrders = $this->urgentOrderService->getUrgentOrdersForUser($user);
        
        // Get unique order statuses from urgent orders
        $orderStatuses = collect($urgentOrders)
            ->flatten(1)
            ->pluck('order_status')
            ->unique()
            ->values()
            ->toArray();
        
        // Get urgency levels
        $urgencyLevels = [
            'overdue' => 'Overdue',
            'critical_12h' => 'Critical (12h)',
            'urgent_24h' => 'Urgent (24h)',
            'warning_48h' => 'Warning (48h)'
        ];
        
        return response()->json([
            'success' => true,
            'data' => [
                'urgency_levels' => $urgencyLevels,
                'order_statuses' => $orderStatuses
            ]
        ]);
    }
} 