<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Http\Requests\Auth\LoginRequest;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Route;
use Inertia\Inertia;
use Inertia\Response;

class OrderAuthenticatedSessionController extends Controller
{
    /**
     * Display the login view for order context.
     */
    public function create(): Response
    {
        return Inertia::render('Auth/OrderLogin', [
            'canResetPassword' => Route::has('password.request'),
            'status' => session('status'),
        ]);
    }

    /**
     * Handle an incoming authentication request for order context.
     */
    public function store(LoginRequest $request): RedirectResponse
    {
        // Retrieve order context BEFORE session regeneration
        // Session regeneration migrates data, but we want to preserve the context explicitly
        $orderContext = \App\Services\OrderContextService::getOrderContext();
        
        $request->authenticate();

        // Regenerate session after authentication
        $request->session()->regenerate();

        $user = Auth::user();

        // Check if the user is inactive
        if ($user->status === 'inactive') {
            Auth::guard('web')->logout();
            $request->session()->invalidate();
            $request->session()->regenerateToken();

            return redirect()->route('inactive')->with('message', 'Your account is deactivated. Please contact the admin at admin@example.com.');
        }

        // Restore order context if it existed before login
        if ($orderContext && \App\Services\OrderContextService::validateContext($orderContext)) {
            // Update the context with new session ID and CSRF token after regeneration
            $updatedContext = [
                'data' => $orderContext['data'] ?? [],
                'step' => $orderContext['step'] ?? 1,
                'timestamp' => now()->timestamp,
                'session_id' => $request->session()->getId(),
                'csrf_token' => csrf_token()
            ];
            
            // Store updated context temporarily to pass to redirect
            session()->put('order_context_restore_temp', $updatedContext);
            
            // Redirect back to order form with preserved context
            return redirect()->route('order.create')->with([
                'order_context_restored' => true,
                'order_step' => $orderContext['step'] ?? 1,
                'order_data' => $orderContext['data'] ?? [],
                'welcome_message' => 'Welcome back! Your order details have been restored.'
            ]);
        }

        // Fallback to dashboard if no order context
        return redirect()->intended(route('dashboard', absolute: false));
    }

    /**
     * Destroy an authenticated session.
     */
    public function destroy(Request $request): RedirectResponse
    {
        Auth::guard('web')->logout();

        $request->session()->invalidate();

        $request->session()->regenerateToken();

        return redirect('/login');
    }
} 