<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Notifications\ClientRegistered;
use Illuminate\Auth\Events\Registered;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\Rules;
use Inertia\Inertia;
use Inertia\Response;

class OrderRegisteredUserController extends Controller
{
    /**
     * Display the registration view for order context.
     */
    public function create(): Response
    {
        return Inertia::render('Auth/OrderRegister');
    }

    /**
     * Handle an incoming registration request for order context.
     *
     * @throws \Illuminate\Validation\ValidationException
     */
    public function store(Request $request): RedirectResponse
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|string|lowercase|email|max:255|unique:' . User::class,
            'password' => ['required', 'confirmed', Rules\Password::defaults()],
        ]);

        $user = User::create([
            'name' => $request->name,
            'email' => $request->email,
            'user_type' => 'client',
            'password' => Hash::make($request->password),
            'status' => 'active',
        ]);

        // For order context, we'll skip email verification to allow immediate order completion
        // The user can verify their email later from their dashboard
        $user->markEmailAsVerified();

        event(new Registered($user));

        $user->notify(new ClientRegistered($user));

        Auth::login($user);

        // Retrieve order context BEFORE session regeneration (if any)
        $orderContext = \App\Services\OrderContextService::getOrderContext();
        
        // Regenerate session after authentication
        $request->session()->regenerate();
        
        if ($orderContext && \App\Services\OrderContextService::validateContext($orderContext)) {
            // Store updated context temporarily to pass to redirect
            session()->put('order_context_restore_temp', [
                'data' => $orderContext['data'] ?? [],
                'step' => $orderContext['step'] ?? 1,
                'timestamp' => now()->timestamp,
                'session_id' => $request->session()->getId(),
                'csrf_token' => csrf_token()
            ]);
            
            // Clear the order context from session
            \App\Services\OrderContextService::clearOrderContext();
            
            // Redirect back to order form with preserved context
            return redirect()->route('order.create')->with([
                'order_context_restored' => true,
                'order_step' => $orderContext['step'] ?? 1,
                'order_data' => $orderContext['data'] ?? [],
                'welcome_message' => 'Account created successfully! You can now continue with your order.'
            ]);
        }

        // Fallback to dashboard if no order context
        return redirect(route('dashboard', absolute: false));
    }
} 