<?php

namespace App\Http\Controllers;

use App\Models\Bid;
use App\Models\Order;
use App\Models\User;
use App\Notifications\ClientBidPlacedNotification;
use App\Notifications\AdminBidPlacedNotification;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Notification;

class BidsController extends Controller
{
    // List all bids for the logged-in writer
    public function index()
    {

        $bids = Bid::with('order')
            ->where('writer_id', Auth::id())
            ->latest()
            ->get();

        return inertia('Writer/Bids/MyBids', [
            'bids' => $bids,
        ]);
    }

    public function store(Request $request, $orderId)
    {
        $order = Order::findOrFail($orderId);

        $request->validate([
            'bid_amount' => 'required|numeric|min:0.01|max:' . $order->writer_amount,
            'completion_time' => 'required|date|after_or_equal:' . now()->toDateTimeString() . '|before_or_equal:' . $order->writer_deadline,
            'comments' => 'nullable|string|max:500',
        ]);

        // Ensure the writer hasn't already placed a bid for this order
        if (Bid::where('order_id', $orderId)->where('writer_id', Auth::id())->exists()) {
            return response()->json(['error' => 'You have already placed a bid for this order.'], 400);
        }

        // Create the bid
        $bid = Bid::create([
            'order_id' => $orderId,
            'writer_id' => Auth::id(),
            'bid_amount' => $request->bid_amount,
            'completion_time' => $request->completion_time,
            'comments' => $request->comments,
            'status' => 'pending',
        ]);

        // Send notifications
        try {
            // Notify the client
            $order->user->notify(new ClientBidPlacedNotification($order, $bid));
            
            // Notify admins
            $admins = User::where('user_type', 'admin')->get();
            Notification::send($admins, new AdminBidPlacedNotification($order, $bid));
            
        } catch (\Exception $e) {
            // Log the error but don't fail the bid creation
            \Log::error('Failed to send bid notifications', [
                'error' => $e->getMessage(),
                'order_id' => $orderId,
                'bid_id' => $bid->id
            ]);
        }

        return response()->json([
            'message' => "Bid for Order #{$order->order_number} has been placed successfully!",
            'bid' => $bid,
        ], 201);
    }

    // Show a specific bid
    public function show($id)
    {
        $bid = Bid::with('order:id,order_number')->where('id', $id)->where('writer_id', Auth::id())->firstOrFail();

        return inertia('Writer/Bids/ShowBid', [
            'bid' => $bid,
        ]);
    }

    // Edit a bid
    public function edit($id)
    {
        $bid = Bid::with(['order' => function ($query) {
            $query->select('id', 'dateposted', 'deadline', 'writer_amount', 'writer_deadline');
        }])
            ->where('id', $id)
            ->where('writer_id', Auth::id())
            ->firstOrFail();

        return inertia('Writer/Bids/EditBids', [
            'bid' => $bid,
        ]);
    }



    // Delete a bid
    public function destroy($id)
    {
        $bid = Bid::where('id', $id)->where('writer_id', Auth::id())->firstOrFail();

        $bid->delete();

        return redirect()
            ->route('writer.bids')
            ->with('message', "Bid for Order #{$bid->order->order_number} has been removed successfully!");
    }
}
