<?php

namespace App\Http\Controllers;

use App\Models\BlogPost;
use App\Models\BlogCategory;
use Illuminate\Http\Request;
use Inertia\Inertia;
use Inertia\Response;

class BlogController extends Controller
{
    /**
     * Blog index page - DATABASE DRIVEN
     */
    public function index(Request $request): Response
    {
        // Build the posts query with filtering
        $postsQuery = BlogPost::select([
            'id',
            'slug',
            'title',
            'excerpt',
            'featured_image',
            'featured_image_alt',
            'category_id',
            'author_id',
            'published_at',
            'read_time',
            'difficulty'
        ])
            ->with([
                'category:id,name,slug',
                'author:id,name',
                'tags:id,name'
            ])
            ->published();

        // Apply category filter
        if ($request->has('category') && $request->category !== 'all') {
            $postsQuery->whereHas('category', function ($query) use ($request) {
                $query->where('slug', $request->category);
            });
        }

        // Apply search filter
        if ($request->has('search') && !empty($request->search)) {
            $searchTerm = $request->search;
            $postsQuery->where(function ($query) use ($searchTerm) {
                $query->where('title', 'like', "%{$searchTerm}%")
                    ->orWhere('excerpt', 'like', "%{$searchTerm}%")
                    ->orWhere('content', 'like', "%{$searchTerm}%")
                    ->orWhereHas('tags', function ($tagQuery) use ($searchTerm) {
                        $tagQuery->where('name', 'like', "%{$searchTerm}%");
                    });
            });
        }

        // Apply sorting
        switch ($request->input('sort', 'newest')) {
            case 'oldest':
                $postsQuery->oldest('published_at');
                break;
            case 'popular':
                // You can implement popularity logic here
                // For now, using view count or featured posts as proxy
                $postsQuery->orderBy('featured', 'desc')
                    ->latest('published_at');
                break;
            case 'newest':
            default:
                $postsQuery->latest('published_at');
                break;
        }

        // Get paginated posts
        $posts = $postsQuery->paginate(6);

        // Get categories with post counts
        $categories = BlogCategory::select([
            'id',
            'name',
            'slug'
        ])
            ->withCount('publishedPosts')
            ->orderBy('name')
            ->get();

        // Get featured posts (only if no filters are applied)
        $featured = collect();
        if (
            !$request->hasAny(['search', 'category', 'sort']) ||
            ($request->category === 'all' && empty($request->search) && $request->sort === 'newest')
        ) {

            $featured = BlogPost::select([
                'id',
                'slug',
                'title',
                'excerpt',
                'featured_image',
                'featured_image_alt',
                'category_id',
                'author_id',
                'published_at',
                'read_time',
                'difficulty'
            ])
                ->with([
                    'category:id,name,slug',
                    'author:id,name'
                ])
                ->published()
                ->where('featured', true)
                ->latest('published_at')
                ->limit(3)
                ->get();

            // If not enough featured posts, get recent posts
            if ($featured->count() < 3) {
                $additionalFeatured = BlogPost::select([
                    'id',
                    'slug',
                    'title',
                    'excerpt',
                    'featured_image',
                    'featured_image_alt',
                    'category_id',
                    'author_id',
                    'published_at',
                    'read_time',
                    'difficulty'
                ])
                    ->with([
                        'category:id,name,slug',
                        'author:id,name'
                    ])
                    ->published()
                    ->whereNotIn('id', $featured->pluck('id'))
                    ->latest('published_at')
                    ->limit(3 - $featured->count())
                    ->get();

                $featured = $featured->merge($additionalFeatured);
            }
        }

        // Prepare filters for frontend
        $filters = [
            'search' => $request->input('search', ''),
            'category' => $request->input('category', 'all'),
            'sort' => $request->input('sort', 'newest'),
        ];

        return Inertia::render('Blog/BlogIndex', [
            'posts' => $posts,
            'categories' => $categories,
            'featured' => $featured,
            'filters' => $filters,
            'meta' => [
                'title' => 'Academic Writing Blog | Expert Tips & Guides',
                'description' => 'Discover expert academic writing tips, essay guides, and study strategies to improve your academic performance.',
            ],
        ]);
    }

    public function category(Request $request, string $category): Response
    {
        $categoryModel = BlogCategory::select([
            'id',
            'slug',
            'name',
            'description',
            'meta_title',
            'meta_description',
            'icon',
            'color'
        ])
            ->where('slug', $category)
            ->firstOrFail();

        $posts = BlogPost::select([
            'id',
            'slug',
            'title',
            'excerpt',
            'featured_image',
            'featured_image_alt',
            'category_id',
            'author_id',
            'published_at',
            'read_time',
            'difficulty'
        ])
            ->with([
                'category:id,name,slug',
                'author:id,name',
                'tags:id,name'
            ])
            ->published()
            ->where('category_id', $categoryModel->id)
            ->when($request->search, function ($query, $search) {
                $query->where(function ($q) use ($search) {
                    $q->where('title', 'like', "%{$search}%")
                        ->orWhere('excerpt', 'like', "%{$search}%");
                });
            })
            ->when($request->sort, function ($query, $sort) {
                switch ($sort) {
                    case 'oldest':
                        $query->oldest('published_at');
                        break;
                    case 'newest':
                    default:
                        $query->latest('published_at');
                        break;
                }
            })
            ->paginate(12);

        $categories = BlogCategory::select([
            'id',
            'slug',
            'name'
        ])
            ->withCount('publishedPosts')
            ->orderBy('name')
            ->get();

        // Force filters to be an object, not an array
        $filters = (object) [
            'search' => $request->input('search', ''),
            'sort' => $request->input('sort', 'newest'),
        ];

        return Inertia::render('Blog/BlogCategory', [
            'category' => $categoryModel,
            'posts' => $posts,
            'categories' => $categories,
            'filters' => $filters,
        ]);
    }

    /**
     * Single blog post - DATABASE DRIVEN
     */
    public function show(string $slug): Response
    {
        $post = BlogPost::with(['category', 'author', 'tags'])
            ->published()
            ->where('slug', $slug)
            ->firstOrFail();

        // Get related posts from same category - OPTIMIZED
        $relatedPosts = BlogPost::select([
            'id',
            'slug',
            'featured_image',
            'title',
            'published_at'
        ])
            ->published()
            ->where('category_id', $post->category_id)
            ->where('id', '!=', $post->id)
            ->limit(3)
            ->get();

        // If not enough related posts, get recent posts - OPTIMIZED
        if ($relatedPosts->count() < 3) {
            $additionalPosts = BlogPost::select([
                'id',
                'slug',
                'featured_image',
                'title',
                'published_at'
            ])
                ->published()
                ->where('id', '!=', $post->id)
                ->whereNotIn('id', $relatedPosts->pluck('id'))
                ->latest('published_at')
                ->limit(3 - $relatedPosts->count())
                ->get();

            $relatedPosts = $relatedPosts->merge($additionalPosts);
        }

        // Prepare Open Graph data from social_sharing or meta_data
        $socialSharing = $post->social_sharing ?? [];
        $metaData = $post->meta_data ?? [];
        
        // Build Open Graph data - prioritize social_sharing, fallback to meta_data
        $openGraph = [
            'title' => $socialSharing['facebook_title'] ?? $socialSharing['og_title'] ?? $metaData['title'] ?? $post->title,
            'description' => $socialSharing['facebook_description'] ?? $socialSharing['og_description'] ?? $metaData['description'] ?? $post->excerpt,
            'image' => $socialSharing['og_image'] ?? $post->featured_image ?? null,
            'image_alt' => $socialSharing['og_image_alt'] ?? $post->featured_image_alt ?? $post->title,
            'type' => 'article',
            'url' => url("/blog/{$post->slug}"),
            'site_name' => $socialSharing['og_site_name'] ?? 'Academic Scribe',
            'locale' => $socialSharing['og_locale'] ?? 'en_US',
        ];
        
        // Build Twitter Card data
        $twitterCard = [
            'card' => $socialSharing['twitter_card'] ?? 'summary_large_image',
            'site' => $socialSharing['twitter_site'] ?? '@AcademicScribe',
            'creator' => $socialSharing['twitter_creator'] ?? '@AcademicScribe',
            'title' => $socialSharing['twitter_title'] ?? $openGraph['title'],
            'description' => $socialSharing['twitter_description'] ?? $openGraph['description'],
            'image' => $socialSharing['twitter_image'] ?? $openGraph['image'],
            'image_alt' => $socialSharing['twitter_image_alt'] ?? $openGraph['image_alt'],
        ];

        return Inertia::render('Blog/BlogPost', [
            'post' => $post,
            'relatedPosts' => $relatedPosts,
            'meta' => [
                'title' => $metaData['title'] ?? $post->title,
                'description' => $metaData['description'] ?? $post->excerpt,
                'canonical' => "/blog/{$post->slug}",
            ],
            'openGraph' => $openGraph,
            'twitterCard' => $twitterCard,
            'breadcrumb' => [
                ['name' => 'Home', 'url' => '/'],
                ['name' => 'Blog', 'url' => '/blog'],
                ['name' => $post->category->name, 'url' => "/blog/category/{$post->category->slug}"],
                ['name' => $post->title, 'url' => "/blog/{$post->slug}"],
            ],
        ]);
    }

    /**
     * API endpoint for dynamic post loading
     */
    public function apiPosts(Request $request)
    {
        $posts = BlogPost::with(['category', 'author', 'tags'])
            ->published()
            ->when($request->category, function ($query, $category) {
                $query->whereHas('category', function ($q) use ($category) {
                    $q->where('slug', $category);
                });
            })
            ->when($request->search, function ($query, $search) {
                $query->where(function ($q) use ($search) {
                    $q->where('title', 'like', "%{$search}%")
                        ->orWhere('excerpt', 'like', "%{$search}%")
                        ->orWhereHas('tags', function ($tagQuery) use ($search) {
                            $tagQuery->where('name', 'like', "%{$search}%");
                        });
                });
            })
            ->when($request->sort, function ($query, $sort) {
                switch ($sort) {
                    case 'oldest':
                        $query->oldest('published_at');
                        break;
                    case 'popular':
                        $query->orderBy('featured', 'desc')
                            ->latest('published_at');
                        break;
                    case 'newest':
                    default:
                        $query->latest('published_at');
                        break;
                }
            })
            ->paginate($request->per_page ?? 12);

        return response()->json($posts);
    }
}
