<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;
use Carbon\Carbon;

class BlogUploadController extends Controller
{
    /**
     * Upload blog JavaScript files
     */
    public function upload(Request $request): JsonResponse
    {
        try {
            $config = config('blog_uploads');

            // Validate request
            $request->validate([
                'blog_files' => 'required|array',
                'blog_files.*' => 'required|file|max:' . ($config['file_validation']['max_file_size'] / 1024),
            ]);

            $files = $request->file('blog_files');
            $uploadResults = [];
            $uploadId = Str::uuid();

            foreach ($files as $file) {
                $result = $this->processFileUpload($file, $uploadId, $config);
                $uploadResults[] = $result;
            }

            // Log upload attempt
            if ($config['logging']['log_uploads']) {
                Log::info('Blog files uploaded', [
                    'upload_id' => $uploadId,
                    'files_count' => count($files),
                    'results' => $uploadResults,
                    'user_ip' => $request->ip(),
                ]);
            }

            return response()->json([
                'success' => true,
                'upload_id' => $uploadId,
                'files' => $uploadResults,
                'message' => 'Files uploaded successfully',
            ]);
        } catch (\Exception $e) {
            Log::error('Blog upload failed', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);

            return response()->json([
                'success' => false,
                'error' => 'Upload failed: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Get upload status
     */
    public function getUploadStatus(string $uploadId): JsonResponse
    {
        try {
            $config = config('blog_uploads');
            $uploadDisk = Storage::disk($config['storage']['upload_disk']);
            $tempDisk = Storage::disk($config['storage']['temp_disk']);

            // Check if upload exists
            $uploadPath = $config['storage']['directories']['uploads'] . '/' . $uploadId;

            if (!$uploadDisk->exists($uploadPath)) {
                return response()->json([
                    'success' => false,
                    'error' => 'Upload not found',
                ], 404);
            }

            // Get upload files
            $files = $uploadDisk->files($uploadPath);
            $fileStatuses = [];

            foreach ($files as $file) {
                $fileName = basename($file);
                $fileStatuses[] = [
                    'filename' => $fileName,
                    'size' => $uploadDisk->size($file),
                    'uploaded_at' => Carbon::createFromTimestamp($uploadDisk->lastModified($file))->toISOString(),
                    'status' => 'uploaded',
                ];
            }

            return response()->json([
                'success' => true,
                'upload_id' => $uploadId,
                'files' => $fileStatuses,
                'total_files' => count($fileStatuses),
            ]);
        } catch (\Exception $e) {
            Log::error('Failed to get upload status', [
                'upload_id' => $uploadId,
                'error' => $e->getMessage(),
            ]);

            return response()->json([
                'success' => false,
                'error' => 'Failed to get upload status',
            ], 500);
        }
    }

    /**
     * Delete uploaded files
     */
    public function deleteUpload(string $uploadId): JsonResponse
    {
        try {
            $config = config('blog_uploads');
            $uploadDisk = Storage::disk($config['storage']['upload_disk']);

            $uploadPath = $config['storage']['directories']['uploads'] . '/' . $uploadId;

            if (!$uploadDisk->exists($uploadPath)) {
                return response()->json([
                    'success' => false,
                    'error' => 'Upload not found',
                ], 404);
            }

            // Delete upload directory and all files
            $uploadDisk->deleteDirectory($uploadPath);

            Log::info('Upload deleted', [
                'upload_id' => $uploadId,
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Upload deleted successfully',
            ]);
        } catch (\Exception $e) {
            Log::error('Failed to delete upload', [
                'upload_id' => $uploadId,
                'error' => $e->getMessage(),
            ]);

            return response()->json([
                'success' => false,
                'error' => 'Failed to delete upload',
            ], 500);
        }
    }

    /**
     * Get upload progress (for large files)
     */
    public function getUploadProgress(string $uploadId): JsonResponse
    {
        try {
            $config = config('blog_uploads');
            $tempDisk = Storage::disk($config['storage']['temp_disk']);

            // Check for progress file
            $progressFile = 'progress/' . $uploadId . '.json';

            if (!$tempDisk->exists($progressFile)) {
                return response()->json([
                    'success' => false,
                    'error' => 'Progress not found',
                ], 404);
            }

            $progressData = json_decode($tempDisk->get($progressFile), true);

            return response()->json([
                'success' => true,
                'progress' => $progressData,
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'error' => 'Failed to get progress',
            ], 500);
        }
    }

    /**
     * Process individual file upload
     */
    private function processFileUpload($file, string $uploadId, array $config): array
    {
        $originalName = $file->getClientOriginalName();
        $sanitizedName = $this->sanitizeFileName($originalName);
        $uploadPath = $config['storage']['directories']['uploads'] . '/' . $uploadId;

        // Store the file
        $filePath = $file->storeAs(
            $uploadPath,
            $sanitizedName,
            $config['storage']['upload_disk']
        );

        // Create backup if enabled
        if ($config['processing']['backup_originals']) {
            $backupPath = $config['storage']['directories']['backups'] . '/' . $uploadId;
            $file->storeAs(
                $backupPath,
                $sanitizedName,
                $config['storage']['backup_disk']
            );
        }

        return [
            'original_name' => $originalName,
            'stored_name' => $sanitizedName,
            'file_path' => $filePath,
            'file_size' => $file->getSize(),
            'mime_type' => $file->getMimeType(),
            'uploaded_at' => Carbon::now()->toISOString(),
            'status' => 'uploaded',
        ];
    }

    /**
     * Sanitize file name
     */
    private function sanitizeFileName(string $fileName): string
    {
        // Remove path traversal attempts
        $fileName = basename($fileName);

        // Remove or replace unsafe characters
        $fileName = preg_replace('/[^a-zA-Z0-9._-]/', '_', $fileName);

        // Ensure it ends with .js
        if (!str_ends_with(strtolower($fileName), '.js')) {
            $fileName .= '.js';
        }

        return $fileName;
    }
}
