<?php

namespace App\Http\Controllers\Client;

use App\Http\Controllers\Controller;
use App\Models\Order;
use App\Services\OrderStatusService;
use App\Notifications\OrderApprovedNotification;
use App\Notifications\RevisionRequestedNotification;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class OrderApprovalController extends Controller
{
    protected OrderStatusService $orderStatusService;

    public function __construct(OrderStatusService $orderStatusService)
    {
        $this->orderStatusService = $orderStatusService;
        $this->middleware('auth');
        $this->middleware('client');
    }

    /**
     * Show the order approval interface.
     */
    public function showApprovalInterface(Order $order)
    {
        // Verify the client owns this order
        if ($order->user_id !== auth()->id()) {
            abort(403, 'Unauthorized');
        }

        // Verify the order is under review
        if ($order->order_status !== 'under_review') {
            abort(400, 'Order is not ready for approval');
        }

        return inertia('Client/OrderApproval', [
            'order' => $order->load(['writer', 'orderFiles' => function ($query) {
                $query->where('upload_type', 'admin_delivery');
            }])
        ]);
    }

    /**
     * Approve the order.
     */
    public function approveOrder(Request $request, Order $order)
    {
        // Verify the client owns this order
        if ($order->user_id !== auth()->id()) {
            return response()->json(['message' => 'Unauthorized'], 403);
        }

        // Verify the order is under review
        if ($order->order_status !== 'under_review') {
            return response()->json(['message' => 'Order is not ready for approval'], 400);
        }

        try {
            // Use OrderStatusService to handle status transition
            $this->orderStatusService->handleClientApproval($order);

            // Send notification to writer
            $this->sendApprovalNotification($order);

            return response()->json([
                'message' => 'Order approved successfully!',
                'order' => $order->fresh()->load(['writer'])
            ]);

        } catch (\Exception $e) {
            Log::error('Failed to approve order: ' . $e->getMessage(), [
                'order_id' => $order->id,
                'client_id' => auth()->id(),
                'exception' => $e
            ]);

            return response()->json([
                'message' => 'Failed to approve order: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Request revision for the order.
     */
    public function requestRevision(Request $request, Order $order)
    {
        // Verify the client owns this order
        if ($order->user_id !== auth()->id()) {
            return response()->json(['message' => 'Unauthorized'], 403);
        }

        // Verify the order is under review
        if ($order->order_status !== 'under_review') {
            return response()->json(['message' => 'Order is not ready for revision request'], 400);
        }

        $request->validate([
            'revision_notes' => 'required|string|max:1000'
        ]);

        try {
            $revisionNotes = $request->input('revision_notes');

            // Use OrderStatusService to handle status transition
            $this->orderStatusService->handleClientRejection($order, $revisionNotes);

            // Send notification to writer
            $this->sendRevisionNotification($order, $revisionNotes);

            return response()->json([
                'message' => 'Revision requested successfully. Writer has been notified.',
                'order' => $order->fresh()->load(['writer'])
            ]);

        } catch (\Exception $e) {
            Log::error('Failed to request revision: ' . $e->getMessage(), [
                'order_id' => $order->id,
                'client_id' => auth()->id(),
                'exception' => $e
            ]);

            return response()->json([
                'message' => 'Failed to request revision: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get order status timeline.
     */
    public function getOrderTimeline(Order $order)
    {
        // Verify the client owns this order
        if ($order->user_id !== auth()->id()) {
            return response()->json(['message' => 'Unauthorized'], 403);
        }

        $statusHistory = $order->statusHistory()
            ->with('changedByUser')
            ->orderBy('created_at', 'asc')
            ->get();

        return response()->json([
            'timeline' => $statusHistory
        ]);
    }

    /**
     * Send approval notification to writer.
     */
    private function sendApprovalNotification(Order $order)
    {
        try {
            $writer = $order->writer;
            if ($writer) {
                $writer->notify(new \App\Notifications\OrderApprovedNotification($order));
            }
        } catch (\Exception $e) {
            Log::error('Error sending approval notification: ' . $e->getMessage(), [
                'order_id' => $order->id,
                'exception' => $e
            ]);
        }
    }

    /**
     * Send revision notification to writer.
     */
    private function sendRevisionNotification(Order $order, string $revisionNotes)
    {
        try {
            $writer = $order->writer;
            if ($writer) {
                $writer->notify(new \App\Notifications\RevisionRequestedNotification($order, $revisionNotes));
            }
        } catch (\Exception $e) {
            Log::error('Error sending revision notification: ' . $e->getMessage(), [
                'order_id' => $order->id,
                'exception' => $e
            ]);
        }
    }
}
