<?php

namespace App\Http\Controllers;

use App\Models\Bid;
use App\Models\User;
use App\Models\Order;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use App\Notifications\BidAcceptedNotification;
use App\Notifications\BidRejectedNotification;
use App\Notifications\BidAssignmentConfirmationNotification;
use App\Notifications\OrderAssignmentAdminNotification;
use App\Services\AdminNotificationService;
use App\Services\OrderStatusService;

class ClientBidController extends Controller
{
    protected OrderStatusService $orderStatusService;

    public function __construct(OrderStatusService $orderStatusService)
    {
        $this->orderStatusService = $orderStatusService;
    }

    /**
     * Display a listing of all bids across the client's orders.
     *
     * @param Request $request
     * @return \Inertia\Response
     */
    // public function index(Request $request)
    // {
    //     // Get authenticated user's order IDs for security (only show bids on their orders)
    //     $orderIds = Auth::user()->orders()->pluck('id');

    //     // Build base query with necessary relationships
    //     $query = Bid::whereIn('order_id', $orderIds)
    //         ->with([
    //             'writer' => function ($query) {
    //                 $query->select(['id', 'name', 'email']);
    //                 // Eager load writer profile to get nickname
    //                 $query->with(['writerProfile' => function ($query) {
    //                     $query->select(['id', 'user_id', 'nickname', 'experience_years']);
    //                 }]);
    //             },
    //             'order:id,order_number,title,deadline,net_amount'
    //         ]);

    //     // Apply filters using scopes
    //     if ($request->has('status') && $request->status !== 'all') {
    //         $query->where('status', $request->status);
    //     }

    //     if ($request->has('from_date') && $request->from_date) {
    //         $query->whereDate('created_at', '>=', $request->from_date);
    //     }

    //     if ($request->has('to_date') && $request->to_date) {
    //         $query->whereDate('created_at', '<=', $request->to_date);
    //     }

    //     if ($request->has('search') && $request->search) {
    //         $search = $request->search;
    //         $query->where(function ($q) use ($search) {
    //             $q->whereHas('order', function ($q) use ($search) {
    //                 $q->where('order_number', 'like', "%{$search}%")
    //                     ->orWhere('title', 'like', "%{$search}%");
    //             })
    //                 ->orWhereHas('writer', function ($q) use ($search) {
    //                     $q->where('name', 'like', "%{$search}%");
    //                 })
    //                 ->orWhereHas('writer.writerProfile', function ($q) use ($search) {
    //                     $q->where('nickname', 'like', "%{$search}%");
    //                 });
    //         });
    // //     }

    //     // Apply sorting
    //     $sortBy = $request->input('sort_by', 'created_at');
    //     $sortDirection = $request->input('sort_direction', 'desc');

    //     // Validate sort field to prevent SQL injection
    //     $allowedSortFields = ['created_at', 'bid_amount', 'completion_time', 'status'];
    //     if (!in_array($sortBy, $allowedSortFields)) {
    //         $sortBy = 'created_at';
    //     }

    //     $query->orderBy($sortBy, $sortDirection === 'asc' ? 'asc' : 'desc');

    //     // Get paginated results
    //     $bids = $query->paginate(10)->appends($request->query());

    //     // Transform the bids to include writer nickname and experience
    //     $bids->through(function ($bid) {
    //         // Get safe writer info
    //         $writerInfo = $bid->getSafeWriterInfo();

    //         // Add writer attributes
    //         $bid->writer_nickname = $writerInfo['nickname'];
    //         $bid->writer_experience = $writerInfo['experience_years'];

    //         return $bid;
    //     });

    //     // Get bid statistics for the client
    //     $stats = [
    //         'total' => Bid::whereIn('order_id', $orderIds)->count(),
    //         'pending' => Bid::whereIn('order_id', $orderIds)->where('status', 'pending')->count(),
    //         'accepted' => Bid::whereIn('order_id', $orderIds)->where('status', 'accepted')->count(),
    //         'rejected' => Bid::whereIn('order_id', $orderIds)->where('status', 'rejected')->count(),
    //         'average_amount' => round(Bid::whereIn('order_id', $orderIds)->avg('bid_amount') ?? 0, 2),
    //     ];

    //     return inertia('Client/Bids/Index', [
    //         'bids' => $bids,
    //         'filters' => $request->only([
    //             'status',
    //             'search',
    //             'from_date',
    //             'to_date',
    //             'sort_by',
    //             'sort_direction'
    //         ]),
    //         'stats' => $stats
    //     ]);
    // }

    /**
     * Display a listing of all bids across the client's orders.
     *
     * @param Request $request
     * @return \Inertia\Response
     */
    public function index(Request $request)
    {

        Log::info('Raw request parameters', [
            'all' => $request->all(),
            'query' => $request->query(),
            'status_param' => $request->query('status'),
            'status_input' => $request->input('status'),
        ]);
        // Debug what's being received
        Log::info('Bid filter request', [
            'status' => $request->input('status'),
            'from_date' => $request->input('from_date'),
            'to_date' => $request->input('to_date'),
            'search' => $request->input('search'),
        ]);

        // Get authenticated user's order IDs for security (only show bids on their orders)
        $orderIds = Auth::user()->orders()->pluck('id');

        // Build base query with necessary relationships
        $query = Bid::whereIn('order_id', $orderIds)
            ->with([
                'writer' => function ($query) {
                    $query->select(['id', 'name', 'email']);
                    // Eager load writer profile to get nickname
                    $query->with(['writerProfile' => function ($query) {
                        $query->select(['id', 'user_id', 'nickname', 'experience_years']);
                    }]);
                },
                'order:id,order_number,title,deadline,net_amount'
            ]);

        // Apply status filter manually for debugging
        $status = $request->input('status');
        Log::info('Applying status filter', ['status' => $status]);

        if ($status && $status !== 'all') {
            $query->where('status', $status);

            // Log the raw SQL for debugging
            Log::info('SQL after status filter', [
                'sql' => $query->toSql(),
                'bindings' => $query->getBindings()
            ]);
        }

        // Apply date filters
        if ($request->filled('from_date')) {
            $query->whereDate('created_at', '>=', $request->input('from_date'));
        }

        if ($request->filled('to_date')) {
            $query->whereDate('created_at', '<=', $request->input('to_date'));
        }

        // Apply search filter
        if ($request->filled('search')) {
            $search = $request->input('search');
            $query->where(function ($q) use ($search) {
                $q->whereHas('order', function ($q) use ($search) {
                    $q->where('order_number', 'like', "%{$search}%")
                        ->orWhere('title', 'like', "%{$search}%");
                })
                    ->orWhereHas('writer', function ($q) use ($search) {
                        $q->where('name', 'like', "%{$search}%");
                    });
            });
        }

        // Apply sorting
        $sortBy = $request->input('sort_by', 'created_at');
        $sortDirection = $request->input('sort_direction', 'desc');

        // Validate sort field to prevent SQL injection
        $allowedSortFields = ['created_at', 'bid_amount', 'completion_time', 'status'];
        if (!in_array($sortBy, $allowedSortFields)) {
            $sortBy = 'created_at';
        }

        $query->orderBy($sortBy, $sortDirection === 'asc' ? 'asc' : 'desc');

        // Get paginated results
        $bids = $query->paginate(10)->appends($request->query());

        // Log the ACTUAL results being sent to the frontend
        Log::info('Bids being sent to frontend', [
            'count' => $bids->count(),
            'statuses' => $bids->pluck('status')->toArray(),
            'ids' => $bids->pluck('id')->toArray()
        ]);

        // Transform the bids to include writer nickname and experience
        $bids->through(function ($bid) {
            // Debug the actual bid data to make sure it's not being changed
            Log::info("Processing bid #{$bid->id}", [
                'id' => $bid->id,
                'status_before' => $bid->status
            ]);

            // Create a safe writer info array
            $writerInfo = [
                'nickname' => 'Anonymous Writer',
                'experience_years' => null
            ];

            if ($bid->writer && $bid->writer->writerProfile) {
                $writerInfo = [
                    'nickname' => $bid->writer->writerProfile->nickname ?? "Writer #" . $bid->writer_id,
                    'experience_years' => $bid->writer->writerProfile->experience_years
                ];
            }

            // Add writer attributes
            $bid->writer_nickname = $writerInfo['nickname'];
            $bid->writer_experience = $writerInfo['experience_years'];

            // Log after transformation to see if status changed
            Log::info("After transformation bid #{$bid->id}", [
                'status_after' => $bid->status
            ]);

            return $bid;
        });

        // Get bid statistics for the client
        $stats = [
            'total' => Bid::whereIn('order_id', $orderIds)->count(),
            'pending' => Bid::whereIn('order_id', $orderIds)->where('status', 'pending')->count(),
            'accepted' => Bid::whereIn('order_id', $orderIds)->where('status', 'accepted')->count(),
            'rejected' => Bid::whereIn('order_id', $orderIds)->where('status', 'rejected')->count(),
            'average_amount' => round(Bid::whereIn('order_id', $orderIds)->avg('bid_amount') ?? 0, 2),
        ];

        Log::info('Bid statistics', $stats);

        return inertia('Client/Bids/Index', [
            'bids' => $bids,
            'filters' => $request->only([
                'status',
                'search',
                'from_date',
                'to_date',
                'sort_by',
                'sort_direction'
            ]),
            'stats' => $stats
        ]);
    }




    /**
     * Accept a bid and assign the order to the writer
     */
    public function acceptBid(Request $request, Order $order, Bid $bid)
    {
        // Verify the user owns this order
        if ($order->user_id !== auth()->id()) {
            return response()->json(['message' => 'Unauthorized'], 403);
        }

        // Verify the bid belongs to the order
        if ($bid->order_id !== $order->id) {
            return response()->json(['message' => 'Bid does not belong to this order'], 400);
        }

        // Verify the order is still in bidding status
        if ($order->order_status !== 'bidding') {
            return response()->json(['message' => 'This order is no longer accepting bids'], 400);
        }

        // Begin transaction to ensure all operations succeed or fail together
        DB::beginTransaction();

        try {
            // Update the accepted bid status
            $bid->status = 'accepted';
            $bid->save();

            // Use OrderStatusService to handle status transition and writer assignment
            $this->orderStatusService->handleBidAccepted($order, $bid->writer_id, $bid->id);

            // Get rejected bids before updating them
            $rejectedBids = Bid::where('order_id', $order->id)
                ->where('id', '!=', $bid->id)
                ->with('writer')
                ->get();

            // Reject all other bids for this order
            Bid::where('order_id', $order->id)
                ->where('id', '!=', $bid->id)
                ->update(['status' => 'rejected']);

            DB::commit();

            // After successful database updates, send notifications to all parties
            $this->sendNotifications($order, $bid, $rejectedBids);

            return response()->json([
                'message' => 'Bid accepted successfully! The writer has been notified.',
                'order' => $order
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Failed to accept bid: ' . $e->getMessage(), [
                'order_id' => $order->id,
                'bid_id' => $bid->id,
                'exception' => $e
            ]);
            return response()->json(['message' => 'Failed to accept bid: ' . $e->getMessage()], 500);
        }
    }

    /**
     * Send notifications to all relevant parties after a bid is accepted.
     *
     * @param Order $order The order that had a bid accepted
     * @param Bid $bid The bid that was accepted
     * @param \Illuminate\Database\Eloquent\Collection $rejectedBids Collection of bids that were rejected
     * @return void
     */
    private function sendNotifications(Order $order, Bid $bid, $rejectedBids)
    {
        try {
            // 1. Notify the client who accepted the bid
            $client = $order->user;
            if ($client) {
                $client->notify(new BidAssignmentConfirmationNotification($order, $bid));
            }

            // 2. Notify the accepted writer
            $writer = $bid->writer;
            if ($writer) {
                $writer->notify(new BidAcceptedNotification($order, $bid));
            }

            // 3. Notify all admins about the assignment
            AdminNotificationService::notifyAllAdmins(
                new OrderAssignmentAdminNotification($order, $bid)
            );

            // 4. Notify all rejected writers
            foreach ($rejectedBids as $rejectedBid) {
                if ($rejectedBid->writer) {
                    $rejectedBid->writer->notify(new BidRejectedNotification($order));
                }
            }
        } catch (\Exception $e) {
            // Log notification failures but don't interrupt the process
            Log::error('Error sending bid acceptance notifications: ' . $e->getMessage(), [
                'order_id' => $order->id,
                'bid_id' => $bid->id,
                'exception' => $e
            ]);
        }
    }
}
