<?php

namespace App\Http\Controllers;

use App\Models\ClientPreference;
use App\Models\NotificationPreference;
use App\Models\AcademicLevel;
use App\Models\PaperType;
use App\Models\Subject;
use App\Models\Language;
use App\Models\Country;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Inertia\Inertia;
use Inertia\Response;

class ClientSettingsController extends Controller
{
    /**
     * Display the settings page.
     */
    public function index(): Response
    {
        $user = Auth::user();
        
        // Get or create preferences
        $clientPreferences = ClientPreference::getOrCreateForUser($user->id);
        $notificationPreferences = NotificationPreference::getOrCreateForUser($user->id);
        
        // Load relationships
        $clientPreferences->load([
            'defaultAcademicLevel',
            'defaultPaperType',
            'defaultSubject',
            'defaultLanguage'
        ]);
        
        $user->load(['country', 'billingCountry']);
        
        return Inertia::render('Settings/Index', [
            'clientPreferences' => $clientPreferences,
            'notificationPreferences' => $notificationPreferences,
            'user' => [
                'id' => $user->id,
                'name' => $user->name,
                'email' => $user->email,
                'phone' => $user->phone,
                'country_id' => $user->country_id,
                'country' => $user->country,
                'billing_street_address' => $user->billing_street_address,
                'billing_city' => $user->billing_city,
                'billing_state_province' => $user->billing_state_province,
                'billing_postal_code' => $user->billing_postal_code,
                'billing_country_id' => $user->billing_country_id,
                'billing_country' => $user->billingCountry,
                'tax_id' => $user->tax_id,
                'company_name' => $user->company_name,
            ],
            'academicLevels' => AcademicLevel::all(),
            'paperTypes' => PaperType::all(),
            'subjects' => Subject::all(),
            'languages' => Language::all(),
            'countries' => Country::all(),
        ]);
    }

    /**
     * Update notification preferences.
     */
    public function updateNotifications(Request $request)
    {
        $user = Auth::user();
        
        $preferences = NotificationPreference::getOrCreateForUser($user->id);
        
        $validated = $request->validate([
            // Order notifications
            'order_status_updates' => 'sometimes|boolean',
            'order_status_updates_email' => 'sometimes|boolean',
            'order_status_updates_in_app' => 'sometimes|boolean',
            'writer_assignment' => 'sometimes|boolean',
            'writer_assignment_email' => 'sometimes|boolean',
            'writer_assignment_in_app' => 'sometimes|boolean',
            'new_bids' => 'sometimes|boolean',
            'new_bids_email' => 'sometimes|boolean',
            'new_bids_in_app' => 'sometimes|boolean',
            'deadline_reminders' => 'sometimes|boolean',
            'deadline_reminders_email' => 'sometimes|boolean',
            'deadline_reminders_in_app' => 'sometimes|boolean',
            'order_completion' => 'sometimes|boolean',
            'order_completion_email' => 'sometimes|boolean',
            'order_completion_in_app' => 'sometimes|boolean',
            'revision_requests' => 'sometimes|boolean',
            'revision_requests_email' => 'sometimes|boolean',
            'revision_requests_in_app' => 'sometimes|boolean',
            // Payment notifications
            'payment_confirmations' => 'sometimes|boolean',
            'payment_confirmations_email' => 'sometimes|boolean',
            'payment_confirmations_in_app' => 'sometimes|boolean',
            'payment_failures' => 'sometimes|boolean',
            'payment_failures_email' => 'sometimes|boolean',
            'payment_failures_in_app' => 'sometimes|boolean',
            'refunds' => 'sometimes|boolean',
            'refunds_email' => 'sometimes|boolean',
            'refunds_in_app' => 'sometimes|boolean',
            'invoice_ready' => 'sometimes|boolean',
            'invoice_ready_email' => 'sometimes|boolean',
            'invoice_ready_in_app' => 'sometimes|boolean',
            // Communication notifications
            'new_messages' => 'sometimes|boolean',
            'new_messages_email' => 'sometimes|boolean',
            'new_messages_in_app' => 'sometimes|boolean',
            'writer_questions' => 'sometimes|boolean',
            'writer_questions_email' => 'sometimes|boolean',
            'writer_questions_in_app' => 'sometimes|boolean',
            // System notifications
            'system_announcements' => 'sometimes|boolean',
            'system_announcements_email' => 'sometimes|boolean',
            'system_announcements_in_app' => 'sometimes|boolean',
            'promotional_emails' => 'sometimes|boolean',
            'coupon_alerts' => 'sometimes|boolean',
            'coupon_alerts_email' => 'sometimes|boolean',
            'coupon_alerts_in_app' => 'sometimes|boolean',
        ]);
        
        $preferences->update($validated);
        
        return back()->with('success', 'Notification preferences updated successfully.');
    }

    /**
     * Update client preferences.
     */
    public function updatePreferences(Request $request)
    {
        $user = Auth::user();
        
        $preferences = ClientPreference::getOrCreateForUser($user->id);
        
        $validated = $request->validate([
            'default_academic_level_id' => 'nullable|exists:academic_levels,id',
            'default_paper_type_id' => 'nullable|exists:paper_types,id',
            'default_subject_id' => 'nullable|exists:subjects,id',
            'default_spacing' => 'nullable|in:single,double,1.5',
            'default_citation_style' => 'nullable|string|max:100',
            'default_language_id' => 'nullable|exists:languages,id',
            'default_currency' => 'nullable|string|max:10',
            'auto_assign_writer' => 'sometimes|boolean',
            'preferred_writer_ids' => 'nullable|array',
            'preferred_writer_ids.*' => 'integer|exists:users,id',
            'blacklisted_writer_ids' => 'nullable|array',
            'blacklisted_writer_ids.*' => 'integer|exists:users,id',
            'default_deadline_buffer_hours' => 'nullable|integer|min:0|max:168',
            'allow_writer_contact_after_completion' => 'sometimes|boolean',
            'show_order_history_to_writers' => 'sometimes|boolean',
            'profile_visible_to_writers' => 'sometimes|boolean',
            'preferred_communication_method' => 'nullable|in:in_app,email,both',
            'timezone' => 'nullable|string|max:100',
            'date_format' => 'nullable|string|max:50',
            'time_format' => 'nullable|string|max:20',
            'theme' => 'nullable|in:light,dark,auto',
            'auto_send_invoices' => 'sometimes|boolean',
            'invoice_format' => 'nullable|in:pdf,email,both',
        ]);
        
        $preferences->update($validated);
        
        return back()->with('success', 'Preferences updated successfully.');
    }

    /**
     * Update billing address.
     */
    public function updateBilling(Request $request)
    {
        $user = Auth::user();
        
        $validated = $request->validate([
            'billing_street_address' => 'nullable|string|max:255',
            'billing_city' => 'nullable|string|max:100',
            'billing_state_province' => 'nullable|string|max:100',
            'billing_postal_code' => 'nullable|string|max:20',
            'billing_country_id' => 'nullable|exists:countries,id',
            'tax_id' => 'nullable|string|max:50',
            'company_name' => 'nullable|string|max:255',
        ]);
        
        $user->update($validated);
        
        return back()->with('success', 'Billing address updated successfully.');
    }
}