<?php

namespace App\Http\Controllers;

use App\Models\Currency;
use Illuminate\Http\Request;
use Inertia\Inertia;

class CurrencyController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $currencies = Currency::orderBy('name')->get();

        return Inertia::render('Admin/Currencies/Index', [
            'currencies' => $currencies,
        ]);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return Inertia::render('Admin/Currencies/Create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'currency_code' => 'required|string|max:10|unique:currencies,currency_code',
            'value' => 'required|numeric|min:0',
            'is_active' => 'boolean',
        ]);

        Currency::create($validated);

        return redirect()
            ->route('currencies.index')
            ->with('message', 'Currency created successfully!');
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Currency $currency)
    {
        return Inertia::render('Admin/Currencies/Edit', [
            'currency' => $currency,
        ]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Currency $currency)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'currency_code' => 'required|string|max:10|unique:currencies,currency_code,' . $currency->id,
            'value' => 'required|numeric|min:0',
            'is_active' => 'boolean',
        ]);

        $currency->update($validated);

        return redirect()
            ->route('currencies.index')
            ->with('message', 'Currency updated successfully!');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Currency $currency)
    {
        // Prevent deleting USD as it's the base currency
        if ($currency->currency_code === 'USD') {
            return redirect()
                ->route('currencies.index')
                ->with('error', 'Cannot delete the base currency (USD).');
        }

        $currency->delete();

        return redirect()
            ->route('currencies.index')
            ->with('message', 'Currency deleted successfully!');
    }

    /**
     * Toggle the is_active status of a currency.
     */
    public function toggleStatus(Currency $currency)
    {
        // Prevent deactivating USD as it's the base currency
        if ($currency->currency_code === 'USD' && $currency->is_active) {
            return redirect()
                ->route('currencies.index')
                ->with('error', 'Cannot deactivate the base currency (USD).');
        }

        $currency->update([
            'is_active' => !$currency->is_active,
        ]);

        $status = $currency->is_active ? 'activated' : 'deactivated';

        return redirect()
            ->route('currencies.index')
            ->with('message', "Currency {$status} successfully!");
    }
}
