<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Inertia\Inertia;
use Inertia\Response;
use App\Services\PageService;
use App\Services\SEOService;
use App\Models\Page;

class DynamicPageController extends Controller
{
    protected PageService $pageService;
    protected SEOService $seoService;

    public function __construct(PageService $pageService, SEOService $seoService)
    {
        $this->pageService = $pageService;
        $this->seoService = $seoService;
    }

    /**
     * Handle dynamic page rendering based on slug
     */
    public function show(Request $request, string $slug): Response
    {
        // Get page from CMS
        $page = $this->pageService->getPageWithContent($slug);

        if (!$page) {
            abort(404, 'Page not found');
        }

        // Check if page is published
        if ($page->status !== 'published') {
            abort(404, 'Page not found');
        }

    // Get SEO data (may be null) and prepare safe array
    $seoData = $page->seo ?? null;
    $seoArray = $seoData?->toArray() ?? [];

    // Generate dynamic SEO metadata
    $openGraphData = $this->seoService->generateOpenGraphData($page->toArray(), $seoArray);
    $twitterData = $this->seoService->generateTwitterCardData($page->toArray(), $seoArray);
        
        // Get media files
        $mediaFiles = $page->media;
        
        // Process content blocks
        $contentBlocks = $this->processContentBlocks($page->contentBlocks);
        
        // Determine the appropriate component based on page type
        $component = $this->getComponentForPageType($page->page_type, $page->template);
        
        // Build page data (null-safe)
        $pageData = [
            'title' => $seoData?->meta_title ?? $page->title,
            'description' => $seoData?->meta_description ?? '',
            'keywords' => $seoData?->meta_keywords ?? '',
            'canonicalUrl' => $seoData?->canonical_url ?? url('/' . $page->slug),
            'ogImage' => $seoData?->og_image ?? '/images/default-og-image.jpg',
            'ogImageAlt' => $seoData?->og_image_alt ?? $page->title,
            'ogUrl' => $seoData?->og_url ?? url('/' . $page->slug),
            'ogType' => $seoData?->og_type ?? 'website',
            'ogSiteName' => $seoData?->og_site_name ?? 'AcademicScribe',
            'twitterCard' => $seoData?->twitter_card ?? 'summary_large_image',
            'twitterSite' => $seoData?->twitter_site ?? '@AcademicScribe1',
            'pageType' => $page->page_type,
            'structuredData' => $seoData?->structured_data ?? null,
        ];

        return Inertia::render($component, [
            'page' => $page,
            'contentBlocks' => $contentBlocks,
            'mediaFiles' => $mediaFiles,
            'pageData' => $pageData,
            'openGraph' => $openGraphData,
            'twitterCard' => $twitterData,
            'seoData' => [
                'canonicalUrl' => $pageData['canonicalUrl'],
                'ogUrl' => $pageData['ogUrl'],
                'heroImageUrl' => $pageData['ogImage'],
            ],
            'isCMS' => true,
        ]);
    }

    /**
     * Handle nested page routing (e.g., /services/research-papers)
     */
    public function showNested(Request $request, string $parentSlug, string $childSlug): Response
    {
        // Try to find the page by combining slugs
        $fullSlug = $parentSlug . '/' . $childSlug;
        $page = $this->pageService->getPageWithContent($fullSlug);

        if (!$page) {
            // Try to find by child slug only
            $page = $this->pageService->getPageWithContent($childSlug);
        }

        if (!$page) {
            abort(404, 'Page not found');
        }

        return $this->show($request, $page->slug);
    }

    /**
     * Get component name based on page type and template
     */
    private function getComponentForPageType(string $pageType, ?string $template): string
    {
        // If template is specified, use it
        if ($template && $template !== 'default') {
            return $this->getTemplateComponent($template);
        }

        // Otherwise, use page type
        return $this->getPageTypeComponent($pageType);
    }

    /**
     * Get component for specific template
     */
    private function getTemplateComponent(string $template): string
    {
        $templateComponents = [
            'service' => 'Services/DynamicService',
            'landing' => 'Landing/DynamicLanding',
            'blog' => 'Blog/DynamicBlog',
            'contact' => 'Contact/DynamicContact',
            'about' => 'About/DynamicAbout',
        ];

        return $templateComponents[$template] ?? 'Pages/DynamicPage';
    }

    /**
     * Get component for page type
     */
    private function getPageTypeComponent(string $pageType): string
    {
        $pageTypeComponents = [
            'homepage' => 'Home',
            'service' => 'Services/DynamicService',
            'blog' => 'Blog/DynamicBlog',
            'contact' => 'Contact/DynamicContact',
            'about' => 'About/DynamicAbout',
            'landing' => 'Landing/DynamicLanding',
        ];

        return $pageTypeComponents[$pageType] ?? 'Pages/DynamicPage';
    }

    /**
     * Process content blocks for frontend rendering
     */
    private function processContentBlocks($contentBlocks)
    {
        $processed = [];
        
        foreach ($contentBlocks as $block) {
            $processed[] = [
                'id' => $block->id,
                'type' => $block->block_type,
                'key' => $block->block_key,
                'content' => $block->content,
                'order' => $block->order,
                'is_active' => $block->is_active,
            ];
        }
        
        return $processed;
    }

    /**
     * Get page data for API endpoints
     */
    public function getPageData(string $slug)
    {
        $page = $this->pageService->getPageWithContent($slug);

        if (!$page || $page->status !== 'published') {
            return response()->json([
                'success' => false,
                'message' => 'Page not found'
            ], 404);
        }

    // Get SEO data (may be null) and prepare safe array
    $seoData = $page->seo ?? null;
    $seoArray = $seoData?->toArray() ?? [];
    $openGraphData = $this->seoService->generateOpenGraphData($page->toArray(), $seoArray);
    $twitterData = $this->seoService->generateTwitterCardData($page->toArray(), $seoArray);

        return response()->json([
            'success' => true,
            'page' => $page,
            'seo' => $seoData,
            'openGraph' => $openGraphData,
            'twitterCard' => $twitterData,
            'contentBlocks' => $this->processContentBlocks($page->contentBlocks),
            'mediaFiles' => $page->media,
        ]);
    }

    /**
     * Get all published pages for sitemap
     */
    public function getSitemapData()
    {
        $pages = Page::where('status', 'published')
            ->with(['seo'])
            ->select('slug', 'title', 'updated_at')
            ->orderBy('seo_priority', 'desc')
            ->orderBy('updated_at', 'desc')
            ->get();

        return response()->json([
            'success' => true,
            'pages' => $pages,
        ]);
    }
}