<?php

namespace App\Http\Controllers;

use Inertia\Inertia;
use Inertia\Response;
use App\Services\PageService;
use App\Services\SEOService;
use App\Models\Page;
use Illuminate\Http\Request;

class HomeController extends Controller
{
    protected PageService $pageService;
    protected SEOService $seoService;

    public function __construct(PageService $pageService, SEOService $seoService)
    {
        $this->pageService = $pageService;
        $this->seoService = $seoService;
    }

    /**
     * Display the home page with CMS data - no fallback to static data
     *
     * @return Response
     */
    public function index(): Response
    {
        // Get homepage from CMS - no fallback to static data
        $cmsPage = $this->pageService->getPageWithContent('homepage');
        
        if (!$cmsPage) {
            abort(404, 'Homepage not found');
        }
        
        // Use CMS data
        return $this->renderCMSPage($cmsPage);
    }

    /**
     * Render page using CMS data
     */
    private function renderCMSPage(Page $page): Response
    {
        // Get SEO data
        $seoData = $page->seo;
        
        // Validate that SEO data exists
        if (!$seoData) {
            abort(500, 'SEO data is missing for homepage. Please add SEO data via admin panel.');
        }
        
        // Process content blocks
        $contentBlocks = $this->processContentBlocks($page->contentBlocks);
        
        // Build page data - ALL from database, NO fallbacks
        // All SEO data is consolidated here (no duplicates)
        $pageData = [
            'title' => $seoData->meta_title,
            'description' => $seoData->meta_description,
            'keywords' => $seoData->meta_keywords,
            'canonicalUrl' => $seoData->canonical_url,
            'ogImage' => $seoData->og_image,
            'ogImageAlt' => $seoData->og_image_alt,
            'ogUrl' => $seoData->og_url,
            'ogType' => $seoData->og_type,
            'ogSiteName' => $seoData->og_site_name,
            'ogLocale' => $seoData->og_locale,
            'twitterCard' => $seoData->twitter_card,
            'twitterSite' => $seoData->twitter_site,
            'twitterCreator' => $seoData->twitter_creator,
            'twitterTitle' => $seoData->twitter_title,
            'twitterDescription' => $seoData->twitter_description,
            'twitterImage' => $seoData->twitter_image,
            'twitterImageAlt' => $seoData->twitter_image_alt,
            'pageType' => $page->page_type,
            'contentGroup' => 'main_landing',
            'structuredData' => $seoData->structured_data,
        ];

        return Inertia::render('Home', [
            'contentBlocks' => $contentBlocks,
            'pageData' => $pageData,
            'isCMS' => true,
        ]);
    }

    /**
     * Process content blocks for frontend rendering
     */
    private function processContentBlocks($contentBlocks)
    {
        $processed = [];
        
        foreach ($contentBlocks as $block) {
            $processed[] = [
                'id' => $block->id,
                'type' => $block->block_type,
                'key' => $block->block_key,
                'content' => $block->content,
                'order' => $block->order,
                'is_active' => $block->is_active,
            ];
        }
        
        return $processed;
    }
}
