<?php

namespace App\Http\Controllers;

use Exception;
use Inertia\Inertia;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Auth;
use App\Services\NotificationService;
use Illuminate\Support\Facades\Cache;

class NotificationController extends Controller
{
    /**
     * The notification service instance.
     *
     * @var NotificationService
     */
    protected $notificationService;

    /**
     * Create a new controller instance.
     *
     * @param NotificationService $notificationService
     * @return void
     */
    public function __construct(NotificationService $notificationService)
    {
        $this->notificationService = $notificationService;
        $this->middleware('auth');
    }

    /**
     * Display a paginated list of the user's notifications.
     *
     * @param Request $request
     * @return \Inertia\Response
     */
    public function index(Request $request)
    {
        $filters = $request->validate([
            'page' => 'nullable|integer|min:1',
            'per_page' => 'nullable|integer|min:5|max:100',
            'type' => 'nullable|string|max:100',
            'category' => 'nullable|string|max:100',
            'read' => 'nullable|boolean',
            'search' => 'nullable|string|max:100',
            'from_date' => 'nullable|date',
            'to_date' => 'nullable|date|after_or_equal:from_date',
            'sort_field' => 'nullable|string|in:created_at,read_at',
            'sort_direction' => 'nullable|string|in:asc,desc',
        ]);

        $notifications = $this->notificationService->getForUser(Auth::user(), $filters);
        $unreadCount = $this->notificationService->getUnreadCount(Auth::user());

        return Inertia::render('Notifications/Index', [
            'notifications' => $notifications,
            'unreadCount' => $unreadCount,
            'filters' => $filters,
        ]);
    }

    /**
     * Show a single notification.
     *
     * @param string $id
     * @return \Inertia\Response|\Illuminate\Http\RedirectResponse
     */
    public function show(string $id)
    {
        try {
            // Find the notification
            $notification = Auth::user()->notifications()->findOrFail($id);

            // Mark as read if not already read
            if (!$notification->read_at) {
                Log::info('Marking notification as read on view', [
                    'notification_id' => $id,
                    'user_id' => Auth::id()
                ]);

                $notification->markAsRead();

                // Update unread count for any components that might need it
                $this->notificationService->invalidateUserCache(Auth::id());
            }

            // Parse notification data if needed
            $data = $notification->data;
            if (is_string($data)) {
                $data = json_decode($data, true);
            }

            // Add any additional information needed for the show page
            $notification->parsed_data = $data;

            // Return the Inertia view with the notification
            return Inertia::render('Notifications/Show', [
                'notification' => $notification,
                'actions' => $data['actions'] ?? [],
                'relatedData' => $this->getRelatedData($notification),
            ]);
        } catch (Exception $e) {
            Log::error('Error showing notification', [
                'notification_id' => $id,
                'user_id' => Auth::id(),
                'exception' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return redirect()->route('notifications.index')
                ->with('error', 'Notification not found or could not be displayed.');
        }
    }

    private function getRelatedData($notification)
    {
        $data = $notification->data;
        if (is_string($data)) {
            $data = json_decode($data, true);
        }

        $relatedData = [];

        // Check for order-related data
        if (isset($data['order_id'])) {
            try {
                $order = \App\Models\Order::find($data['order_id']);
                if ($order) {
                    $relatedData['order'] = [
                        'id' => $order->id,
                        'order_number' => $order->order_number,
                        'title' => $order->title,
                        'status' => $order->order_status,
                        'payment_status' => $order->payment_status,
                        // Include route names instead of pre-constructed URLs
                        'routes' => [
                            'view' => 'orders.show',
                            'payment' => 'orders.payment',
                            'payment_history' => 'orders.payment-history',
                        ]
                    ];
                }
            } catch (\Exception $e) {
                Log::warning('Error loading related order data', [
                    'notification_id' => $notification->id,
                    'order_id' => $data['order_id'],
                    'error' => $e->getMessage()
                ]);
            }
        }

        // Check for payment-related data
        if (isset($data['payment_id'])) {
            try {
                $payment = \App\Models\Payment::find($data['payment_id']);
                if ($payment) {
                    $relatedData['payment'] = [
                        'id' => $payment->id,
                        'amount' => $payment->amount,
                        'status' => $payment->status,
                        'payment_method' => $payment->payment_method,
                        'transaction_id' => $payment->transaction_id,
                        'date' => $payment->created_at->format('Y-m-d H:i:s'),
                        'order_id' => $payment->order_id,
                        // Include route names
                        'routes' => [
                            'status' => 'payments.status',
                        ]
                    ];
                }
            } catch (\Exception $e) {
                Log::warning('Error loading related payment data', [
                    'notification_id' => $notification->id,
                    'payment_id' => $data['payment_id'],
                    'error' => $e->getMessage()
                ]);
            }
        }

        return $relatedData;
    }




    /**
     * Get unread notifications for the authenticated user.
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function unread(Request $request): JsonResponse
    {
        try {
            $filters = $request->validate([
                'page' => 'nullable|integer|min:1',
                'per_page' => 'nullable|integer|min:5|max:100',
                'type' => 'nullable|string|max:100',
                'category' => 'nullable|string|max:100',
                'search' => 'nullable|string|max:100',
                // Remove 'read' from validation here since we'll set it ourselves
            ]);

            // Explicitly set read to false (boolean) to avoid validation issues
            $filters['read'] = false;

            $notifications = $this->notificationService->getForUser(Auth::user(), $filters);

            return response()->json([
                'success' => true,
                'data' => $notifications,
            ]);
        } catch (Exception $e) {
            Log::error('Error fetching unread notifications', [
                'user_id' => Auth::id(),
                'exception' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve unread notifications: ' . $e->getMessage(),
            ], 500);
        }
    }


    /**
     * Mark a notification as read.
     *
     * @param Request $request
     * @param string $id
     * @return JsonResponse
     */
    public function markAsRead(Request $request, string $id): JsonResponse
    {
        try {
            Log::info('Attempting to mark notification as read', [
                'notification_id' => $id,
                'user_id' => Auth::id()
            ]);

            // First check if the notification exists and belongs to the user
            $notification = Auth::user()->notifications()->find($id);

            if (!$notification) {
                Log::warning('Notification not found or does not belong to user', [
                    'notification_id' => $id,
                    'user_id' => Auth::id()
                ]);

                return response()->json([
                    'success' => false,
                    'message' => 'Notification not found or does not belong to you',
                ], 404);
            }

            // If already read, return success without performing update
            if ($notification->read_at !== null) {
                Log::info('Notification already marked as read', [
                    'notification_id' => $id,
                    'user_id' => Auth::id(),
                    'read_at' => $notification->read_at
                ]);

                return response()->json([
                    'success' => true,
                    'message' => 'Notification already marked as read',
                    'notification' => $notification
                ]);
            }

            // Attempt to mark as read
            $success = $this->notificationService->markAsRead($id, Auth::user());

            // Fetch the updated notification to return
            $updatedNotification = Auth::user()->notifications()->find($id);

            Log::info('Mark as read operation completed', [
                'notification_id' => $id,
                'user_id' => Auth::id(),
                'success' => $success,
                'now_read_at' => $updatedNotification ? $updatedNotification->read_at : null
            ]);

            return response()->json([
                'success' => $success,
                'message' => $success ? 'Notification marked as read' : 'Failed to mark notification as read',
                'notification' => $updatedNotification
            ]);
        } catch (Exception $e) {
            Log::error('Error marking notification as read', [
                'user_id' => Auth::id(),
                'notification_id' => $id,
                'exception' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Failed to mark notification as read: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Mark a notification as unread.
     *
     * @param Request $request
     * @param string $id
     * @return JsonResponse
     */
    public function markAsUnread(Request $request, string $id): JsonResponse
    {
        try {
            Log::info('Attempting to mark notification as unread', [
                'notification_id' => $id,
                'user_id' => Auth::id()
            ]);

            // First check if the notification exists and belongs to the user
            $notification = Auth::user()->notifications()->find($id);

            if (!$notification) {
                Log::warning('Notification not found or does not belong to user', [
                    'notification_id' => $id,
                    'user_id' => Auth::id()
                ]);

                return response()->json([
                    'success' => false,
                    'message' => 'Notification not found or does not belong to you',
                ], 404);
            }

            // If already unread, return success without performing update
            if ($notification->read_at === null) {
                Log::info('Notification already marked as unread', [
                    'notification_id' => $id,
                    'user_id' => Auth::id()
                ]);

                return response()->json([
                    'success' => true,
                    'message' => 'Notification already marked as unread',
                    'notification' => $notification
                ]);
            }

            // Attempt to mark as unread
            $success = $this->notificationService->markAsUnread($id, Auth::user());

            // Fetch the updated notification to return
            $updatedNotification = Auth::user()->notifications()->find($id);

            Log::info('Mark as unread operation completed', [
                'notification_id' => $id,
                'user_id' => Auth::id(),
                'success' => $success,
                'now_read_at' => $updatedNotification ? $updatedNotification->read_at : null
            ]);

            return response()->json([
                'success' => $success,
                'message' => $success ? 'Notification marked as unread' : 'Failed to mark notification as unread',
                'notification' => $updatedNotification
            ]);
        } catch (Exception $e) {
            Log::error('Error marking notification as unread', [
                'user_id' => Auth::id(),
                'notification_id' => $id,
                'exception' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Failed to mark notification as unread: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Toggle notification read status.
     *
     * @param Request $request
     * @param string $id
     * @return JsonResponse
     */
    public function toggleReadStatus(Request $request, string $id): JsonResponse
    {
        try {
            Log::info('Attempting to toggle notification read status', [
                'notification_id' => $id,
                'user_id' => Auth::id()
            ]);

            // First check if the notification exists and belongs to the user
            $notification = Auth::user()->notifications()->find($id);

            if (!$notification) {
                Log::warning('Notification not found or does not belong to user', [
                    'notification_id' => $id,
                    'user_id' => Auth::id()
                ]);

                return response()->json([
                    'success' => false,
                    'message' => 'Notification not found or does not belong to you',
                ], 404);
            }

            // Attempt to toggle read status
            $success = $this->notificationService->toggleReadStatus($id, Auth::user());

            // Fetch the updated notification to return
            $updatedNotification = Auth::user()->notifications()->find($id);
            $newStatus = $updatedNotification && $updatedNotification->read_at ? 'read' : 'unread';

            Log::info('Toggle read status operation completed', [
                'notification_id' => $id,
                'user_id' => Auth::id(),
                'success' => $success,
                'new_status' => $newStatus
            ]);

            return response()->json([
                'success' => $success,
                'message' => $success ? "Notification marked as {$newStatus}" : 'Failed to toggle notification status',
                'notification' => $updatedNotification,
                'status' => $newStatus
            ]);
        } catch (Exception $e) {
            Log::error('Error toggling notification read status', [
                'user_id' => Auth::id(),
                'notification_id' => $id,
                'exception' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Failed to toggle notification status: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Mark multiple notifications as read.
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function markMultipleAsRead(Request $request): JsonResponse
    {
        $validated = $request->validate([
            'ids' => 'required|array',
            'ids.*' => 'required|string',
        ]);

        try {
            Log::info('Attempting to mark multiple notifications as read', [
                'notification_ids' => $validated['ids'],
                'user_id' => Auth::id()
            ]);

            // First verify these notifications belong to the user
            $userNotificationIds = Auth::user()->notifications()
                ->whereIn('id', $validated['ids'])
                ->pluck('id')
                ->toArray();

            $nonUserNotifications = array_diff($validated['ids'], $userNotificationIds);

            if (!empty($nonUserNotifications)) {
                Log::warning('Some notifications do not belong to the user', [
                    'user_id' => Auth::id(),
                    'non_user_notifications' => $nonUserNotifications
                ]);
            }

            // If no valid notifications found, return error
            if (empty($userNotificationIds)) {
                return response()->json([
                    'success' => false,
                    'message' => 'No valid notifications found to mark as read',
                ], 404);
            }

            // Mark valid notifications as read
            $success = $this->notificationService->markAsRead($userNotificationIds, Auth::user());

            // Get updated notifications
            $updatedNotifications = Auth::user()->notifications()
                ->whereIn('id', $userNotificationIds)
                ->get();

            // Get new unread count
            $unreadCount = $this->notificationService->getUnreadCount(Auth::user());

            Log::info('Mark multiple as read operation completed', [
                'user_id' => Auth::id(),
                'success' => $success,
                'notifications_updated' => count($userNotificationIds),
                'new_unread_count' => $unreadCount
            ]);

            return response()->json([
                'success' => $success,
                'message' => $success ? 'Notifications marked as read' : 'Failed to mark notifications as read',
                'notifications' => $updatedNotifications,
                'unread_count' => $unreadCount
            ]);
        } catch (Exception $e) {
            Log::error('Error marking multiple notifications as read', [
                'user_id' => Auth::id(),
                'notification_ids' => $validated['ids'] ?? [],
                'exception' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Failed to mark notifications as read: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Mark all notifications as read for the authenticated user.
     *
     * @return JsonResponse
     */
    public function markAllAsRead(): JsonResponse
    {
        try {
            $user = Auth::user();

            Log::info('Attempting to mark all notifications as read', [
                'user_id' => $user->id
            ]);

            // Get current unread count for logging
            $beforeCount = $this->notificationService->getUnreadCount($user);

            if ($beforeCount === 0) {
                Log::info('No unread notifications found', [
                    'user_id' => $user->id
                ]);

                return response()->json([
                    'success' => true,
                    'message' => 'No unread notifications to mark',
                    'unread_count' => 0
                ]);
            }

            // Attempt to mark all as read
            $success = $this->notificationService->markAllAsRead($user);

            // Get the new unread count
            $afterCount = $this->notificationService->getUnreadCount($user);

            Log::info('Mark all as read operation completed', [
                'user_id' => $user->id,
                'success' => $success,
                'before_count' => $beforeCount,
                'after_count' => $afterCount
            ]);

            return response()->json([
                'success' => $success,
                'message' => $success ? 'All notifications marked as read' : 'Failed to mark all notifications as read',
                'unread_count' => $afterCount,
                'marked_count' => $beforeCount - $afterCount
            ]);
        } catch (Exception $e) {
            Log::error('Error marking all notifications as read', [
                'user_id' => Auth::id(),
                'exception' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Failed to mark all notifications as read: ' . $e->getMessage(),
            ], 500);
        }
    }


    /**
     * Get the count of unread notifications for the authenticated user.
     *
     * @return JsonResponse
     */
    public function count(): JsonResponse
    {
        try {
            $user = Auth::user();
            $count = $this->notificationService->getUnreadCount($user);

            return response()->json([
                'success' => true,
                'count' => $count
            ]);
        } catch (Exception $e) {
            Log::error('Error fetching notification count', [
                'user_id' => Auth::id(),
                'exception' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            // Even on error, return a response with count 0 to prevent UI errors
            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve notification count',
                'count' => 0
            ]);
        }
    }


    /**
     * Delete a notification.
     *
     * @param Request $request
     * @param string $id
     * @return JsonResponse
     */
    public function delete(Request $request, string $id): JsonResponse
    {
        try {
            Log::info('Attempting to delete notification', [
                'notification_id' => $id,
                'user_id' => Auth::id()
            ]);

            // First verify the notification belongs to the user
            $notification = Auth::user()->notifications()->where('id', $id)->first();

            if (!$notification) {
                Log::warning('Notification not found or does not belong to user', [
                    'notification_id' => $id,
                    'user_id' => Auth::id()
                ]);

                return response()->json([
                    'success' => false,
                    'message' => 'Notification not found or does not belong to you',
                ], 404);
            }

            // Delete the notification
            $success = $notification->delete();

            // Invalidate cache
            $this->notificationService->invalidateUserCache(Auth::id());

            Log::info('Delete notification operation completed', [
                'notification_id' => $id,
                'user_id' => Auth::id(),
                'success' => $success
            ]);

            return response()->json([
                'success' => $success,
                'message' => $success ? 'Notification deleted' : 'Failed to delete notification',
            ]);
        } catch (Exception $e) {
            Log::error('Error deleting notification', [
                'user_id' => Auth::id(),
                'notification_id' => $id,
                'exception' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Failed to delete notification: ' . $e->getMessage(),
            ], 500);
        }
    }


    /**
     * Invalidate notification cache for a user.
     *
     * @param int $userId
     * @return void
     */
    public function invalidateUserCache(int $userId)
    {
        try {
            Log::info('Invalidating notification cache for user', [
                'user_id' => $userId
            ]);

            Cache::forget("user:{$userId}:unread_notifications_count");
            Cache::forget("user:{$userId}:recent_notifications");
        } catch (Exception $e) {
            Log::error('Error invalidating user notification cache', [
                'exception' => $e->getMessage(),
                'user_id' => $userId
            ]);
        }
    }

    public function getNotifications(Request $request): JsonResponse
    {
        try {
            $filters = $request->validate([
                'page' => 'nullable|integer|min:1',
                'per_page' => 'nullable|integer|min:5|max:100',
                'type' => 'nullable|string|max:100',
                'category' => 'nullable|string|max:100',
                // Change this to accept string values that will be converted to boolean
                'read' => 'nullable',
                'search' => 'nullable|string|max:100',
                'from_date' => 'nullable|date',
                'to_date' => 'nullable|date|after_or_equal:from_date',
                'sort_field' => 'nullable|string|in:created_at,read_at',
                'sort_direction' => 'nullable|string|in:asc,desc',
            ]);

            // Convert the read parameter to a proper boolean
            if (isset($filters['read']) && $filters['read'] !== null) {
                // Convert strings like "true", "1", "yes" to true and "false", "0", "no" to false
                $filters['read'] = filter_var($filters['read'], FILTER_VALIDATE_BOOLEAN);
            }

            $notifications = $this->notificationService->getForUser(Auth::user(), $filters);

            return response()->json([
                'success' => true,
                'data' => $notifications,
            ]);
        } catch (Exception $e) {
            Log::error('Error fetching notifications', [
                'user_id' => Auth::id(),
                'exception' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve notifications: ' . $e->getMessage(),
            ], 500);
        }
    }
}
