<?php

namespace App\Http\Controllers;

use App\Models\User;
use App\Models\Order;
use App\Models\Coupon;
use App\Models\Pricing;
use App\Models\Subject;
use App\Models\OrderFile;
use App\Models\PaperType;
use App\Models\AdminSetting;
use App\Models\SeasonalType;
use Illuminate\Http\Request;
use App\Models\AcademicLevel;
use App\Services\CouponService;
use App\Services\RefundService;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Auth;
use App\Notifications\AdminNotifiable;
use App\Services\AdminNotificationService;
use App\Notifications\NewOrderNotification;
use Illuminate\Support\Facades\Notification;

class OrdersController extends Controller
{
    /**
     * @var RefundService
     */
    protected $refundService;

    /**
     * @var CouponService
     */
    protected $couponService;

    /**
     * Create a new controller instance.
     *
     * @param RefundService $refundService
     * @param CouponService $couponService
     * @return void
     */
    public function __construct(RefundService $refundService, CouponService $couponService)
    {
        $this->refundService = $refundService;
        $this->couponService = $couponService;
    }

    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        // Clean and prepare search term
        $search = trim($request->input('search', ''));

        $query = Auth::user()->orders()->latest('dateposted');

        // Apply search filter if search term exists
        if ($search !== '') {
            $query->where(function ($q) use ($search) {
                $q->where('order_number', 'LIKE', "%{$search}%")
                    ->orWhere('title', 'LIKE', "%{$search}%")
                    ->orWhere('subject', 'LIKE', "%{$search}%")
                    ->orWhere('type_of_paper', 'LIKE', "%{$search}%")
                    ->orWhere('payment_status', 'LIKE', "%{$search}%");
            });
        }

        // Get paginated results
        $orders = $query->orderBy('dateposted', 'desc')
            ->with(['payments.refunds' => function ($query) {
                $query->where('status', 'completed');
            }, 'bids', 'writer'])
            ->paginate(5)
            ->withQueryString()
            ->through(function ($order) {
                // Add refund information to each order
                $refundInfo = $this->refundService->getOrderRefundInfo($order->id);

                return array_merge($order->toArray(), [
                    'is_refunded' => $order->isRefunded(),
                    'is_partially_refunded' => $order->isPartiallyRefunded(),
                    'refund_info' => $refundInfo,
                    'bids_count' => $order->bids()->count(),
                    'is_assigned' => !is_null($order->writer_id)
                ]);
            });

        return inertia('Order/MyOrders', [
            'orders' => $orders,
            'filters' => [
                'search' => $search
            ]
        ]);
    }

    public function create()
    {
        $academicLevels = AcademicLevel::all();
        $deadlines = Pricing::all();
        $paperTypes = PaperType::all();
        $subjects = Subject::all();
        // Only fetch active currencies
        $currencies  = \App\Models\Currency::where('is_active', 1)->get();

        $couponCode = \App\Models\Coupon::where('id', 1)->value('coupon_code');
        $costPerSlide = AdminSetting::where('price_per_slide', '>', 0)->value('price_per_slide');

        // Check if user is authenticated
        $isAuthenticated = Auth::check();
        $user = Auth::user();

        // Get available coupons for the current user (empty array for anonymous users)
        $suggestedCoupons = $this->getSuggestedCoupons();

        // For anonymous users, show generic discount information
        $discountInfo = $this->getDiscountInfo();

        // Check for restored order context from authentication
        $restoredOrderData = null;
        $restoredStep = 1;
        $welcomeMessage = null;
        $needsTokenRefresh = false;

        if (session('order_context_restored')) {
            $restoredOrderData = session('order_data', []);
            $restoredStep = session('order_step', 1);
            $welcomeMessage = session('welcome_message');
            
            // Also check for temporary context restore from login
            if (session('order_context_restore_temp')) {
                $tempContext = session('order_context_restore_temp');
                if (!empty($tempContext['data'])) {
                    $restoredOrderData = $tempContext['data'];
                    $restoredStep = $tempContext['step'] ?? 1;
                }
                session()->forget('order_context_restore_temp');
            }
            
            // Check if order context needs CSRF token refresh
            if (Auth::check()) {
                $needsTokenRefresh = \App\Services\OrderContextService::needsTokenRefresh();
                
                // Refresh token if needed
                if ($needsTokenRefresh) {
                    \App\Services\OrderContextService::refreshContextToken();
                }
            }
            
            // Clear the session data after retrieving it
            session()->forget(['order_context_restored', 'order_data', 'order_step', 'welcome_message']);
        }

        return inertia('Order/Create', [
            'academicLevels' => $academicLevels,
            'deadlines' => $deadlines,
            'paperTypes' => $paperTypes,
            'subjects' => $subjects,
            'currencies' => $currencies,
            'couponEnabled' => AdminSetting::where('enable_coupon', 1)->exists(),
            'couponCode' => $couponCode,
            'singleSlidePrice' => $costPerSlide,
            'suggestedCoupons' => $suggestedCoupons,
            // New data for supporting anonymous users
            'isAuthenticated' => $isAuthenticated,
            'user' => $user,
            'discountInfo' => $discountInfo,
            // Order context restoration data
            'restoredOrderData' => $restoredOrderData,
            'restoredStep' => $restoredStep,
            'welcomeMessage' => $welcomeMessage,
            'needsTokenRefresh' => $needsTokenRefresh
        ]);
    }

    /**
     * Calculate pricing for anonymous users (public endpoint)
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function calculateAnonymousPricing(Request $request)
    {
        try {
            $validated = $request->validate([
                'academic_level_id' => 'required|exists:academic_levels,id',
                'paper_type_id' => 'required|exists:paper_types,id',
                'deadline_id' => 'required|exists:pricing,id',
                'pages' => 'required|integer|min:1',
                'slides' => 'nullable|integer|min:0',
                'subject_id' => 'required|exists:subjects,id',
            ]);

            // Calculate base pricing (same logic as authenticated users but without user-specific discounts)
            $basePrice = $this->calculateBasePrice($validated);
            
            // Get discount information for display
            $discountInfo = $this->getDiscountInfo();
            
            // Calculate estimated savings range
            $minSavings = $basePrice * ($discountInfo['minimum_discount'] / 100);
            $maxSavings = $basePrice * ($discountInfo['newcomer_discount'] / 100);
            
            return response()->json([
                'success' => true,
                'base_price' => $basePrice,
                'currency' => 'USD', // or get from settings
                'estimated_savings' => [
                    'min' => $minSavings,
                    'max' => $maxSavings,
                    'min_final_price' => $basePrice - $maxSavings,
                    'max_final_price' => $basePrice - $minSavings,
                ],
                'discount_info' => $discountInfo
            ]);
            
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Unable to calculate pricing. Please try again.',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 400);
        }
    }

    /**
     * Store order context before authentication
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function storeOrderContext(Request $request)
    {
        $request->validate([
            'orderData' => 'required|array',
            'currentStep' => 'required|integer|min:1|max:5'
        ]);

        try {
            $success = \App\Services\OrderContextService::storeOrderContext(
                $request->orderData,
                $request->currentStep
            );

            if ($success) {
                return response()->json([
                    'success' => true,
                    'message' => 'Order context stored successfully'
                ]);
            } else {
                return response()->json([
                    'success' => false,
                    'message' => 'Failed to store order context - invalid data'
                ], 400);
            }
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to store order context',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 500);
        }
    }

    /**
     * Get discount information for anonymous users (public endpoint)
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function getDiscountInformation()
    {
        return response()->json([
            'success' => true,
            'discount_info' => $this->getDiscountInfo()
        ]);
    }

    /**
     * Calculate base price for given order parameters
     *
     * @param array $validated
     * @return float
     */
    protected function calculateBasePrice($validated)
    {
        // Get pricing information
        $academicLevel = AcademicLevel::find($validated['academic_level_id']);
        $paperType = PaperType::find($validated['paper_type_id']);
        $deadline = Pricing::find($validated['deadline_id']);
        
        if (!$academicLevel || !$paperType || !$deadline) {
            throw new \Exception('Invalid pricing parameters provided.');
        }

        // Get the raw values from the database (these are the multipliers)
        $aclevel = $academicLevel->value;
        $urgency = $deadline->value;
        $subject = isset($validated['subject_id']) ? Subject::find($validated['subject_id'])->value : 1;
        $paperTypeValue = $paperType->value;
        
        // Calculate spacing multiplier
        $spacing = 1; // Default to double spaced
        if (isset($validated['spacing'])) {
            if ($validated['spacing'] === 'single') {
                $spacing = 2; // Single spaced doubles the price
            } elseif ($validated['spacing'] === '1.5') {
                $spacing = 1.5; // 1.5 spacing is 1.5x the price
            }
        }
        
        // Calculate base cost per page (as if double spaced, spacing = 1)
        // Formula: (academic_level + subject + paper_type) + urgency
        $baseCostPerPage = ($aclevel + $subject + $paperTypeValue) + $urgency;
        
        // Multiply entire cost per page by spacing multiplier
        // This ensures single space (2) = 2x double space (1), and 1.5 space = 1.5x double space
        $costPerPage = $baseCostPerPage * $spacing;
        
        // Calculate base price: cost per page * pages
        $basePrice = $costPerPage * $validated['pages'];
        
        // Add slides cost if applicable
        if (isset($validated['slides']) && $validated['slides'] > 0) {
            $slidePrice = AdminSetting::where('price_per_slide', '>', 0)->value('price_per_slide') ?? 5;
            $basePrice += ($validated['slides'] * $slidePrice);
        }
        
        Log::info('Cost per page calculation:', [
            'aclevel' => $aclevel,
            'urgency' => $urgency,
            'spacing' => $spacing,
            'subject' => $subject,
            'paperType' => $paperTypeValue,
            'baseCostPerPage' => $baseCostPerPage,
            'costPerPage' => $costPerPage,
            'pages' => $validated['pages'],
            'basePrice' => $basePrice
        ]);
        
        return round($basePrice, 2);
    }

    /**
     * Get generic discount information for anonymous users
     *
     * @return array
     */
    protected function getDiscountInfo()
    {
        return [
            'minimum_discount' => 10, // Everyone gets at least 10%
            'newcomer_discount' => 20, // New customers get 20%
            'volume_discount_threshold' => 10, // 10+ pages get volume discount
            'loyalty_available' => true, // Returning customers get loyalty rewards
            'flash_discounts_available' => Coupon::where('coupon_type', 'flash')->active()->validByDate()->exists(),
            'estimated_savings_range' => [10, 25], // Typical savings range
            'guarantee_message' => 'All customers save with our discount system - minimum 10% guaranteed!'
        ];
    }

    /**
     * Get suggested coupons for the current user
     *
     * @return array
     */
    protected function getSuggestedCoupons()
    {
        $user = Auth::user();
        if (!$user) {
            return [];
        }

        // Default order context (will be refined on the frontend)
        $orderContext = [
            'order_total' => 0,
            'pages' => 1,
            'previous_orders' => $user->orders()->count()
        ];

        // Get the best discount for the user
        $bestDiscount = $this->couponService->getBestDiscount($user, $orderContext);

        if (!$bestDiscount) {
            return [];
        }

        $coupon = $bestDiscount['coupon'];
        $discountType = $bestDiscount['type'];

        return [
            [
                'code' => $coupon->coupon_code,
                'type' => $coupon->coupon_type,
                'discount_type' => $coupon->discount_type,
                'discount_amount' => $coupon->discount_amount,
                'description' => $this->getCouponDescription($coupon),
                'expiry_date' => $coupon->expiry_date ? $coupon->expiry_date->format('Y-m-d H:i:s') : null,
                'is_default' => $discountType === 'default',
                'is_special' => $discountType === 'special',
            ]
        ];
    }

    /**
     * Generate a human-readable description for a coupon
     *
     * @param Coupon $coupon
     * @return string
     */
    protected function getCouponDescription(Coupon $coupon)
    {
        $amount = $coupon->discount_amount;
        $type = $coupon->discount_type === 'percentage' ? '%' : '$';

        switch ($coupon->coupon_type) {
            case Coupon::TYPE_FIRST_ORDER:
                return "First order discount: {$amount}{$type} off your first order!";

            case Coupon::TYPE_FLASH:
                $expiry = $coupon->expiry_date ? ' until ' . $coupon->expiry_date->format('M d') : '';
                return "Flash sale: {$amount}{$type} off{$expiry}!";

            case Coupon::TYPE_LOYALTY:
                if ($coupon->loyaltyTier) {
                    return "{$coupon->loyaltyTier->name} customer: {$amount}{$type} off!";
                }
                return "Loyalty discount: {$amount}{$type} off!";

            case Coupon::TYPE_ANNIVERSARY:
                return "Anniversary special: {$amount}{$type} off to celebrate!";

            case Coupon::TYPE_SEASONAL:
                if ($coupon->seasonalType) {
                    return "{$coupon->seasonalType->name}: {$amount}{$type} off!";
                }
                return "Seasonal discount: {$amount}{$type} off!";

            case Coupon::TYPE_VOLUME:
                $pages = $coupon->min_pages ?? 0;
                return "Volume discount: {$amount}{$type} off orders with {$pages}+ pages!";

            case Coupon::TYPE_DEFAULT:
                return "Default discount: {$amount}{$type} off your order!";

            case Coupon::TYPE_REFERRAL:
                return "Referral discount: {$amount}{$type} off your order!";

            default:
                return "Special offer: {$amount}{$type} off your order!";
        }
    }

    /**
     * Validate a coupon code with enhanced logic
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function validateCoupon(Request $request)
    {
        // Validate request
        $request->validate([
            'code' => 'required|string',
            'pages' => 'nullable|integer|min:1',
            'order_total' => 'nullable|numeric|min:0'
        ]);

        $code = $request->input('code');
        $user = Auth::user();

        // Create context for validation
        $context = [
            'user_id' => $user->id,
            'pages' => $request->input('pages', 1),
            'order_total' => $request->input('order_total', 0),
            'previous_orders' => $user->orders()->count()
        ];

        // Get the coupon for additional details
        $coupon = Coupon::where('coupon_code', $code)
            ->where('is_active', 1)
            ->first();

        if (!$coupon) {
            return response()->json([
                'valid' => false,
                'message' => 'Coupon not found or inactive.'
            ]);
        }

        // Block referral coupons from manual validation
        if ($coupon->coupon_type === Coupon::TYPE_REFERRAL) {
            return response()->json([
                'valid' => false,
                'message' => 'Referral coupons are currently not available.'
            ]);
        }

        // Check for first_order coupon type and validate
        if ($coupon->coupon_type === 'first_order') {
            $previousOrdersCount = $user->orders()->where('payment_status', '!=', 'unpaid')->count();
            if ($previousOrdersCount > 0) {
                return response()->json([
                    'valid' => false,
                    'message' => 'This coupon is only valid for first-time customers.'
                ]);
            }
        }

        // Continue with other validations...
        // Check for expiry date
        if ($coupon->expiry_date && now()->isAfter($coupon->expiry_date)) {
            return response()->json([
                'valid' => false,
                'message' => 'This coupon has expired.'
            ]);
        }

        // Check for usage limits
        if ($coupon->max_uses && $coupon->uses_count >= $coupon->max_uses) {
            return response()->json([
                'valid' => false,
                'message' => 'This coupon has reached its usage limit.'
            ]);
        }

        // Check for minimum order requirements
        if ($coupon->min_order_amount && $request->input('order_total', 0) < $coupon->min_order_amount) {
            return response()->json([
                'valid' => false,
                'message' => "This coupon requires a minimum order of {$coupon->min_order_amount}."
            ]);
        }

        // Check for minimum pages requirements
        if ($coupon->min_pages && $request->input('pages', 1) < $coupon->min_pages) {
            return response()->json([
                'valid' => false,
                'message' => "This coupon requires a minimum of {$coupon->min_pages} pages."
            ]);
        }

        // Check per-user usage limit
        if ($coupon->per_user_limit) {
            // $userUsageCount = DB::table('coupon_usage')
            //     ->where('coupon_id', $coupon->id)
            //     ->where('user_id', $user->id)
            //     ->count();
            $userUsageCount = \App\Models\UserCoupon::getUserUsageCount($user->id, $coupon->id);

            if ($userUsageCount >= $coupon->per_user_limit) {
                return response()->json([
                    'valid' => false,
                    'message' => "You've already used this coupon the maximum number of times."
                ]);
            }
        }

        // Calculate the discount
        $discount = 0;
        if ($request->has('order_total') && $request->input('order_total') > 0) {
            $discount = $this->couponService->calculateDiscount(
                $coupon,
                $request->input('order_total'),
                $request->input('pages', 1)
            );
        }

        return response()->json([
            'valid' => true,
            'type' => $coupon->discount_type,
            'amount' => $coupon->discount_amount,
            'calculated_discount' => $discount,
            'coupon_type' => $coupon->coupon_type,
            'description' => $this->getCouponDescription($coupon),
            'expiry_date' => $coupon->expiry_date ? $coupon->expiry_date->format('Y-m-d') : null
        ]);
    }




    /**
     * Suggest the optimal coupon based on order details
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function suggestCoupon(Request $request)
    {
        $request->validate([
            'pages' => 'required|integer|min:1',
            'order_total' => 'required|numeric|min:0'
        ]);

        $user = Auth::user();
        $context = [
            'user_id' => $user->id,
            'pages' => $request->input('pages'),
            'order_total' => $request->input('order_total'),
            'previous_orders' => $user->orders()->count()
        ];

        // Find the best coupon for this order
        $bestCoupon = $this->couponService->findOptimalCoupon($user, $context);

        if (!$bestCoupon) {
            return response()->json([
                'has_suggestion' => false,
                'message' => 'No applicable coupons found.'
            ]);
        }

        // Calculate the discount
        $discount = $this->couponService->calculateDiscount(
            $bestCoupon,
            $request->input('order_total'),
            $request->input('pages')
        );

        return response()->json([
            'has_suggestion' => true,
            'coupon' => [
                'code' => $bestCoupon->coupon_code,
                'type' => $bestCoupon->discount_type,
                'amount' => $bestCoupon->discount_amount,
                'calculated_discount' => $discount,
                'description' => $this->getCouponDescription($bestCoupon),
                'coupon_type' => $bestCoupon->coupon_type
            ]
        ]);
    }

    /**
     * Store a newly created order
     */
    public function store(Request $request)
    {
        // LOG: Track order creation attempt with file upload information
        Log::info('Order creation attempt started', [
            'user_id' => Auth::id(),
            'has_files' => $request->hasFile('files'),
            'files_count' => $request->hasFile('files') ? count($request->file('files')) : 0,
            'files_data' => $request->hasFile('files') ? array_map(function($file) {
                return [
                    'name' => $file->getClientOriginalName(),
                    'size' => $file->getSize(),
                    'mime' => $file->getMimeType(),
                ];
            }, $request->file('files')) : [],
            'all_request_keys' => array_keys($request->all()),
        ]);

        $timezone = $request->input('clientTimezone');

        // Validate the request data
        try {
            $validated = $request->validate([
                'title' => 'required|string|max:250',
                'instructions' => 'required|string',
                'pages' => 'required|integer|min:1',
                'subject' => 'required|string',
                'aclevel' => 'required|string',
                'papertype' => 'required|string',
                'ppslides' => 'nullable|integer|min:0',
                'deadline' => 'required|string',
                'spacing' => 'required|string',
                'totalAmount' => 'required|numeric|min:0',
                'styles' => 'required|string',
                'sources' => 'required|integer|min:0',
                'language' => 'required|string',
                'files.*' => 'nullable|file|max:10240', // 10MB max per file
                'currency' => 'required|string|exists:currencies,currency_code', // Validate currency exists
                'currencyRate' => 'required|numeric|min:0', // Validate currency rate is a positive number
                'couponCode' => 'nullable|string', // Allow coupon code input
            ]);
        } catch (\Illuminate\Validation\ValidationException $e) {
            Log::error('Order validation failed', [
                'user_id' => Auth::id(),
                'errors' => $e->errors(),
                'has_files' => $request->hasFile('files'),
                'files_count' => $request->hasFile('files') ? count($request->file('files')) : 0,
            ]);
            throw $e;
        }

        // Generate order number
        $maxOrderId = Order::getMaxOrderId() ?? 59700;
        $orderNumber = $maxOrderId + 1;

        // Get client timezone from request
        $clientTimezone = request()->input('clientTimezone');

        // Set timezone for deadline calculation
        $deadline = now()->setTimezone($clientTimezone);

        // Parse deadline string to get actual deadline date
        $deadlineParts = explode('#', $validated['deadline']);
        $deadlineValue = (int)$deadlineParts[1];
        $deadlineUnit = $deadlineParts[2] ?? 'Hours';

        $deadline = $deadlineUnit === 'Days'
            ? $deadline->addDays($deadlineValue)
            : $deadline->addHours($deadlineValue);

        $academicLevel = optional(explode('#', $request->aclevel))[1] ?? null;
        $subject = optional(explode('#', $request->subject))[1] ?? null;
        $paperType = optional(explode('#', $request->papertype))[1] ?? null;
        $spacing  = optional(explode('#', $request->spacing))[1] ?? null;

        // Create the order
        $order = new Order();

        $writerDeadline = $order->calculateWriterDeadline($deadline);
        $order->writer_deadline = $writerDeadline;

        // Initialize values
        $netAmount = $request->input('totalAmount', 0);
        $discountAmount = 0;

        // Apply coupon if provided
        if ($request->filled('couponCode')) {
            $couponCode = $request->input('couponCode');

            // Create context for coupon validation
            $context = [
                'user_id' => Auth::id(),
                'pages' => $validated['pages'],
                'order_total' => $validated['totalAmount'],
                'previous_orders' => Auth::user()->orders()->count()
            ];

            // Validate coupon
            $validationResult = $this->couponService->validateCoupon($couponCode, $context);

            if ($validationResult['valid']) {
                // Get the coupon
                $coupon = Coupon::where('coupon_code', $couponCode)
                    ->active()
                    ->first();

                if ($coupon) {
                    // Calculate discount
                    $discountAmount = $this->couponService->calculateDiscount(
                        $coupon,
                        $validated['totalAmount'],
                        $validated['pages']
                    );

                    // Update net amount
                    $netAmount = $validated['totalAmount'] - $discountAmount;

                    // Set coupon data on order
                    $order->coupon = $couponCode;
                    $order->coupon_id = $coupon->id;
                }
            }
        }

        $writerAmount = $order->calculateWriterAmount($netAmount);
        $order->writer_amount = $writerAmount;
        $order->order_number = $orderNumber;
        $order->user_id = Auth::id();
        $order->title = $validated['title'];
        $order->type_of_paper = $paperType;
        $order->subject = $subject;
        $order->academic_level = $academicLevel;
        $order->urgency = $request->input('urgency');
        $order->pages = $validated['pages'];
        $order->instructions = $validated['instructions'];
        $order->styles = $validated['styles'];
        $order->dateposted = now()->setTimezone($timezone);
        $order->deadline = $deadline;
        $order->client_timezone = $timezone;
        $order->sources = $validated['sources'];
        $order->order_amount = $validated['totalAmount'];
        $order->net_amount = $netAmount;
        $order->discount = $discountAmount;

        // Save currency information
        $order->currency = $request->input('currency', 'USD');
        $order->currency_rate = $request->input('currencyRate', 1.0);

        $order->payment_status = 'unpaid';
        $order->order_status = 'pending_payment';
        $order->powerpoint_slides = $validated['ppslides'];
        $order->spacing = $spacing;

        $order->save();

        // LOG: Order saved successfully, now checking for files
        Log::info('Order saved successfully, checking for file uploads', [
            'order_id' => $order->id,
            'order_number' => $order->order_number,
            'has_files' => $request->hasFile('files'),
            'files_count' => $request->hasFile('files') ? count($request->file('files')) : 0,
        ]);

        // Handle file uploads
        if ($request->hasFile('files')) {
            Log::info('Files detected in request, starting upload process', [
                'order_id' => $order->id,
                'files_count' => count($request->file('files')),
            ]);

            try {
                foreach ($request->file('files') as $index => $file) {
                    Log::info('Processing individual file', [
                        'order_id' => $order->id,
                        'file_index' => $index,
                        'file_name' => $file->getClientOriginalName(),
                        'file_size' => $file->getSize(),
                        'file_mime' => $file->getMimeType(),
                        'is_valid' => $file->isValid(),
                    ]);

                    // Validate file before processing
                    if (!$file->isValid()) {
                        Log::error('Invalid file detected during upload', [
                            'order_id' => $order->id,
                            'file_index' => $index,
                            'file_name' => $file->getClientOriginalName(),
                            'error' => $file->getError(),
                            'error_message' => $file->getErrorMessage(),
                        ]);
                        continue; // Skip invalid file
                    }

                    $fileName = $file->getClientOriginalName();
                    $fileType = $file->getClientMimeType();
                    $fileSize = $file->getSize();

                    // Store file in storage/app/public/orders
                    $filePath = $file->store('orders', 'public');

                    if (!$filePath) {
                        Log::error('Failed to store file', [
                            'order_id' => $order->id,
                            'file_name' => $fileName,
                        ]);
                        continue; // Skip this file
                    }

                    Log::info('File stored in storage', [
                        'order_id' => $order->id,
                        'file_name' => $fileName,
                        'file_path' => $filePath,
                        'storage_path' => storage_path('app/public/' . $filePath),
                        'file_exists' => file_exists(storage_path('app/public/' . $filePath)),
                    ]);

                    // Create order file record
                    $orderFile = OrderFile::create([
                        'order_id' => $order->id,
                        'file_name' => $fileName,
                        'file_type' => $fileType,
                        'uploader_id' => Auth::id(),
                        'uploader_name' => Auth::user()->name,
                        'uploader_user_type' => 'client',
                        'file_url' => $filePath,
                        'file_size' => $fileSize,
                        'file_status' => OrderFile::STATUS_PENDING,  // Fixed: Use proper status constant
                        'file_category' => OrderFile::CATEGORY_INSTRUCTIONS,  // Fixed: Set proper category
                        'access_control' => OrderFile::ACCESS_CLIENT_ADMIN,  // Fixed: Set access control
                        'date_uploaded' => now()->setTimezone($clientTimezone)
                    ]);

                    Log::info('OrderFile record created', [
                        'order_id' => $order->id,
                        'order_file_id' => $orderFile->id,
                        'file_name' => $orderFile->file_name,
                        'file_url' => $orderFile->file_url,
                    ]);
                }

                Log::info('All files processed successfully', [
                    'order_id' => $order->id,
                    'files_processed' => count($request->file('files')),
                ]);
            } catch (\Exception $e) {
                Log::error('Exception during file upload processing', [
                    'order_id' => $order->id,
                    'error' => $e->getMessage(),
                    'trace' => $e->getTraceAsString(),
                ]);
                // Don't throw - allow order creation to succeed even if file upload fails
            }
        } else {
            Log::info('No files in request', [
                'order_id' => $order->id,
                'request_has_files_key' => $request->has('files'),
                'request_files_value' => $request->input('files'),
            ]);
        }

        // Record coupon usage if a valid coupon was applied
        if ($order->coupon_id && $discountAmount > 0) {
            $this->couponService->trackCouponUsage(
                Coupon::find($order->coupon_id),
                Auth::id(),
                $order->id,
                $validated['totalAmount'],
                $discountAmount
            );
        }

        //  after saving the order and file uploads, send notifications
        $order->user->notify(new NewOrderNotification($order)); // User notification

        // $admin = new AdminNotifiable(env('ADMIN_EMAIL'));
        // Notification::send($admin, new NewOrderNotification($order, true));

        // Notify all admins of the new order
        AdminNotificationService::notifyAllAdmins(
            new NewOrderNotification($order, true)
        );

        return redirect()->route('orders.show', $order->id)->with('message', 'Order created successfully!');
    }

    /**
     * Display the specified resource.
     */
    public function show(Order $order)
    {
        if (Auth::id() != $order->user_id) {
            abort(403, 'Unauthorized action.');
        }

        // Determine if the user can delete the file
        $isAdmin = in_array(Auth::user()->user_type, ['admin', 'super_admin']);

        // Load the writer relationship with their profile
        $order->load(['writer:id,name', 'writer.writerProfile:id,user_id,nickname', 'orderFiles', 'revisions:id,order_id,status,revision_type,created_at']);

        // Get refund information from the service
        $refundInfo = $this->refundService->getOrderRefundInfo($order->id);

        // Fetch bids for this order
        $bids = $order->bids()
            ->with('writer:id,name,email', 'writer.writerProfile:id,user_id,nickname')
            ->latest()
            ->get()
            ->map(function ($bid) {
                // Ensure writer is never null in the output
                if (!$bid->writer) {
                    $bid->writer = (object)['id' => null, 'name' => 'Unknown Writer', 'email' => ''];
                }
                return $bid;
            });

        // Get coupon information if applicable
        $couponInfo = null;
        if ($order->coupon_code) {
            $coupon = Coupon::where('coupon_code', $order->coupon_code)->first();
            if ($coupon) {
                $couponInfo = [
                    'code' => $coupon->coupon_code,
                    'type' => $coupon->discount_type,
                    'amount' => $coupon->discount_amount,
                    'description' => $this->getCouponDescription($coupon),
                ];
            }
        }

        return inertia('Order/Show', [
            'order' => $order,
            'bids' => $bids,
            'isAdmin' => $isAdmin,
            'refundInfo' => $refundInfo,
            'couponInfo' => $couponInfo,
            'revisionConfig' => [
                'maxPerOrder' => config('revisions.max_per_order', 3),
                'types' => config('revisions.types', [])
            ]
        ]);
    }

    /**
     * Get refund information for an order.
     *
     * @param Order $order
     * @return array
     */
    private function getOrderRefundInfo(Order $order): array
    {
        // Default refund info
        $refundInfo = [
            'has_refunds' => false,
            'refunded_amount' => 0,
            'refund_date' => null,
            'refund_reason' => null,
            'is_fully_refunded' => $order->isRefunded(),
            'is_partially_refunded' => $order->isPartiallyRefunded(),
            'refunds' => []
        ];

        // If order has payments with refunds
        if ($order->payments->isNotEmpty()) {
            $refundedAmount = 0;
            $latestRefundDate = null;
            $refunds = [];

            foreach ($order->payments as $payment) {
                if ($payment->refunds->isEmpty()) {
                    continue;
                }

                // Calculate total refunded for this payment
                $paymentRefundedAmount = $payment->getRefundedAmount();
                $refundedAmount += $paymentRefundedAmount;

                // Track latest refund
                foreach ($payment->refunds as $refund) {
                    // Add refund details
                    $refunds[] = [
                        'id' => $refund->id,
                        'amount' => (float) $refund->amount,
                        'reason' => $refund->reason,
                        'date' => $refund->processed_at ? $refund->processed_at->toDateTimeString() : $refund->created_at->toDateTimeString(),
                        'currency' => $refund->currency ?? 'USD'
                    ];

                    // Update latest refund date
                    if (!$latestRefundDate || ($refund->processed_at && $refund->processed_at->isAfter($latestRefundDate))) {
                        $latestRefundDate = $refund->processed_at;
                        $refundInfo['refund_reason'] = $refund->reason;
                    }
                }
            }

            // Update refund info
            $refundInfo['has_refunds'] = true;
            $refundInfo['refunded_amount'] = $refundedAmount;
            $refundInfo['refund_date'] = $latestRefundDate ? $latestRefundDate->toDateTimeString() : null;
            $refundInfo['refunds'] = $refunds;
        }

        return $refundInfo;
    }

    /**
     * Get available loyalty coupons for the current user
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function getLoyaltyCoupons()
    {
        $user = Auth::user();
        $loyaltyTier = $this->couponService->getUserLoyaltyTier($user);

        if (!$loyaltyTier) {
            return response()->json([
                'has_coupons' => false,
                'tier' => null,
                'coupons' => []
            ]);
        }

        $coupons = Coupon::where('coupon_type', Coupon::TYPE_LOYALTY)
            ->where('loyalty_tier_id', $loyaltyTier->id)
            ->active()
            ->validByDate()
            ->withinUsageLimit()
            ->get()
            ->map(function ($coupon) {
                return [
                    'code' => $coupon->coupon_code,
                    'type' => $coupon->discount_type,
                    'amount' => $coupon->discount_amount,
                    'description' => $this->getCouponDescription($coupon),
                ];
            });

        return response()->json([
            'has_coupons' => $coupons->isNotEmpty(),
            'tier' => [
                'id' => $loyaltyTier->id,
                'name' => $loyaltyTier->name,
                'description' => $loyaltyTier->description,
                'required_orders' => $loyaltyTier->required_orders,
            ],
            'coupons' => $coupons
        ]);
    }

    /**
     * Get available seasonal coupons
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function getSeasonalCoupons()
    {
        $activeSeasons = SeasonalType::getCurrentlyActive()->pluck('id');

        if ($activeSeasons->isEmpty()) {
            return response()->json([
                'has_seasons' => false,
                'seasons' => [],
                'coupons' => []
            ]);
        }

        $seasons = SeasonalType::whereIn('id', $activeSeasons)->get()
            ->map(function ($season) {
                return [
                    'id' => $season->id,
                    'name' => $season->name,
                    'description' => $season->description,
                ];
            });

        $coupons = Coupon::where('coupon_type', Coupon::TYPE_SEASONAL)
            ->whereIn('seasonal_type_id', $activeSeasons)
            ->active()
            ->validByDate()
            ->withinUsageLimit()
            ->get()
            ->map(function ($coupon) {
                return [
                    'code' => $coupon->coupon_code,
                    'type' => $coupon->discount_type,
                    'amount' => $coupon->discount_amount,
                    'description' => $this->getCouponDescription($coupon),
                    'seasonal_type_id' => $coupon->seasonal_type_id,
                ];
            });

        return response()->json([
            'has_seasons' => true,
            'seasons' => $seasons,
            'coupons' => $coupons
        ]);
    }

    /**
     * Get first-order discount if applicable
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function getFirstOrderCoupon()
    {
        $user = Auth::user();

        if (!$this->couponService->isEligibleForFirstOrderDiscount($user)) {
            return response()->json([
                'eligible' => false,
                'coupon' => null
            ]);
        }

        $coupon = $this->couponService->getFirstOrderCoupon($user);

        if (!$coupon) {
            return response()->json([
                'eligible' => true,
                'coupon' => null
            ]);
        }

        return response()->json([
            'eligible' => true,
            'coupon' => [
                'code' => $coupon->coupon_code,
                'type' => $coupon->discount_type,
                'amount' => $coupon->discount_amount,
                'description' => $this->getCouponDescription($coupon),
            ]
        ]);
    }

    /**
     * Get qualified discounts for the current authenticated user
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function getQualifiedDiscounts(Request $request)
    {
        $user = Auth::user();
        
        if (!$user) {
            return response()->json([
                'success' => false,
                'message' => 'Authentication required'
            ], 401);
        }

        $orderContext = [
            'order_total' => $request->get('order_total', 0),
            'pages' => $request->get('pages', 1),
            'previous_orders' => $user->orders()->count()
        ];

        // Get all applicable coupons (both automatic and manual)
        // Automatic discounts: first_order, loyalty, seasonal, anniversary
        // Manual coupons: default, flash, volume (can be applied by user)
        $applicableDiscounts = $this->couponService->getApplicableCoupons($user, $orderContext);
        
        // Separate automatic vs manual coupons for display purposes
        $automaticDiscountTypes = [
            Coupon::TYPE_FIRST_ORDER,
            Coupon::TYPE_LOYALTY,
            Coupon::TYPE_SEASONAL,
            Coupon::TYPE_ANNIVERSARY,
        ];

        // Format discounts for frontend
        $discounts = $applicableDiscounts->map(function ($coupon) use ($automaticDiscountTypes) {
            return [
                'code' => $coupon->coupon_code,
                'type' => $coupon->coupon_type,
                'discount_type' => $coupon->discount_type,
                'discount_amount' => $coupon->discount_amount,
                'description' => $this->getCouponDescription($coupon),
                'is_automatic' => in_array($coupon->coupon_type, $automaticDiscountTypes)
            ];
        });

        return response()->json([
            'success' => true,
            'discounts' => $discounts->toArray()
        ]);
    }

    public function getWriterDetails(Request $request, $writerId)
    {
        // Fetch the writer with their profile and related details
        $writer = User::with([
            'writerProfile.educationLevel',
            'writerProfile.specialization',
            'writerProfile.writerLevel',
        ])->findOrFail($writerId);

        // Get the current client ID
        $clientId = Auth::id();

        // Prepare the response data
        $data = [
            'writer' => $writer,
            'activeOrdersCount' => $writer->activeOrdersCount(),
            'approvedOrdersCount' => $writer->approvedOrdersCount(),
            'previousOrders' => $writer->previousOrdersForClient($clientId),
        ];

        // Return the data as JSON
        return response()->json($data);
    }

    /**
     * Calculate order pricing with coupon validation
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function calculateOrderPricing(Request $request)
    {
        try {
            // Debug: Log the incoming request
            Log::info('calculateOrderPricing called', [
                'request_data' => $request->all(),
                'headers' => $request->headers->all()
            ]);
            
            $validated = $request->validate([
                'academic_level_id' => 'required|integer',
                'paper_type_id' => 'required|integer',
                'deadline_id' => 'required|integer',
                'pages' => 'required|integer|min:1',
                'slides' => 'nullable|integer|min:0',
                'subject_id' => 'required|integer',
                'spacing' => 'nullable|string',
                'coupon_code' => 'nullable|string',
            ]);

            // Calculate base pricing
            $basePrice = $this->calculateBasePrice($validated);
            
            // Initialize pricing response
            $pricing = [
                'base_price' => $basePrice,
                'total_before_discount' => $basePrice,
                'final_price' => $basePrice,
                'discount_amount' => 0,
                'breakdown' => [
                    'pages_cost' => $basePrice,
                    'slides_cost' => 0,
                    'cost_per_page' => $basePrice / $validated['pages'],
                ]
            ];

            // Add slides cost if applicable
            if (isset($validated['slides']) && $validated['slides'] > 0) {
                $slidePrice = AdminSetting::where('price_per_slide', '>', 0)->value('price_per_slide') ?? 5;
                $slidesCost = $validated['slides'] * $slidePrice;
                $pricing['breakdown']['slides_cost'] = $slidesCost;
                $pricing['total_before_discount'] += $slidesCost;
                $pricing['final_price'] += $slidesCost;
            }

            // Apply coupon discount if provided
            if (!empty($validated['coupon_code'])) {
                $coupon = Coupon::where('coupon_code', $validated['coupon_code'])
                    ->active()
                    ->validByDate()
                    ->withinUsageLimit()
                    ->first();

                if ($coupon) {
                    $discount = $this->couponService->calculateDiscount($coupon, $pricing['total_before_discount'], $validated['pages']);
                    $pricing['discount_amount'] = $discount;
                    $pricing['final_price'] = max(0, $pricing['total_before_discount'] - $discount);
                }
            }

            return response()->json([
                'success' => true,
                'pricing' => $pricing
            ]);
            
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Unable to calculate pricing. Please try again.',
                'error' => config('app.debug') ? $e->getMessage() : null
            ], 400);
        }
    }
}
