<?php

namespace App\Http\Controllers;

use App\Models\Order;
use App\Models\Payment;
use App\Models\PaymentMethod;
use App\Services\Payments\PayPalPaymentService;
use App\Services\Payments\PaymentServiceFactory;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;

class PayPalController extends Controller
{
    /**
     * Get PayPal client ID for frontend
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function getClientId()
    {
        try {
            // Try to get PayPal payment method from database
            $paypalMethod = PaymentMethod::where('name', 'paypal')
                ->where('is_active', true)
                ->first();

            if ($paypalMethod) {
                // Use the payment service to get client config
                $paymentService = new PayPalPaymentService($paypalMethod);
                $config = $paymentService->getClientConfig();

                return response()->json([
                    'clientId' => $config['clientId']
                ]);
            }

            // Fallback to config file for backward compatibility
            $mode = config('paypal.mode');
            $clientIdKey = $mode === 'sandbox' ? 'sandbox_client_id' : 'live_client_id';

            return response()->json([
                'clientId' => config("paypal.{$clientIdKey}")
            ]);
        } catch (\Exception $e) {
            Log::error('Error getting PayPal client ID: ' . $e->getMessage());

            return response()->json([
                'error' => 'Failed to get PayPal configuration'
            ], 500);
        }
    }

    /**
     * Create a PayPal order
     *
     * @param Request $request
     * @param int $orderId
     * @return \Illuminate\Http\JsonResponse
     */
    public function createOrder(Request $request, $orderId)
    {
        $order = Order::findOrFail($orderId);

        // Check if the order belongs to the authenticated user
        if ($order->user_id !== Auth::id()) {
            return response()->json(['error' => 'Unauthorized'], 403);
        }

        try {
            // Get the PayPal payment service
            $paypalMethod = PaymentMethod::where('name', 'paypal')
                ->where('is_active', true)
                ->first();

            if (!$paypalMethod) {
                return response()->json(['error' => 'PayPal payment method is not available'], 400);
            }

            $paymentService = new PayPalPaymentService($paypalMethod);

            // Initialize the payment
            $result = $paymentService->initializePayment($order);

            if (!$result['success']) {
                return response()->json(['error' => $result['error'] ?? 'Failed to create PayPal order'], 500);
            }

            return response()->json([
                'id' => $result['paypal_order_id']
            ]);
        } catch (\Exception $e) {
            Log::error('PayPal create order error: ' . $e->getMessage(), [
                'order_id' => $orderId,
                'trace' => $e->getTraceAsString()
            ]);

            return response()->json(['error' => $e->getMessage()], 500);
        }
    }

    /**
     * Process payment after PayPal approval
     *
     * @param Request $request
     * @param int $orderId
     * @return \Illuminate\Http\RedirectResponse
     */
    public function processPayment(Request $request, $orderId)
    {
        $order = Order::findOrFail($orderId);

        // Check if the order belongs to the authenticated user
        if ($order->user_id !== Auth::id()) {
            abort(403, 'Unauthorized action.');
        }

        try {
            // Get the PayPal payment service
            $paypalMethod = PaymentMethod::where('name', 'paypal')
                ->where('is_active', true)
                ->first();

            if (!$paypalMethod) {
                return redirect()->route('orders.show', $order->id)
                    ->with('error', 'PayPal payment method is not available');
            }

            $paymentService = new PayPalPaymentService($paypalMethod);

            // Process the payment
            $paymentData = [
                'paymentId' => $request->input('paymentId'),
                'payerId' => $request->input('payerId'),
                'details' => $request->input('details', []),
            ];

            $payment = $paymentService->processPayment($order, $paymentData);

            if ($payment->isCompleted()) {
                return redirect()->route('orders.show', $order->id)
                    ->with('message', 'Payment completed successfully!');
            } else {
                return redirect()->route('orders.show', $order->id)
                    ->with('error', 'Payment verification failed. Please contact support.');
            }
        } catch (\Exception $e) {
            Log::error('PayPal payment error: ' . $e->getMessage(), [
                'order_id' => $orderId,
                'trace' => $e->getTraceAsString()
            ]);

            return redirect()->route('orders.show', $order->id)
                ->with('error', 'Payment processing failed: ' . $e->getMessage());
        }
    }

    /**
     * Handle PayPal webhook events
     *
     * @param Request $request
     * @return \Illuminate\Http\Response
     */
    public function handleWebhook(Request $request)
    {
        $payload = $request->all();

        try {
            // Log the webhook event
            Log::info('PayPal webhook received', [
                'event_type' => $payload['event_type'] ?? 'unknown'
            ]);

            // Get the PayPal payment service
            $paypalMethod = PaymentMethod::where('name', 'paypal')->first();

            if (!$paypalMethod) {
                Log::error('PayPal payment method not found for webhook handling');
                return response('PayPal payment method not configured', 500);
            }

            $paymentService = new PayPalPaymentService($paypalMethod);

            // Handle the webhook
            $result = $paymentService->handleWebhook($payload);

            if ($result) {
                return response('Webhook processed successfully', 200);
            } else {
                return response('Failed to process webhook', 500);
            }
        } catch (\Exception $e) {
            Log::error('Error processing PayPal webhook: ' . $e->getMessage(), [
                'payload' => $payload,
                'trace' => $e->getTraceAsString()
            ]);

            return response('Webhook processing error: ' . $e->getMessage(), 500);
        }
    }
}
