<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Carbon\Carbon;

class SitemapController extends Controller
{
    /**
     * Generate XML sitemap
     */
    public function index()
    {
        $sitemap = $this->generateSitemap();

        return response($sitemap, 200)
            ->header('Content-Type', 'application/xml');
    }

    /**
     * Generate sitemap content
     */
    private function generateSitemap()
    {
        $baseUrl = config('app.url');
        $now = Carbon::now()->toISOString();

        // Define all URLs with their properties
        $urls = [
            // Main homepage
            [
                'url' => $baseUrl,
                'lastmod' => $now,
                'changefreq' => 'daily',
                'priority' => '1.0'
            ],

            // Main service pages (auto-discovered from routes)
            [
                'url' => $baseUrl . '/services',
                'lastmod' => $now,
                'changefreq' => 'weekly',
                'priority' => '0.9'
            ],
        ];

        // Auto-discover all service pages from routes
        $serviceUrls = $this->discoverServiceRoutes();
        $urls = array_merge($urls, $serviceUrls);

        // Add dynamic service pages from CMS
        $dynamicServiceUrls = $this->getDynamicServiceUrls();
        $urls = array_merge($urls, $dynamicServiceUrls);

        // Add static pages to URLs array
        $staticPages = [

            // Blog section
            [
                'url' => $baseUrl . '/blog',
                'lastmod' => $now,
                'changefreq' => 'daily',
                'priority' => '0.8'
            ],

            // Blog category pages
            [
                'url' => $baseUrl . '/blog/category/essay-writing',
                'lastmod' => $now,
                'changefreq' => 'weekly',
                'priority' => '0.7'
            ],
            [
                'url' => $baseUrl . '/blog/category/research-papers',
                'lastmod' => $now,
                'changefreq' => 'weekly',
                'priority' => '0.7'
            ],
            [
                'url' => $baseUrl . '/blog/category/academic-writing',
                'lastmod' => $now,
                'changefreq' => 'weekly',
                'priority' => '0.7'
            ],
            [
                'url' => $baseUrl . '/blog/category/study-tips',
                'lastmod' => $now,
                'changefreq' => 'weekly',
                'priority' => '0.7'
            ],
            [
                'url' => $baseUrl . '/blog/category/case-studies',
                'lastmod' => $now,
                'changefreq' => 'weekly',
                'priority' => '0.7'
            ],
            [
                'url' => $baseUrl . '/blog/category/literature-review',
                'lastmod' => $now,
                'changefreq' => 'weekly',
                'priority' => '0.7'
            ],
            // NEW: Added 3 new blog categories
            [
                'url' => $baseUrl . '/blog/category/assignments',
                'lastmod' => $now,
                'changefreq' => 'weekly',
                'priority' => '0.7'
            ],
            [
                'url' => $baseUrl . '/blog/category/dissertations',
                'lastmod' => $now,
                'changefreq' => 'weekly',
                'priority' => '0.7'
            ],
            [
                'url' => $baseUrl . '/blog/category/phd-writing',
                'lastmod' => $now,
                'changefreq' => 'weekly',
                'priority' => '0.7'
            ],

            // About Us section pages
            [
                'url' => $baseUrl . '/about',
                'lastmod' => $now,
                'changefreq' => 'monthly',
                'priority' => '0.7'
            ],
            [
                'url' => $baseUrl . '/writers',
                'lastmod' => $now,
                'changefreq' => 'weekly',
                'priority' => '0.7'
            ],
            [
                'url' => $baseUrl . '/guarantees',
                'lastmod' => $now,
                'changefreq' => 'monthly',
                'priority' => '0.7'
            ],
            [
                'url' => $baseUrl . '/reviews',
                'lastmod' => $now,
                'changefreq' => 'weekly',
                'priority' => '0.6'
            ],
            [
                'url' => $baseUrl . '/pricing-page',
                'lastmod' => $now,
                'changefreq' => 'weekly',
                'priority' => '0.7'
            ],

            // Contact page
            [
                'url' => $baseUrl . '/contact',
                'lastmod' => $now,
                'changefreq' => 'monthly',
                'priority' => '0.6'
            ],

            // Legal pages
            [
                'url' => $baseUrl . '/privacy-policy',
                'lastmod' => $now,
                'changefreq' => 'yearly',
                'priority' => '0.3'
            ],
            [
                'url' => $baseUrl . '/terms-of-service',
                'lastmod' => $now,
                'changefreq' => 'yearly',
                'priority' => '0.3'
            ],
            [
                'url' => $baseUrl . '/cookie-policy',
                'lastmod' => $now,
                'changefreq' => 'yearly',
                'priority' => '0.3'
            ],

            // Order placement page (high priority for conversions)
            [
                'url' => $baseUrl . '/place-order',
                'lastmod' => $now,
                'changefreq' => 'daily',
                'priority' => '0.9'
            ],

            // Authentication pages (lower priority but still indexed)
            [
                'url' => $baseUrl . '/login',
                'lastmod' => $now,
                'changefreq' => 'monthly',
                'priority' => '0.4'
            ],
            [
                'url' => $baseUrl . '/register',
                'lastmod' => $now,
                'changefreq' => 'monthly',
                'priority' => '0.4'
            ],
        ];

        // Merge all URLs together
        $urls = array_merge($urls, $staticPages);
        
        // Add dynamic blog URLs
        $dynamicUrls = $this->getDynamicBlogUrls();
        $urls = array_merge($urls, $dynamicUrls);

        // Valid changefreq values according to sitemap protocol
        $validChangefreq = ['always', 'hourly', 'daily', 'weekly', 'monthly', 'yearly', 'never'];

        // Generate XML
        $xml = '<?xml version="1.0" encoding="UTF-8"?>' . "\n";
        $xml .= '<urlset xmlns="http://www.sitemaps.org/schemas/sitemap/0.9">' . "\n";

        foreach ($urls as $url) {
            // Validate and sanitize changefreq - default to 'weekly' if invalid
            $changefreq = $url['changefreq'] ?? 'weekly';
            if (empty($changefreq) || !in_array(strtolower($changefreq), $validChangefreq)) {
                $changefreq = 'weekly';
            } else {
                $changefreq = strtolower($changefreq);
            }

            // Validate priority - ensure it's between 0.0 and 1.0
            $priority = $url['priority'] ?? '0.5';
            $priorityNum = (float) $priority;
            if ($priorityNum < 0) $priorityNum = 0;
            if ($priorityNum > 1) $priorityNum = 1;
            $priority = number_format($priorityNum, 1, '.', '');

            $xml .= '  <url>' . "\n";
            $xml .= '    <loc>' . htmlspecialchars($url['url']) . '</loc>' . "\n";
            $xml .= '    <lastmod>' . htmlspecialchars($url['lastmod']) . '</lastmod>' . "\n";
            $xml .= '    <changefreq>' . htmlspecialchars($changefreq) . '</changefreq>' . "\n";
            $xml .= '    <priority>' . htmlspecialchars($priority) . '</priority>' . "\n";
            $xml .= '  </url>' . "\n";
        }

        $xml .= '</urlset>';

        return $xml;
    }

    /**
     * Auto-discover service routes dynamically
     */
    private function discoverServiceRoutes()
    {
        $urls = [];
        $baseUrl = config('app.url');
        $now = \Carbon\Carbon::now()->toISOString();

        // Get all registered routes
        $routes = \Illuminate\Support\Facades\Route::getRoutes();

        $serviceRoutes = [];

        foreach ($routes as $route) {
            $uri = $route->uri();
            
            // Filter for service routes only
            if (
                strpos($uri, 'services/') === 0 &&  // Starts with 'services/'
                !strpos($uri, '{') &&                 // No route parameters
                !strpos($uri, 'api/') &&              // Not API routes
                $uri !== 'services'                   // Not the main services page
            ) {
                // Store unique service routes
                $serviceRoutes[$uri] = true;
            }
        }

        // Generate sitemap entries for each service route
        foreach (array_keys($serviceRoutes) as $uri) {
            $urls[] = [
                'url' => $baseUrl . '/' . $uri,
                'lastmod' => $now,
                'changefreq' => 'weekly',
                'priority' => '0.8'
            ];
        }

        return $urls;
    }

    /**
     * Get dynamic service page URLs from database
     * This method queries the database for all published service pages created via CMS
     */
    private function getDynamicServiceUrls()
    {
        $urls = [];
        $baseUrl = config('app.url');

        try {
            // Query published service pages from database
            $servicePages = \App\Models\Page::where('status', 'published')
                ->where(function ($query) {
                    $query->where('page_type', 'service')
                          ->orWhere('page_type', 'service_page');
                })
                ->orderBy('updated_at', 'desc')
                ->get();

            // Generate URLs for each service page
            foreach ($servicePages as $page) {
                // Determine priority based on SEO priority or default
                $priority = $page->seo_priority ? (string)($page->seo_priority / 100) : '0.8';
                
                // Determine change frequency based on last update
                $updatedDate = \Carbon\Carbon::parse($page->updated_at);
                $daysSinceUpdate = $updatedDate->diffInDays(now());
                
                if ($daysSinceUpdate < 7) {
                    $changefreq = 'daily';
                } elseif ($daysSinceUpdate < 30) {
                    $changefreq = 'weekly';
                } elseif ($daysSinceUpdate < 90) {
                    $changefreq = 'weekly';
                } else {
                    $changefreq = 'monthly';
                }

                $urls[] = [
                    'url' => $baseUrl . '/services/' . $page->slug,
                    'lastmod' => $page->updated_at->toIso8601String(),
                    'changefreq' => $changefreq,
                    'priority' => $priority
                ];
            }
        } catch (\Exception $e) {
            // If there's an error (e.g., table doesn't exist), just return empty array
            \Log::warning('Error fetching dynamic service URLs for sitemap: ' . $e->getMessage());
        }

        return $urls;
    }

    /**
     * Get dynamic blog URLs from database
     * This method queries the database for all published blog posts
     */
    private function getDynamicBlogUrls()
    {
        $urls = [];
        $baseUrl = config('app.url');

        // Query published blog posts from database
        $blogPosts = \App\Models\BlogPost::where('status', 'published')
            ->orderBy('published_at', 'desc')
            ->get();

        // Generate URLs for each blog post
        foreach ($blogPosts as $post) {
            // Determine priority based on featured status
            $priority = $post->featured ? '0.8' : '0.7';
            
            // Determine change frequency based on post age
            $publishedDate = \Carbon\Carbon::parse($post->published_at);
            $daysOld = $publishedDate->diffInDays(now());
            
            if ($daysOld < 30) {
                $changefreq = 'weekly';
            } elseif ($daysOld < 90) {
                $changefreq = 'weekly';
            } else {
                $changefreq = 'monthly';
            }

            $urls[] = [
                'url' => $baseUrl . '/blog/' . $post->slug,
                'lastmod' => $post->updated_at->toIso8601String(),
                'changefreq' => $changefreq,
                'priority' => $priority
            ];
        }

        // Add blog category URLs
        $categories = \App\Models\BlogCategory::all();
        foreach ($categories as $category) {
            $urls[] = [
                'url' => $baseUrl . '/blog/category/' . $category->slug,
                'lastmod' => now()->toIso8601String(),
                'changefreq' => 'weekly',
                'priority' => '0.7'
            ];
        }

        return $urls;
    }

    /**
     * Generate blog-specific sitemap
     */
    public function blogSitemap()
    {
        $sitemap = $this->generateBlogSitemap();

        return response($sitemap, 200)
            ->header('Content-Type', 'application/xml');
    }

    /**
     * Generate blog sitemap content (dynamic from database)
     */
    private function generateBlogSitemap()
    {
        $baseUrl = config('app.url');
        $now = Carbon::now()->toISOString();

        // Blog main page
        $urls = [
            [
                'url' => $baseUrl . '/blog',
                'lastmod' => $now,
                'changefreq' => 'daily',
                'priority' => '0.8'
            ]
        ];

        // Get blog categories from database
        $categories = \App\Models\BlogCategory::all();
        foreach ($categories as $category) {
            $urls[] = [
                'url' => $baseUrl . '/blog/category/' . $category->slug,
                'lastmod' => $category->updated_at->toIso8601String(),
                'changefreq' => 'weekly',
                'priority' => '0.7'
            ];
        }

        // Get all published blog posts from database
        $blogPosts = \App\Models\BlogPost::where('status', 'published')
            ->orderBy('published_at', 'desc')
            ->get();

        foreach ($blogPosts as $post) {
            // Determine priority based on featured status
            $priority = $post->featured ? '0.8' : '0.7';
            
            // Determine change frequency based on post age
            $publishedDate = Carbon::parse($post->published_at);
            $daysOld = $publishedDate->diffInDays(now());
            
            if ($daysOld < 30) {
                $changefreq = 'weekly';
            } elseif ($daysOld < 90) {
                $changefreq = 'weekly';
            } else {
                $changefreq = 'monthly';
            }

            $urls[] = [
                'url' => $baseUrl . '/blog/' . $post->slug,
                'lastmod' => $post->updated_at->toIso8601String(),
                'changefreq' => $changefreq,
                'priority' => $priority
            ];
        }

        // Valid changefreq values according to sitemap protocol
        $validChangefreq = ['always', 'hourly', 'daily', 'weekly', 'monthly', 'yearly', 'never'];

        // Generate XML
        $xml = '<?xml version="1.0" encoding="UTF-8"?>' . "\n";
        $xml .= '<urlset xmlns="http://www.sitemaps.org/schemas/sitemap/0.9">' . "\n";

        foreach ($urls as $url) {
            // Validate and sanitize changefreq - default to 'weekly' if invalid
            $changefreq = $url['changefreq'] ?? 'weekly';
            if (empty($changefreq) || !in_array(strtolower($changefreq), $validChangefreq)) {
                $changefreq = 'weekly';
            } else {
                $changefreq = strtolower($changefreq);
            }

            // Validate priority - ensure it's between 0.0 and 1.0
            $priority = $url['priority'] ?? '0.5';
            $priorityNum = (float) $priority;
            if ($priorityNum < 0) $priorityNum = 0;
            if ($priorityNum > 1) $priorityNum = 1;
            $priority = number_format($priorityNum, 1, '.', '');

            $xml .= '  <url>' . "\n";
            $xml .= '    <loc>' . htmlspecialchars($url['url']) . '</loc>' . "\n";
            $xml .= '    <lastmod>' . htmlspecialchars($url['lastmod']) . '</lastmod>' . "\n";
            $xml .= '    <changefreq>' . htmlspecialchars($changefreq) . '</changefreq>' . "\n";
            $xml .= '    <priority>' . htmlspecialchars($priority) . '</priority>' . "\n";
            $xml .= '  </url>' . "\n";
        }

        $xml .= '</urlset>';

        return $xml;
    }

    /**
     * Generate news sitemap
     */
    public function newsSitemap()
    {
        $sitemap = $this->generateNewsSitemap();

        return response($sitemap, 200)
            ->header('Content-Type', 'application/xml');
    }

    /**
     * Generate news sitemap content (dynamic - last 2 days for Google News)
     */
    private function generateNewsSitemap()
    {
        $baseUrl = config('app.url');
        $now = Carbon::now();

        // Get recent blog posts (last 2 days for Google News sitemap)
        $recentPosts = \App\Models\BlogPost::where('status', 'published')
            ->where('published_at', '>=', $now->copy()->subDays(2))
            ->orderBy('published_at', 'desc')
            ->get();

        // Generate XML
        $xml = '<?xml version="1.0" encoding="UTF-8"?>' . "\n";
        $xml .= '<urlset xmlns="http://www.sitemaps.org/schemas/sitemap/0.9" xmlns:news="http://www.google.com/schemas/sitemap-news/0.9">' . "\n";

        foreach ($recentPosts as $post) {
            // Generate keywords from categories and tags
            $keywords = [];
            if ($post->category) {
                $keywords[] = $post->category->name;
            }
            if ($post->tags) {
                $keywords = array_merge($keywords, $post->tags->pluck('name')->toArray());
            }
            $keywordsString = implode(', ', array_slice($keywords, 0, 10)); // Max 10 keywords

            $xml .= '  <url>' . "\n";
            $xml .= '    <loc>' . htmlspecialchars($baseUrl . '/blog/' . $post->slug) . '</loc>' . "\n";
            $xml .= '    <news:news>' . "\n";
            $xml .= '      <news:publication>' . "\n";
            $xml .= '        <news:name>Academic Scribe Blog</news:name>' . "\n";
            $xml .= '        <news:language>en</news:language>' . "\n";
            $xml .= '      </news:publication>' . "\n";
            $xml .= '      <news:publication_date>' . Carbon::parse($post->published_at)->toIso8601String() . '</news:publication_date>' . "\n";
            $xml .= '      <news:title>' . htmlspecialchars($post->title) . '</news:title>' . "\n";
            if ($keywordsString) {
                $xml .= '      <news:keywords>' . htmlspecialchars($keywordsString) . '</news:keywords>' . "\n";
            }
            $xml .= '    </news:news>' . "\n";
            $xml .= '  </url>' . "\n";
        }

        $xml .= '</urlset>';

        return $xml;
    }

    /**
     * Generate images sitemap
     */
    public function imageSitemap()
    {
        $sitemap = $this->generateImageSitemap();

        return response($sitemap, 200)
            ->header('Content-Type', 'application/xml');
    }

    /**
     * Generate image sitemap content (dynamic from database)
     */
    private function generateImageSitemap()
    {
        $baseUrl = config('app.url');
        $images = [];

        // Get all published blog posts with featured images
        $blogPosts = \App\Models\BlogPost::where('status', 'published')
            ->whereNotNull('featured_image')
            ->orderBy('published_at', 'desc')
            ->get();

        foreach ($blogPosts as $post) {
            $images[] = [
                'loc' => $baseUrl . '/blog/' . $post->slug,
                'image_loc' => $post->featured_image,
                'title' => $post->title,
                'caption' => $post->featured_image_alt ?? $post->excerpt
            ];
        }

        // Add static service page images
        $serviceImages = [
            [
                'loc' => $baseUrl . '/services/essay-writing',
                'image_loc' => $baseUrl . '/images/services/essay-writing-service.jpg',
                'title' => 'Professional Essay Writing Service',
                'caption' => 'Expert essay writing assistance for students'
            ],
            [
                'loc' => $baseUrl . '/services/research-papers',
                'image_loc' => $baseUrl . '/images/services/research-paper-service.jpg',
                'title' => 'Research Paper Writing Service',
                'caption' => 'Professional research paper writing assistance'
            ],
            [
                'loc' => $baseUrl . '/services/dissertation-writing',
                'image_loc' => $baseUrl . '/images/services/dissertation-writing-service.jpg',
                'title' => 'Dissertation Writing Service',
                'caption' => 'Expert dissertation writing assistance'
            ],
            [
                'loc' => $baseUrl . '/services/assignments',
                'image_loc' => $baseUrl . '/images/services/assignment-writing-service.jpg',
                'title' => 'Assignment Writing Service',
                'caption' => 'Professional assignment writing help'
            ],
        ];

        $images = array_merge($images, $serviceImages);

        // Generate XML
        $xml = '<?xml version="1.0" encoding="UTF-8"?>' . "\n";
        $xml .= '<urlset xmlns="http://www.sitemaps.org/schemas/sitemap/0.9" xmlns:image="http://www.google.com/schemas/sitemap-image/1.1">' . "\n";

        foreach ($images as $image) {
            $xml .= '  <url>' . "\n";
            $xml .= '    <loc>' . htmlspecialchars($image['loc']) . '</loc>' . "\n";
            $xml .= '    <image:image>' . "\n";
            $xml .= '      <image:loc>' . htmlspecialchars($image['image_loc']) . '</image:loc>' . "\n";
            $xml .= '      <image:title>' . htmlspecialchars($image['title']) . '</image:title>' . "\n";
            $xml .= '      <image:caption>' . htmlspecialchars(substr($image['caption'], 0, 200)) . '</image:caption>' . "\n";
            $xml .= '    </image:image>' . "\n";
            $xml .= '  </url>' . "\n";
        }

        $xml .= '</urlset>';

        return $xml;
    }

    /**
     * Generate sitemap index
     */
    public function sitemapIndex()
    {
        $baseUrl = config('app.url');
        $now = Carbon::now()->toISOString();

        $sitemaps = [
            [
                'loc' => $baseUrl . '/sitemap.xml',
                'lastmod' => $now
            ],
            [
                'loc' => $baseUrl . '/sitemap-blog.xml',
                'lastmod' => $now
            ],
            [
                'loc' => $baseUrl . '/sitemap-news.xml',
                'lastmod' => $now
            ],
            [
                'loc' => $baseUrl . '/sitemap-images.xml',
                'lastmod' => $now
            ],
        ];

        // Generate XML
        $xml = '<?xml version="1.0" encoding="UTF-8"?>' . "\n";
        $xml .= '<sitemapindex xmlns="http://www.sitemaps.org/schemas/sitemap/0.9">' . "\n";

        foreach ($sitemaps as $sitemap) {
            $xml .= '  <sitemap>' . "\n";
            $xml .= '    <loc>' . htmlspecialchars($sitemap['loc']) . '</loc>' . "\n";
            $xml .= '    <lastmod>' . $sitemap['lastmod'] . '</lastmod>' . "\n";
            $xml .= '  </sitemap>' . "\n";
        }

        $xml .= '</sitemapindex>';

        return response($xml, 200)
            ->header('Content-Type', 'application/xml');
    }

    /**
     * Generate robots.txt
     */
    public function robots()
    {
        $baseUrl = config('app.url');

        $content = "User-agent: *\n";
        $content .= "Allow: /\n";
        $content .= "Disallow: /dashboard\n";
        $content .= "Disallow: /admin\n";
        $content .= "Disallow: /writer\n";
        $content .= "Disallow: /api\n";
        $content .= "Disallow: /login\n";
        $content .= "Disallow: /register\n";
        $content .= "Disallow: /password\n";
        $content .= "Disallow: /webhooks\n";
        $content .= "Disallow: /files\n";
        $content .= "Disallow: /notifications\n";
        $content .= "Disallow: /profile\n";
        $content .= "Disallow: /orders\n";
        $content .= "Disallow: /bids\n";
        $content .= "Disallow: /payments\n";
        $content .= "\n";
        $content .= "# Sitemap\n";
        $content .= "Sitemap: {$baseUrl}/sitemap-index.xml\n";
        $content .= "Sitemap: {$baseUrl}/sitemap.xml\n";
        $content .= "Sitemap: {$baseUrl}/sitemap-blog.xml\n";
        $content .= "Sitemap: {$baseUrl}/sitemap-news.xml\n";
        $content .= "Sitemap: {$baseUrl}/sitemap-images.xml\n";
        $content .= "\n";
        $content .= "# Crawl-delay\n";
        $content .= "Crawl-delay: 1\n";

        return response($content, 200)
            ->header('Content-Type', 'text/plain');
    }
}
