<?php

namespace App\Http\Controllers;

use App\Models\Order;
use App\Models\Payment;
use App\Models\PaymentMethod;
use App\Services\Payments\StripePaymentService;
use App\Services\OrderStatusService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;

class StripeController extends Controller
{
    /**
     * Get Stripe publishable key for frontend
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function getPublishableKey()
    {
        try {
            // Try to get Stripe payment method from database
            $stripeMethod = PaymentMethod::where('name', 'stripe')
                ->where('is_active', true)
                ->first();

            if ($stripeMethod) {
                // Use the payment service to get client config
                $orderStatusService = app(OrderStatusService::class);
                $paymentService = new StripePaymentService($stripeMethod, $orderStatusService);
                $config = $paymentService->getClientConfig();

                return response()->json([
                    'publishableKey' => $config['publishableKey']
                ]);
            }

            // Fallback to env for backward compatibility
            return response()->json([
                'publishableKey' => env('STRIPE_KEY')
            ]);
        } catch (\Exception $e) {
            Log::error('Error getting Stripe publishable key: ' . $e->getMessage());

            return response()->json([
                'error' => 'Failed to get Stripe configuration'
            ], 500);
        }
    }

    /**
     * Create a Stripe payment intent
     *
     * @param Request $request
     * @param int $orderId
     * @return \Illuminate\Http\JsonResponse
     */
    public function createPaymentIntent(Request $request, $orderId)
    {
        $order = Order::findOrFail($orderId);

        // Check if the order belongs to the authenticated user
        if ($order->user_id !== Auth::id()) {
            return response()->json(['error' => 'Unauthorized'], 403);
        }

        try {
            // Get the Stripe payment service
            $stripeMethod = PaymentMethod::where('name', 'stripe')
                ->where('is_active', true)
                ->first();

            if (!$stripeMethod) {
                return response()->json(['error' => 'Stripe payment method is not available'], 400);
            }

            $orderStatusService = app(OrderStatusService::class);
            $paymentService = new StripePaymentService($stripeMethod, $orderStatusService);

            // Initialize the payment
            $result = $paymentService->initializePayment($order);

            if (!$result['success']) {
                return response()->json(['error' => $result['error'] ?? 'Failed to create Stripe payment intent'], 500);
            }

            return response()->json([
                'clientSecret' => $result['client_secret'],
                'paymentIntentId' => $result['payment_intent_id']
            ]);
        } catch (\Exception $e) {
            Log::error('Stripe create payment intent error: ' . $e->getMessage(), [
                'order_id' => $orderId,
                'trace' => $e->getTraceAsString()
            ]);

            return response()->json(['error' => $e->getMessage()], 500);
        }
    }

    /**
     * Process payment after Stripe confirmation
     *
     * @param Request $request
     * @param int $orderId
     * @return \Illuminate\Http\JsonResponse
     */
    public function processPayment(Request $request, $orderId)
    {
        $request->validate([
            'payment_intent_id' => 'required|string',
        ]);

        $order = Order::findOrFail($orderId);

        // Check if the order belongs to the authenticated user
        if ($order->user_id !== Auth::id()) {
            return response()->json(['error' => 'Unauthorized'], 403);
        }

        try {
            // Get the Stripe payment service
            $stripeMethod = PaymentMethod::where('name', 'stripe')
                ->where('is_active', true)
                ->first();

            if (!$stripeMethod) {
                return response()->json(['error' => 'Stripe payment method is not available'], 400);
            }

            $orderStatusService = app(OrderStatusService::class);
            $paymentService = new StripePaymentService($stripeMethod, $orderStatusService);

            // Process the payment
            $paymentData = [
                'payment_intent_id' => $request->input('payment_intent_id'),
            ];

            $payment = $paymentService->processPayment($order, $paymentData);

            if ($payment->isCompleted()) {
                return response()->json([
                    'success' => true,
                    'message' => 'Payment completed successfully',
                    'redirect' => route('orders.show', $order->id)
                ]);
            } else {
                return response()->json([
                    'success' => false,
                    'message' => 'Payment verification failed: ' . ($payment->metadata['error'] ?? 'Unknown error'),
                ]);
            }
        } catch (\Exception $e) {
            Log::error('Stripe payment error: ' . $e->getMessage(), [
                'order_id' => $orderId,
                'trace' => $e->getTraceAsString()
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Payment processing failed: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Handle Stripe webhook events
     *
     * @param Request $request
     * @return \Illuminate\Http\Response
     */
    public function handleWebhook(Request $request)
    {
        $payload = $request->all();

        try {
            // Log the webhook event
            Log::info('Stripe webhook received', [
                'event_type' => $payload['type'] ?? 'unknown'
            ]);

            // Get the Stripe payment service
            $stripeMethod = PaymentMethod::where('name', 'stripe')->first();

            if (!$stripeMethod) {
                Log::error('Stripe payment method not found for webhook handling');
                return response('Stripe payment method not configured', 500);
            }

            $orderStatusService = app(OrderStatusService::class);
            $paymentService = new StripePaymentService($stripeMethod, $orderStatusService);

            // Handle the webhook
            $result = $paymentService->handleWebhook($payload);

            if ($result) {
                return response('Webhook processed successfully', 200);
            } else {
                return response('Failed to process webhook', 500);
            }
        } catch (\Exception $e) {
            Log::error('Error processing Stripe webhook: ' . $e->getMessage(), [
                'payload' => $payload,
                'trace' => $e->getTraceAsString()
            ]);

            return response('Webhook processing error: ' . $e->getMessage(), 500);
        }
    }
}
