<?php

namespace App\Http\Controllers\Webhooks;

use App\Http\Controllers\Controller;
use App\Models\PaymentMethod;
use App\Services\Payments\PayPalDirectPaymentService;
use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Illuminate\Support\Facades\Log;

class PayPalDirectIpnController extends Controller
{
    /**
     * Handle the incoming PayPal Direct IPN.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function handle(Request $request): Response
    {
        $ipnData = $request->all();
        
        // Log the IPN event
        Log::info('PayPal Direct IPN received', [
            'ipn_data' => $ipnData,
            'headers' => $request->headers->all()
        ]);

        try {
            // Get the PayPal Direct payment service
            $paypalDirectMethod = PaymentMethod::where('name', 'paypal_direct')->first();

            if (!$paypalDirectMethod) {
                Log::error('PayPal Direct payment method not found for IPN handling');
                return response('PayPal Direct payment method not configured', 500);
            }

            if (!$paypalDirectMethod->isActive()) {
                Log::error('PayPal Direct payment method is not active');
                return response('PayPal Direct payment method is not active', 500);
            }

            $paymentService = new PayPalDirectPaymentService($paypalDirectMethod);

            // Handle the IPN
            $result = $paymentService->handleWebhook($ipnData);

            if ($result) {
                Log::info('PayPal Direct IPN processed successfully', [
                    'transaction_id' => $ipnData['txn_id'] ?? 'unknown'
                ]);
                return response('OK', 200);
            } else {
                Log::warning('Failed to process PayPal Direct IPN', [
                    'transaction_id' => $ipnData['txn_id'] ?? 'unknown'
                ]);
                return response('FAILED', 500);
            }
        } catch (\Exception $e) {
            Log::error('Error processing PayPal Direct IPN: ' . $e->getMessage(), [
                'ipn_data' => $ipnData,
                'trace' => $e->getTraceAsString()
            ]);

            return response('ERROR: ' . $e->getMessage(), 500);
        }
    }
} 