<?php

namespace App\Http\Controllers\Writer;

use Carbon\Carbon;
use App\Models\User;
use Inertia\Inertia;

use App\Models\AdminSetting;
use Illuminate\Http\Request;
use App\Models\WriterTestQuestion;
use App\Models\WriterTestResponse;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use App\Notifications\AdminNotifiable;
use App\Models\WriterQualificationTest;
use App\Services\AdminNotificationService;
use Illuminate\Support\Facades\Notification;
use App\Notifications\WriterTestResultNotification;


class WriterTestController extends Controller
{
    /**
     * Show the writer test instructions page.
     */
    public function instructions()
    {
        $user = Auth::user();
        $writerProfile = $user->writerProfile;
        $adminSetting = AdminSetting::select('questions_test_duration')->first();
        $testDuration = $adminSetting ? $adminSetting->questions_test_duration : 15; // Default to 15 if not set

        // Check if the writer can take the test based on their current status
        $canTakeTest = $writerProfile->canTakeTest();

        // Check if they're in the waiting period after failing
        $waitingPeriod = false;
        $daysRemaining = 0;
        $retryDate = null;

        if (
            $writerProfile->qualificationTest &&
            $writerProfile->qualificationTest->questions_test_status === 'questions_failed' &&
            $writerProfile->qualificationTest->can_retry_after
        ) {

            $waitingPeriod = now()->lt($writerProfile->qualificationTest->can_retry_after);
            if ($waitingPeriod) {
                $daysRemaining = now()->diffInDays($writerProfile->qualificationTest->can_retry_after);
                $retryDate = Carbon::parse($writerProfile->qualificationTest->can_retry_after)->format('F j, Y');
            }
        }

        // Get the total number of questions in the pool
        $totalQuestionsInPool = WriterTestQuestion::where('is_active', true)->count();

        // Number of questions that will be in the test
        $testQuestionCount = 5;

        // Passing score percentage
        $passingPercentage = 80;

        // Get previous attempt data if available
        $previousAttempt = null;
        if (
            $writerProfile->qualificationTest &&
            $writerProfile->qualificationTest->questions_test_status === 'questions_failed'
        ) {
            $previousAttempt = [
                'score' => $writerProfile->qualificationTest->questions_score,
                'date' => $writerProfile->qualificationTest->questions_completed_at ?
                    Carbon::parse($writerProfile->qualificationTest->questions_completed_at)->format('F j, Y') : null,
            ];
        }

        return Inertia::render('Writer/Test/Instructions', [
            'canTakeTest' => $canTakeTest && !$waitingPeriod,
            'waitingPeriod' => $waitingPeriod,
            'daysRemaining' => $daysRemaining,
            'retryDate' => $retryDate,
            'totalQuestionsInPool' => $totalQuestionsInPool,
            'testQuestionCount' => $testQuestionCount,
            'passingPercentage' => $passingPercentage,
            'passingScore' => ceil(($testQuestionCount * $passingPercentage) / 100),
            'previousAttempt' => $previousAttempt,
            'testDuration' => $testDuration,
        ]);
    }

    /**
     * Start the writer test.
     */
    public function start(Request $request)
    {
        $user = Auth::user();
        $writerProfile = $user->writerProfile;

        // Check if the writer can take the test
        if (!$writerProfile->canTakeTest()) {
            return redirect()->route('writer.dashboard')
                ->with('error', 'You are not eligible to take the test at this time.');
        }

        // Begin a database transaction
        DB::beginTransaction();

        try {
            // Get test duration from admin settings
            $adminSetting = AdminSetting::first();
            $testDurationMinutes = $adminSetting ? $adminSetting->questions_test_duration : 15;

            // Get or create the qualification test record
            $qualificationTest = $writerProfile->qualificationTest;
            if (!$qualificationTest) {
                $qualificationTest = WriterQualificationTest::create([
                    'writer_profile_id' => $writerProfile->id,
                    'questions_test_status' => 'not_started',
                ]);
            }

            // Start the test session with security parameters
            $sessionToken = $qualificationTest->startTestSession(
                $testDurationMinutes,
                $request->ip(),
                $request->userAgent()
            );

            // Store the session token in the user's session
            session(['test_session_token' => $sessionToken]);

            // Clear any previous test responses for this user
            WriterTestResponse::where('user_id', $user->id)->delete();

            // Get random active questions
            // $questions = WriterTestQuestion::where('is_active', true)
            //     ->inRandomOrder()
            //     ->limit($testDurationMinutes === 15 ? 30 : 50) // Adjust question count based on duration
            //     ->get();

            // Get random active questions
            // Get random active questions (temporarily using 5 for testing)
            $questions = WriterTestQuestion::where('is_active', true)
                ->inRandomOrder()
                ->limit(5) // Temporarily set to 5 for testing
                ->get();



            // Store the question IDs in the session
            session(['test_questions' => $questions->pluck('id')->toArray()]);

            DB::commit();

            return redirect()->route('writer.test.take');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Failed to start test: ' . $e->getMessage(), [
                'user_id' => $user->id,
                'trace' => $e->getTraceAsString()
            ]);

            return redirect()->route('writer.test.instructions')
                ->with('error', 'Failed to start the test. Please try again.');
        }
    }

    /**
     * Show the test-taking interface.
     */
    public function take(Request $request)
    {
        $user = Auth::user();
        $writerProfile = $user->writerProfile;
        $qualificationTest = $writerProfile->qualificationTest;

        // Check if the writer has a qualification test record
        if (!$qualificationTest) {
            return redirect()->route('writer.dashboard')
                ->with('error', 'No test session found. Please start a new test.');
        }

        // Check if the writer is currently taking the test
        if ($qualificationTest->questions_test_status !== 'questions_pending') {
            return redirect()->route('writer.dashboard')
                ->with('error', 'You are not currently taking the test.');
        }

        // Verify the test is still within the time window
        if (!$qualificationTest->isWithinTimeWindow()) {
            // Auto-submit the test if time has expired
            return $this->submit(new Request(['answers' => []]));
        }

        // Get the question IDs from the session
        $questionIds = session('test_questions', []);
        $sessionToken = session('test_session_token');

        if (empty($questionIds) || !$sessionToken) {
            return redirect()->route('writer.test.instructions')
                ->with('error', 'Test session expired. Please start again.');
        }

        // Record activity for this page view
        $qualificationTest->recordActivity($sessionToken, $request->ip());

        // Calculate time remaining
        $timeRemaining = $qualificationTest->getRemainingTime();

        // Get the questions with their options (but not the correct answers)
        $questions = WriterTestQuestion::whereIn('id', $questionIds)
            ->get()
            ->map(function ($question) {
                return [
                    'id' => $question->id,
                    'question' => $question->question,
                    'options' => [
                        'a' => $question->option_a,
                        'b' => $question->option_b,
                        'c' => $question->option_c,
                        'd' => $question->option_d,
                    ],
                    'category' => $question->category,
                ];
            });

        return Inertia::render('Writer/Test/Take', [
            'questions' => $questions,
            'totalQuestions' => count($questions),
            'timeRemaining' => $timeRemaining,
            'testStartedAt' => $qualificationTest->started_at->timestamp,
            'sessionToken' => $sessionToken,
        ]);
    }

    /**
     * Record activity during the test to track user engagement.
     */
    public function recordActivity(Request $request)
    {
        $user = Auth::user();
        $writerProfile = $user->writerProfile;
        $qualificationTest = $writerProfile->qualificationTest;

        if (!$qualificationTest || $qualificationTest->questions_test_status !== 'questions_pending') {
            return response()->json(['error' => 'No active test found'], 400);
        }

        $sessionToken = $request->input('sessionToken', session('test_session_token'));

        if (!$sessionToken) {
            return response()->json(['error' => 'Invalid session'], 400);
        }

        $success = $qualificationTest->recordActivity($sessionToken, $request->ip());

        if (!$success) {
            return response()->json(['error' => 'Failed to record activity'], 400);
        }

        return response()->json([
            'success' => true,
            'timeRemaining' => $qualificationTest->getRemainingTime(),
        ]);
    }

    /**
     * Check remaining time for the test.
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function checkTime(Request $request)
    {
        $user = Auth::user();
        $writerProfile = $user->writerProfile;
        $qualificationTest = $writerProfile->qualificationTest;

        if (!$qualificationTest || $qualificationTest->questions_test_status !== 'questions_pending') {
            return response()->json(['error' => 'No active test found'], 400);
        }

        $sessionToken = $request->input('sessionToken', session('test_session_token'));

        if (!$sessionToken) {
            return response()->json(['error' => 'Invalid session'], 400);
        }

        // Record activity
        $qualificationTest->recordActivity($sessionToken, $request->ip());

        // Check if test has expired
        if ($qualificationTest->hasExpiredWithoutSubmission()) {
            return response()->json([
                'isExpired' => true,
                'message' => 'Your test time has expired.',
                'redirect' => route('writer.test.results')
            ]);
        }

        $remainingTime = $qualificationTest->getRemainingTime();
        $isExpired = $remainingTime <= 0;

        return response()->json([
            'remainingTime' => $remainingTime,
            'isExpired' => $isExpired,
            'formattedTime' => $this->formatTime($remainingTime)
        ]);
    }

    /**
     * Auto-submit the test when time expires (client-side trigger).
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function autoSubmit(Request $request)
    {
        $user = Auth::user();
        $writerProfile = $user->writerProfile;
        $qualificationTest = $writerProfile->qualificationTest;

        if (!$qualificationTest || $qualificationTest->questions_test_status !== 'questions_pending') {
            return response()->json(['error' => 'No active test found'], 400);
        }

        // Verify session token
        $sessionToken = $request->input('sessionToken', session('test_session_token'));
        if (!$sessionToken || $sessionToken !== $qualificationTest->session_token) {
            return response()->json(['error' => 'Invalid session'], 403);
        }

        // Check if the test has actually expired
        if (!$qualificationTest->hasExpiredWithoutSubmission()) {
            return response()->json(['error' => 'Test is not expired'], 400);
        }

        // Get any answers that were submitted
        $answers = $request->input('answers', []);

        // Process the submission
        if (!empty($answers)) {
            // Save the answers first
            foreach ($answers as $questionId => $answer) {
                $question = WriterTestQuestion::find($questionId);
                if ($question) {
                    WriterTestResponse::create([
                        'user_id' => $user->id,
                        'question_id' => $questionId,
                        'selected_answer' => $answer,
                        'is_correct' => $answer === $question->correct_answer,
                    ]);
                }
            }

            $qualificationTest->processPartialSubmission($answers);
        } else {
            $qualificationTest->markAsExpired('Auto-processed by client - no answers submitted');
        }

        // Clear session
        session()->forget(['test_questions', 'test_session_token']);

        return response()->json([
            'success' => true,
            'message' => 'Test auto-submitted successfully',
            'redirect' => route('writer.test.results')
        ]);
    }

    /**
     * Submit the writer test answers.
     */
    public function submit(Request $request)
    {
        $user = Auth::user();
        $writerProfile = $user->writerProfile;
        $qualificationTest = $writerProfile->qualificationTest;

        Log::info('Test submission started', ['user_id' => $user->id]);

        // Check if the writer has a qualification test record
        if (!$qualificationTest) {
            Log::error('No qualification test record found');
            return redirect()->route('writer.dashboard')
                ->with('error', 'No test session found. Please start a new test.');
        }

        // Get the question IDs from the session
        $questionIds = session('test_questions', []);
        $sessionToken = session('test_session_token');

        if (empty($questionIds)) {
            Log::error('No test questions found in session');
            return redirect()->route('writer.test.instructions')
                ->with('error', 'Test session expired. Please start again.');
        }

        // Check if the writer is currently taking the test
        if ($qualificationTest->questions_test_status !== 'questions_pending') {
            Log::error('User not in pending test status', ['status' => $qualificationTest->questions_test_status]);
            return redirect()->route('writer.dashboard')
                ->with('error', 'You are not currently taking the test.');
        }

        // Verify submission timing
        if (!$qualificationTest->verifySubmissionTiming($sessionToken)) {
            Log::warning('Submission timing verification failed', [
                'user_id' => $user->id,
                'is_time_violation' => $qualificationTest->is_time_violation,
                'violation_notes' => $qualificationTest->time_violation_notes
            ]);

            // We'll still process the submission but it will be flagged
        }

        try {
            // Validate the request - using array syntax for answers
            $validated = $request->validate([
                'answers' => 'required|array',
                'answers.*' => 'required|in:a,b,c,d',
            ]);

            Log::info('Validated answers:', $validated);

            DB::beginTransaction();

            $questions = WriterTestQuestion::whereIn('id', $questionIds)->get();
            $correctCount = 0;
            $totalQuestions = count($questions);
            $categoryResults = [];

            if ($totalQuestions === 0) {
                Log::error('No questions found for the test');
                DB::rollBack();
                return redirect()->route('writer.test.instructions')
                    ->with('error', 'No questions found for the test. Please try again.');
            }

            foreach ($questions as $question) {
                // Check if the question ID exists in the answers array
                $selectedAnswer = $validated['answers'][$question->id] ?? null;

                Log::info('Processing answer:', [
                    'question_id' => $question->id,
                    'selected_answer' => $selectedAnswer,
                    'correct_answer' => $question->correct_answer
                ]);

                // Initialize category in results if not exists
                if (!isset($categoryResults[$question->category])) {
                    $categoryResults[$question->category] = [
                        'total' => 0,
                        'correct' => 0
                    ];
                }

                $categoryResults[$question->category]['total']++;

                // Only create a response record if the user selected an answer
                if ($selectedAnswer) {
                    $isCorrect = $selectedAnswer === $question->correct_answer;

                    if ($isCorrect) {
                        $correctCount++;
                        $categoryResults[$question->category]['correct']++;
                    }

                    WriterTestResponse::create([
                        'user_id' => $user->id,
                        'question_id' => $question->id,
                        'selected_answer' => $selectedAnswer,
                        'is_correct' => $isCorrect,
                    ]);
                } else {
                    // For unanswered questions, we'll just log them but not create a record
                    Log::info('Skipping unanswered question', [
                        'question_id' => $question->id
                    ]);
                }
            }

            // Calculate score based on correct answers out of total questions
            $scorePercentage = ($correctCount / $totalQuestions) * 100;
            $passed = $scorePercentage >= 90;

            // Store category results in session for the results page
            session(['category_results' => $categoryResults]);

            // Update the qualification test record
            $qualificationTest->update([
                'questions_score' => $scorePercentage,
                'questions_completed_at' => now(),
                'questions_test_status' => $passed ? 'passed' : 'questions_failed',
                'last_failed_at' => $passed ? null : now(),
                'can_retry_after' => $passed ? null : now()->addMonths(3),
                'auto_processed_at' => null, // This was a manual submission, not auto-processed
            ]);

            // Notify the writer about their test results

            /** @var \Illuminate\Notifications\Notifiable $user */
            $user->notify(new WriterTestResultNotification(
                $user,
                $scorePercentage,
                $passed,
                $correctCount,
                $totalQuestions,
                false // This is not an admin notification
            ));

            // Notify all admins of the test results
            AdminNotificationService::notifyAllAdmins(
                new WriterTestResultNotification(
                    $user,
                    $scorePercentage,
                    $passed,
                    $correctCount,
                    $totalQuestions,
                    true // isAdminNotification
                )
            );

            // Clear session
            session()->forget(['test_questions', 'test_session_token']);

            DB::commit();

            Log::info('Test submission completed', [
                'user_id' => $user->id,
                'score' => $scorePercentage,
                'passed' => $passed,
                'is_time_violation' => $qualificationTest->is_time_violation
            ]);

            // If there was a time violation, we'll note it in the flash message
            $message = $passed ?
                'Congratulations! You passed with ' . round($scorePercentage, 1) . '%.' :
                'Test completed. Score: ' . round($scorePercentage, 1) . '%.';

            if ($qualificationTest->is_time_violation) {
                $message .= ' Note: Some timing irregularities were detected during your test.';
            }

            return redirect()->route($passed ? 'writer.essay.instructions' : 'writer.test.results')
                ->with('message', $message);
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Test submission failed:', ['error' => $e->getMessage(), 'trace' => $e->getTraceAsString()]);

            return redirect()->route('writer.test.take')
                ->with('error', 'Failed to submit test: ' . $e->getMessage());
        }
    }

    /**
     * Show the test results.
     */
    public function results()
    {
        $user = Auth::user();
        $writerProfile = $user->writerProfile;
        $qualificationTest = $writerProfile->qualificationTest;

        if (!$qualificationTest) {
            return redirect()->route('writer.dashboard')
                ->with('error', 'No test results found.');
        }

        // Get the latest test responses
        $responses = WriterTestResponse::where('user_id', $user->id)
            ->with('question')
            ->orderBy('created_at', 'desc')
            ->get()
            ->groupBy('created_at->date')
            ->first();

        if (!$responses) {
            return redirect()->route('writer.dashboard')
                ->with('error', 'No test results found.');
        }

        // Calculate statistics
        $totalQuestions = $responses->count();
        $correctAnswers = $responses->where('is_correct', true)->count();
        $scorePercentage = ($correctAnswers / $totalQuestions) * 100;
        $passed = $scorePercentage >= 90;

        // Group questions by category for detailed feedback
        $categorizedResponses = $responses->groupBy(function ($response) {
            return $response->question->category;
        })->map(function ($categoryResponses) {
            $total = $categoryResponses->count();
            $correct = $categoryResponses->where('is_correct', true)->count();

            return [
                'total' => $total,
                'correct' => $correct,
                'percentage' => ($correct / $total) * 100,
            ];
        });

        return Inertia::render('Writer/Test/Results', [
            'score' => [
                'total' => $totalQuestions,
                'correct' => $correctAnswers,
                'percentage' => $scorePercentage,
                'passed' => $passed,
            ],
            'categorizedResults' => $categorizedResponses,
            'canRetryAfter' => $qualificationTest->can_retry_after ? Carbon::parse($qualificationTest->can_retry_after)->format('F j, Y') : null,
            'daysUntilRetry' => $qualificationTest->can_retry_after ? now()->diffInDays($qualificationTest->can_retry_after) : 0,
            'timeViolation' => $qualificationTest->is_time_violation,
            'timeViolationNotes' => $qualificationTest->time_violation_notes,
            'autoProcessed' => !is_null($qualificationTest->auto_processed_at), // Add flag for auto-processed tests
        ]);
    }

    /**
     * Format time in seconds to a readable format.
     *
     * @param  int  $seconds
     * @return string
     */
    private function formatTime($seconds)
    {
        $hours = floor($seconds / 3600);
        $minutes = floor(($seconds % 3600) / 60);
        $secs = $seconds % 60;

        return sprintf('%02d:%02d:%02d', $hours, $minutes, $secs);
    }
}
