<?php

namespace App\Http\Controllers;

use Illuminate\Support\Facades\Log;
use Carbon\Carbon;
use App\Models\Bid;
use App\Models\Order;
use App\Models\User;
use App\Notifications\ClientBidPlacedNotification;
use App\Notifications\AdminBidPlacedNotification;
use App\Services\UrgentOrderNotificationService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Notification;

class WriterController extends Controller
{
    protected UrgentOrderNotificationService $urgentOrderService;

    public function __construct(UrgentOrderNotificationService $urgentOrderService)
    {
        $this->urgentOrderService = $urgentOrderService;
    }

    public function dashboard()
    {
        $user = Auth::user();
        $writerProfile = $user->writerProfile;

        // Get writer qualification status using the relationship methods
        $qualificationStatus = $writerProfile ? $writerProfile->getQualificationStatus() : 'not_started';
        $isFullyQualified = $writerProfile->isApproved();

        // Get profile completion data - don't modify the model directly
        $profileCompletionPercentage = $writerProfile->profile_completion_percentage;
        $missingRequiredFields = $writerProfile->getMissingRequiredFields();






        // Get specific test statuses
        $questionsStatus = 'not_started';
        $essayStatus = 'not_started';

        if ($writerProfile && $writerProfile->qualificationTest) {
            $questionsStatus = $writerProfile->qualificationTest->questions_test_status;

            // Only check essay status if questions test is passed
            if ($questionsStatus === 'passed' && $writerProfile->essayTest) {
                $essayStatus = $writerProfile->essayTest->essay_test_status;
            }
        }

        // Check if essay test is expired but no retry date is set
        if (
            $writerProfile && $writerProfile->essayTest &&
            $writerProfile->essayTest->essay_test_status === 'essay_expired' &&
            !$writerProfile->essayTest->can_retry_after
        ) {

            // Set retry date to 3 months from expiration date or now
            $expiredAt = $writerProfile->essayTest->essay_completed_at ?? now();
            $retryDate = $expiredAt->copy()->addMonths(3);

            // Update the essay test with the retry date
            $writerProfile->essayTest->update([
                'can_retry_after' => $retryDate,
                'last_failed_at' => $expiredAt
            ]);

            Log::info('Set retry date for expired essay test ID: ' . $writerProfile->essayTest->id .
                ' to ' . $retryDate->format('Y-m-d H:i:s'));
        }

        // Calculate days until retry if applicable
        $daysUntilRetry = null;
        $retryDate = null;

        if ($writerProfile) {
            $qualTestRetryDate = null;
            $essayTestRetryDate = null;

            // Get retry date from qualification test if it exists
            if ($writerProfile->qualificationTest && $writerProfile->qualificationTest->can_retry_after) {
                $qualTestRetryDate = $writerProfile->qualificationTest->can_retry_after;
            }

            // Get retry date from essay test if it exists
            if ($writerProfile->essayTest && $writerProfile->essayTest->can_retry_after) {
                $essayTestRetryDate = $writerProfile->essayTest->can_retry_after;
                Log::debug('Essay test retry date retrieved: ' . $essayTestRetryDate);
            }

            // Use the later of the two dates, or whichever one exists
            $effectiveRetryDate = null;
            if ($qualTestRetryDate && $essayTestRetryDate) {
                $effectiveRetryDate = $qualTestRetryDate->gt($essayTestRetryDate) ?
                    $qualTestRetryDate : $essayTestRetryDate;
            } elseif ($qualTestRetryDate) {
                $effectiveRetryDate = $qualTestRetryDate;
            } elseif ($essayTestRetryDate) {
                $effectiveRetryDate = $essayTestRetryDate;
            }

            // Calculate days until retry if we have a date
            if ($effectiveRetryDate) {
                $daysUntilRetry = now()->diffInDays($effectiveRetryDate, false);
                $retryDate = $effectiveRetryDate->format('F j, Y');
                Log::debug('Effective retry date: ' . $retryDate . ', Days until retry: ' . $daysUntilRetry);
            }
        }

        // Get recent bids if writer is fully approved
        $recentBids = collect();
        $activeOrders = collect();

        if ($writerProfile && $writerProfile->isApproved()) {
            $recentBids = Bid::where('writer_id', $user->id)
                ->with(['order' => function ($query) {
                    $query->orderBy('dateposted', 'desc');
                }])
                ->orderBy('id', 'desc') // or orderBy('created_at', 'desc') if bids table has timestamps
                ->take(5)
                ->get();

            $activeOrders = Order::where('writer_id', $user->id)
                ->where('order_status', '!=', 'completed')
                ->orderBy('dateposted', 'desc')
                ->take(5)
                ->get();
        }

        // Get available orders count for approved writers
        $availableOrdersCount = 0;
        if ($writerProfile && $writerProfile->isApproved()) {
            $availableOrdersCount = Order::where('writer_id', null)
                ->where('payment_status', 'paid')
                ->where('order_status', 'bidding')
                ->where('writer_deadline', '>', now())
                ->count();
        }

        // Check if there's a partial submission for an expired essay test
        $hasPartialSubmission = false;
        if (
            $writerProfile && $writerProfile->essayTest &&
            $writerProfile->essayTest->essay_test_status === 'essay_expired' &&
            $writerProfile->essayTest->hasUploadedFile()
        ) {
            $hasPartialSubmission = true;
        }

        // Get writer profile data if it exists
        $profileData = null;
        if ($writerProfile) {
            $profileData = [
                'questions_score' => $writerProfile->qualificationTest ? $writerProfile->qualificationTest->questions_score : null,
                'questions_completed_at' => $writerProfile->qualificationTest && $writerProfile->qualificationTest->questions_completed_at ?
                    $writerProfile->qualificationTest->questions_completed_at->format('F j, Y') : null,
                'essay_completed_at' => $writerProfile->essayTest && $writerProfile->essayTest->essay_completed_at ?
                    $writerProfile->essayTest->essay_completed_at->format('F j, Y') : null,
                'essay_score' => $writerProfile->essayTest ? $writerProfile->essayTest->essay_score : null,
                'essay_feedback' => $writerProfile->essayTest ? $writerProfile->essayTest->essay_feedback : null,
                'has_partial_submission' => $hasPartialSubmission,
                'file_name' => $hasPartialSubmission && $writerProfile->essayTest ?
                    $writerProfile->essayTest->original_filename : null,
            ];
        }

        // Ensure daysUntilRetry is at least 1 for expired tests to prevent immediate retry
        if (
            $writerProfile && $writerProfile->essayTest &&
            $writerProfile->essayTest->essay_test_status === 'essay_expired' &&
            ($daysUntilRetry === null || $daysUntilRetry <= 0)
        ) {

            $daysUntilRetry = 90; // Default to 90 days (3 months)
            $retryDate = now()->addDays(90)->format('F j, Y');
            Log::warning('Forcing daysUntilRetry to 90 for expired essay test ID: ' .
                $writerProfile->essayTest->id);
        }

        // Get urgent orders for writer dashboard
        $urgentOrders = $this->urgentOrderService->getUrgentOrdersForUser(auth()->user());
        $urgentOrdersCount = $this->urgentOrderService->getUrgentOrdersCount(auth()->user());

        return inertia('Writer/Dashboard', [
            'testStatus' => $qualificationStatus,
            'questionsStatus' => $questionsStatus,
            'essayStatus' => $essayStatus,
            'daysUntilRetry' => $daysUntilRetry,
            'retryDate' => $retryDate,
            'recentBids' => $recentBids,
            'activeOrders' => $activeOrders,
            'availableOrdersCount' => $availableOrdersCount,
            'writerProfile' => $profileData,
            'hasTestInProgress' => $writerProfile ? $writerProfile->hasTestInProgress() : false,
            'profileCompletionPercentage' => $profileCompletionPercentage,
            'missingRequiredFields' => $missingRequiredFields,
            'isFullyQualified' => $writerProfile ? $writerProfile->isApproved() : false,
            'urgentOrders' => $urgentOrders,
            'urgentOrdersCount' => $urgentOrdersCount,
        ]);
    }



    public function availableOrders()
    {
        // Check if writer is approved
        $user = Auth::user();
        $writerProfile = $user->writerProfile;

        if (!$writerProfile || !$writerProfile->isApproved()) {
            return redirect()->route('writer.dashboard')
                ->with('error', 'You must complete the qualification process before viewing available orders.');
        }

        $orders = Order::where('writer_id', null)
            ->where('payment_status', 'paid')
            ->where('order_status', 'bidding')
            ->where('writer_deadline', '>', now()) // Ensure the deadline is in the future
            ->latest('dateposted')
            ->paginate(10);

        return inertia('Writer/Index', [
            'orders' => $orders,
        ]);
    }

    public function assignedOrders()
    {
        // Check if writer is approved
        $user = Auth::user();
        $writerProfile = $user->writerProfile;

        if (!$writerProfile || !$writerProfile->isApproved()) {
            return redirect()->route('writer.dashboard')
                ->with('error', 'You must complete the qualification process before viewing your assignments.');
        }

        // Get orders assigned to this writer that are still active
        $assignedOrders = Order::where('writer_id', $user->id)
            ->whereIn('order_status', [
                Order::ORDER_STATUS_ASSIGNED,
                Order::ORDER_STATUS_IN_PROGRESS,
                Order::ORDER_STATUS_SUBMITTED,
                Order::ORDER_STATUS_UNDER_REVIEW,
                Order::ORDER_STATUS_REVISION_REQUESTED
            ])
            ->where('writer_deadline', '>', now()) // Only show orders that haven't expired
            ->with(['orderFiles', 'user', 'writerInfractions'])
            ->latest('writer_assigned_at')
            ->paginate(10);

        // Get expired orders for reference (but don't show them as active)
        $expiredOrders = Order::where('writer_id', $user->id)
            ->whereIn('order_status', [
                Order::ORDER_STATUS_ASSIGNED,
                Order::ORDER_STATUS_IN_PROGRESS,
                Order::ORDER_STATUS_SUBMITTED,
                Order::ORDER_STATUS_UNDER_REVIEW,
                Order::ORDER_STATUS_REVISION_REQUESTED
            ])
            ->where('writer_deadline', '<=', now())
            ->with(['orderFiles', 'user', 'writerInfractions'])
            ->latest('writer_assigned_at')
            ->get();

        // Get completed and approved orders
        $completedOrders = Order::where('writer_id', $user->id)
            ->whereIn('order_status', [
                Order::ORDER_STATUS_APPROVED,
                Order::ORDER_STATUS_CANCELLED
            ])
            ->with(['orderFiles', 'user', 'writerInfractions'])
            ->latest('writer_assigned_at')
            ->get();

        return inertia('Writer/Assignments', [
            'assignedOrders' => $assignedOrders,
            'expiredOrders' => $expiredOrders,
            'completedOrders' => $completedOrders,
        ]);
    }

    public function orderHistory()
    {
        // Check if writer is approved
        $user = Auth::user();
        $writerProfile = $user->writerProfile;

        if (!$writerProfile || !$writerProfile->isApproved()) {
            return redirect()->route('writer.dashboard')
                ->with('error', 'You must complete the qualification process before viewing order history.');
        }

        // Get all orders assigned to this writer with pagination
        $orders = Order::where('writer_id', $user->id)
            ->with(['orderFiles', 'user', 'writerInfractions'])
            ->latest('writer_assigned_at')
            ->paginate(20);

        return inertia('Writer/OrderHistory', [
            'orders' => $orders,
        ]);
    }

    public function show($id)
    {
        // Check if writer is approved
        $user = Auth::user();
        $writerProfile = $user->writerProfile;

        if (!$writerProfile || !$writerProfile->isApproved()) {
            return redirect()->route('writer.dashboard')
                ->with('error', 'You must complete the qualification process before viewing order details.');
        }

        $isAdmin = in_array(Auth::user()->user_type, ['admin', 'super_admin']);
        $order = Order::with(['orderFiles', 'writerInfractions'])->findOrFail($id);

        // Check if the writer has already placed a bid for this order
        $existingBid = Bid::where('order_id', $id)
            ->where('writer_id', Auth::id())
            ->first();

        return inertia('Writer/Show', [
            'order' => $order,
            'isAdmin' => $isAdmin,
            'existingBid' => $existingBid, // Pass the existing bid to the frontend
        ]);
    }

    public function placeBid(Request $request, $orderId)
    {
        // Check if writer is approved
        $user = Auth::user();
        $writerProfile = $user->writerProfile;

        if (!$writerProfile || !$writerProfile->isApproved()) {
            return response()->json(['error' => 'You must complete the qualification process before bidding on orders.'], 403);
        }

        $request->validate([
            'bid_amount' => 'required|numeric|min:0.01|max:' . $request->writer_amount,
            'completion_time' => 'required|date|after_or_equal:' . now()->toDateTimeString() . '|before_or_equal:' . $request->writer_deadline,
            'comments' => 'nullable|string|max:500',
        ]);

        $order = Order::findOrFail($orderId);

        // Ensure the order is still open for bidding
        if ($order->order_status !== 'bidding' || $order->payment_status !== 'paid' || $order->writer_deadline <= now()) {
            return response()->json(['error' => 'This order is no longer available for bidding.'], 400);
        }

        // Save the bid
        $bid = Bid::create([
            'order_id' => $orderId,
            'writer_id' => Auth::id(),
            'bid_amount' => $request->bid_amount,
            'completion_time' => $request->completion_time,
            'comments' => $request->comments,
            'status' => 'pending', // Default status for a new bid
        ]);

        // Send notifications
        try {
            // Notify the client
            $order->user->notify(new ClientBidPlacedNotification($order, $bid));
            
            // Notify admins (including super admins)
            $admins = User::whereIn('user_type', ['admin', 'super_admin'])->get();
            Notification::send($admins, new AdminBidPlacedNotification($order, $bid));
            
        } catch (\Exception $e) {
            // Log the error but don't fail the bid creation
            Log::error('Failed to send bid notifications', [
                'error' => $e->getMessage(),
                'order_id' => $orderId,
                'bid_id' => $bid->id
            ]);
        }

        // Update writer's last bid timestamp
        if ($writerProfile) {
            $writerProfile->update([
                'last_bid_at' => now(),
                'last_active_at' => now(),
            ]);
        }

        return response()->json(['message' => 'Bid placed successfully!', 'bid' => $bid], 201);
    }

    public function earnings()
    {
        // Check if writer is approved
        $user = Auth::user();
        $writerProfile = $user->writerProfile;

        if (!$writerProfile || !$writerProfile->isApproved()) {
            return redirect()->route('writer.dashboard')
                ->with('error', 'You must complete the qualification process before viewing earnings.');
        }

        // Get earnings data for the writer
        $earningsData = $this->calculateEarningsData($user->id);

        return inertia('Writer/Earnings', [
            'earningsData' => $earningsData,
        ]);
    }

    private function calculateEarningsData($writerId)
    {
        $now = now();
        $currentMonth = $now->format('Y-m');
        $previousMonth = $now->copy()->subMonth()->format('Y-m');
        $currentYear = $now->year;

        // Get all completed orders for this writer
        $completedOrders = Order::where('writer_id', $writerId)
            ->whereIn('order_status', ['approved', 'completed'])
            ->whereIn('writer_payment_status', ['available', 'withdrawn'])
            ->get();

        // Calculate lifetime earnings
        $lifetimeEarnings = $completedOrders->sum('writer_amount');

        // Calculate current month earnings
        $currentMonthEarnings = $completedOrders
            ->filter(function ($order) use ($currentMonth) {
                return $order->writer_assigned_at && 
                       $order->writer_assigned_at->format('Y-m') === $currentMonth;
            })
            ->sum('writer_amount');

        // Calculate previous month earnings
        $previousMonthEarnings = $completedOrders
            ->filter(function ($order) use ($previousMonth) {
                return $order->writer_assigned_at && 
                       $order->writer_assigned_at->format('Y-m') === $previousMonth;
            })
            ->sum('writer_amount');

        // Calculate year-to-date earnings
        $yearToDateEarnings = $completedOrders
            ->filter(function ($order) use ($currentYear) {
                return $order->writer_assigned_at && 
                       $order->writer_assigned_at->year === $currentYear;
            })
            ->sum('writer_amount');

        // Calculate order counts
        $totalOrdersCompleted = $completedOrders->count();
        $currentMonthOrders = $completedOrders
            ->filter(function ($order) use ($currentMonth) {
                return $order->writer_assigned_at && 
                       $order->writer_assigned_at->format('Y-m') === $currentMonth;
            })
            ->count();
        $previousMonthOrders = $completedOrders
            ->filter(function ($order) use ($previousMonth) {
                return $order->writer_assigned_at && 
                       $order->writer_assigned_at->format('Y-m') === $previousMonth;
            })
            ->count();
        $yearToDateOrders = $completedOrders
            ->filter(function ($order) use ($currentYear) {
                return $order->writer_assigned_at && 
                       $order->writer_assigned_at->year === $currentYear;
            })
            ->count();

        // Calculate average earnings per order
        $averageEarningsPerOrder = $totalOrdersCompleted > 0 ? $lifetimeEarnings / $totalOrdersCompleted : 0;

        // Get highest single order payment
        $highestSingleOrderPayment = $completedOrders->max('writer_amount') ?? 0;

        return [
            'lifetime_earnings' => round($lifetimeEarnings, 2),
            'current_month_earnings' => round($currentMonthEarnings, 2),
            'previous_month_earnings' => round($previousMonthEarnings, 2),
            'year_to_date_earnings' => round($yearToDateEarnings, 2),
            'average_earnings_per_order' => round($averageEarningsPerOrder, 2),
            'highest_single_order_payment' => round($highestSingleOrderPayment, 2),
            'total_orders_completed' => $totalOrdersCompleted,
            'current_month_orders' => $currentMonthOrders,
            'previous_month_orders' => $previousMonthOrders,
            'year_to_date_orders' => $yearToDateOrders,
        ];
    }
}
