<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Services\WriterPaymentService;
use App\Models\WriterWithdrawalRequest;
use App\Models\WriterProfile;
use App\Models\WriterPaymentMethod;
use App\Models\WriterPaymentAccount;
use App\Models\WriterPaymentRecord;
use Illuminate\Support\Facades\Auth;
use Inertia\Inertia;

class WriterPaymentController extends Controller
{
    protected $writerPaymentService;

    public function __construct(WriterPaymentService $writerPaymentService)
    {
        $this->writerPaymentService = $writerPaymentService;
        $this->middleware('auth');
    }

    /**
     * Get writer payment summary for the authenticated writer.
     */
    public function getSummary()
    {
        $writerId = Auth::id();
        $summary = $this->writerPaymentService->getWriterPaymentSummary($writerId);
        
        return response()->json([
            'success' => true,
            'data' => $summary
        ]);
    }

    /**
     * Get pending orders for the authenticated writer.
     */
    public function getPendingOrders()
    {
        $writerId = Auth::id();
        $orders = $this->writerPaymentService->getPendingOrders($writerId);
        
        return response()->json([
            'success' => true,
            'data' => $orders
        ]);
    }

    /**
     * Get available orders for the authenticated writer.
     */
    public function getAvailableOrders()
    {
        $writerId = Auth::id();
        $orders = $this->writerPaymentService->getAvailableOrders($writerId);
        
        return response()->json([
            'success' => true,
            'data' => $orders
        ]);
    }

    /**
     * Show pending orders page.
     */
    public function showPendingOrders()
    {
        $writerId = Auth::id();
        $orders = $this->writerPaymentService->getPendingOrders($writerId);
        $summary = $this->writerPaymentService->getWriterPaymentSummary($writerId);
        
        return Inertia::render('Writer/Payment/PendingOrders', [
            'orders' => $orders,
            'summary' => $summary
        ]);
    }

    /**
     * Show available orders page.
     */
    public function showAvailableOrders()
    {
        $writerId = Auth::id();
        $orders = $this->writerPaymentService->getAvailableOrders($writerId);
        $summary = $this->writerPaymentService->getWriterPaymentSummary($writerId);
        
        return Inertia::render('Writer/Payment/AvailableOrders', [
            'orders' => $orders,
            'summary' => $summary
        ]);
    }

    /**
     * Check if writer can withdraw.
     */
    public function checkWithdrawalEligibility()
    {
        $writerId = Auth::id();
        $eligibility = $this->writerPaymentService->canWriterWithdraw($writerId);
        
        return response()->json([
            'success' => true,
            'data' => $eligibility
        ]);
    }

    /**
     * Create a withdrawal request.
     */
    public function createWithdrawalRequest(Request $request)
    {
        $request->validate([
            'amount' => 'required|numeric|min:0.01',
            'payment_method' => 'required|string|in:paypal,bank_transfer',
            'payment_details' => 'required|array',
        ]);

        try {
            $writerId = Auth::id();
            $withdrawalRequest = $this->writerPaymentService->createWithdrawalRequest(
                $writerId,
                $request->amount,
                $request->payment_method,
                $request->payment_details
            );

            // Check if request expects JSON response
            if ($request->expectsJson() || $request->header('Accept') === 'application/json') {
                return response()->json([
                    'success' => true,
                    'message' => 'Withdrawal request submitted successfully! It will be reviewed by our admin team.',
                    'data' => $withdrawalRequest
                ]);
            }

            // Redirect back to dashboard with success message
            return redirect()->route('writer.payment.dashboard')->with('success', 'Withdrawal request submitted successfully! It will be reviewed by our admin team.');

        } catch (\Exception $e) {
            // Check if request expects JSON response
            if ($request->expectsJson() || $request->header('Accept') === 'application/json') {
                return response()->json([
                    'success' => false,
                    'message' => 'Withdrawal request failed: ' . $e->getMessage()
                ], 400);
            }

            // Get the current page to redirect back with error
            $referer = $request->headers->get('referer');
            $redirectUrl = $referer ?: route('writer.payment.dashboard');
            
            // Redirect back to the same page with error message
            return redirect($redirectUrl)->with('error', 'Withdrawal request failed: ' . $e->getMessage());
        }
    }

    /**
     * Get withdrawal requests for the authenticated writer.
     */
    public function getWithdrawalRequests()
    {
        $writerId = Auth::id();
        $requests = WriterWithdrawalRequest::where('writer_id', $writerId)
            ->orderBy('id', 'desc')
            ->get();
        
        return response()->json([
            'success' => true,
            'data' => $requests
        ]);
    }

    /**
     * Show writer payment dashboard.
     */
    public function dashboard()
    {
        $writerId = Auth::id();
        $summary = $this->writerPaymentService->getWriterPaymentSummary($writerId);
        $eligibility = $this->writerPaymentService->canWriterWithdraw($writerId);
        
        // Map the summary fields to match what the frontend expects
        $mappedSummary = [
            'totalEarnings' => $summary['total_earnings'] ?? 0,
            'availableForWithdrawal' => $summary['available_amount'] ?? 0,
            'pendingPayments' => $summary['pending_amount'] ?? 0,
            'monthlyEarnings' => $summary['monthly_earnings'] ?? 0,
        ];
        
        $recentWithdrawals = WriterWithdrawalRequest::where('writer_id', $writerId)
            ->orderBy('id', 'desc')
            ->limit(5)
            ->get();
        
        return Inertia::render('Writer/Payment/Dashboard', [
            'summary' => $mappedSummary,
            'eligibility' => $eligibility,
            'recentWithdrawals' => $recentWithdrawals
        ]);
    }

    /**
     * Show payment history page.
     */
    public function paymentHistory()
    {
        $writerId = Auth::id();
        $summary = $this->writerPaymentService->getWriterPaymentSummary($writerId);
        
        // Map the summary fields to match what the frontend expects
        $mappedSummary = [
            'totalEarnings' => $summary['total_earnings'] ?? 0,
            'availableForWithdrawal' => $summary['available_amount'] ?? 0,
            'pendingPayments' => $summary['pending_amount'] ?? 0,
            'monthlyEarnings' => $summary['monthly_earnings'] ?? 0,
        ];
        
        // Get payment records for the writer
        $paymentRecords = WriterPaymentRecord::where('writer_id', $writerId)
            ->orderBy('created_at', 'desc')
            ->paginate(20);
        
        // Get recent withdrawal requests
        $recentWithdrawals = WriterWithdrawalRequest::where('writer_id', $writerId)
            ->orderBy('created_at', 'desc')
            ->limit(10)
            ->get();
        
        return Inertia::render('Writer/Payment/PaymentHistory', [
            'summary' => $mappedSummary,
            'paymentRecords' => $paymentRecords,
            'recentWithdrawals' => $recentWithdrawals
        ]);
    }

    /**
     * Show payment methods page.
     */
    public function paymentMethods()
    {
        $writerId = Auth::id();
        $summary = $this->writerPaymentService->getWriterPaymentSummary($writerId);
        
        // Map the summary fields to match what the frontend expects
        $mappedSummary = [
            'totalEarnings' => $summary['total_earnings'] ?? 0,
            'availableForWithdrawal' => $summary['available_amount'] ?? 0,
            'pendingPayments' => $summary['pending_amount'] ?? 0,
            'monthlyEarnings' => $summary['monthly_earnings'] ?? 0,
        ];
        
        // Get available payment methods
        $availableMethods = WriterPaymentMethod::where('is_active', true)->get();
        
        // Get writer's payment accounts
        $paymentAccounts = WriterPaymentAccount::whereHas('writerProfile', function($query) use ($writerId) {
            $query->where('user_id', $writerId);
        })->with(['paymentMethod'])->get();
        
        return Inertia::render('Writer/Payment/PaymentMethods', [
            'summary' => $mappedSummary,
            'availableMethods' => $availableMethods,
            'paymentAccounts' => $paymentAccounts
        ]);
    }

    /**
     * Store a new payment method.
     */
    public function storePaymentMethod(Request $request)
    {
        $request->validate([
            'payment_method_id' => 'required|exists:writer_payment_methods,id',
            'account_details' => 'required|array',
            'is_primary' => 'boolean'
        ]);

        try {
            $writerId = Auth::id();
            $writerProfile = WriterProfile::where('user_id', $writerId)->first();
            
            if (!$writerProfile) {
                return response()->json(['success' => false, 'message' => 'Writer profile not found'], 404);
            }

            // If set as primary, unset other primary accounts
            if ($request->is_primary) {
                WriterPaymentAccount::whereHas('writerProfile', function($query) use ($writerId) {
                    $query->where('user_id', $writerId);
                })->update(['is_primary' => false]);
            }

            $account = WriterPaymentAccount::create([
                'writer_profile_id' => $writerProfile->id,
                'payment_method_id' => $request->payment_method_id,
                'account_details' => $request->account_details,
                'is_primary' => $request->is_primary ?? false,
                'is_verified' => false
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Payment method added successfully',
                'data' => $account->load('paymentMethod')
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to add payment method: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Update a payment method.
     */
    public function updatePaymentMethod(Request $request, $accountId)
    {
        $request->validate([
            'account_details' => 'required|array',
            'is_primary' => 'boolean'
        ]);

        try {
            $writerId = Auth::id();
            $account = WriterPaymentAccount::whereHas('writerProfile', function($query) use ($writerId) {
                $query->where('user_id', $writerId);
            })->findOrFail($accountId);

            // If setting as primary, unset other primary accounts
            if ($request->is_primary) {
                WriterPaymentAccount::whereHas('writerProfile', function($query) use ($writerId) {
                    $query->where('user_id', $writerId);
                })->where('id', '!=', $accountId)->update(['is_primary' => false]);
            }

            $account->update([
                'account_details' => $request->account_details,
                'is_primary' => $request->is_primary ?? $account->is_primary
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Payment method updated successfully',
                'data' => $account->load('paymentMethod')
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to update payment method: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Delete a payment method.
     */
    public function deletePaymentMethod($accountId)
    {
        try {
            $writerId = Auth::id();
            $account = WriterPaymentAccount::whereHas('writerProfile', function($query) use ($writerId) {
                $query->where('user_id', $writerId);
            })->findOrFail($accountId);

            $account->delete();

            return response()->json([
                'success' => true,
                'message' => 'Payment method deleted successfully'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to delete payment method: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Set a payment method as primary.
     */
    public function setPrimaryPaymentMethod($accountId)
    {
        try {
            $writerId = Auth::id();
            
            // Unset all primary accounts
            WriterPaymentAccount::whereHas('writerProfile', function($query) use ($writerId) {
                $query->where('user_id', $writerId);
            })->update(['is_primary' => false]);

            // Set the selected account as primary
            $account = WriterPaymentAccount::whereHas('writerProfile', function($query) use ($writerId) {
                $query->where('user_id', $writerId);
            })->findOrFail($accountId);

            $account->update(['is_primary' => true]);

            return response()->json([
                'success' => true,
                'message' => 'Primary payment method updated successfully',
                'data' => $account->load('paymentMethod')
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to update primary payment method: ' . $e->getMessage()
            ], 500);
        }
    }
}
