<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Symfony\Component\HttpFoundation\Response;

class CacheControl
{
    /**
     * Handle an incoming request.
     *
     * @param  \Closure(\Illuminate\Http\Request): (\Symfony\Component\HttpFoundation\Response)  $next
     */
    public function handle(Request $request, Closure $next): Response
    {
        $response = $next($request);

        // Don't cache for authenticated users or POST requests
        if ($request->user() || $request->method() !== 'GET') {
            return $response;
        }

        // Get the current path
        $path = $request->path();

        // Don't cache pages with authentication-dependent content
        // These pages need to reflect real-time authentication state
        if ($this->isAuthDependentPage($path)) {
            $response->headers->set('Cache-Control', 'no-cache, no-store, must-revalidate');
            $response->headers->set('Pragma', 'no-cache');
            $response->headers->set('Expires', '0');
            return $response;
        }

        // Set cache headers based on content type
        if ($this->isStaticAsset($path)) {
            // Static assets: 1 year cache
            $response->headers->set('Cache-Control', 'public, max-age=31536000, immutable');
        } elseif ($this->isBlogContent($path)) {
            // Blog content: 1 hour cache
            $response->headers->set('Cache-Control', 'public, max-age=3600, must-revalidate');
        } elseif ($this->isServicePage($path)) {
            // Service pages: 1 day cache
            $response->headers->set('Cache-Control', 'public, max-age=86400, must-revalidate');
        } elseif ($this->isHomepage($path)) {
            // Homepage: 30 minutes cache
            $response->headers->set('Cache-Control', 'public, max-age=1800, must-revalidate');
        } else {
            // Other pages: 2 hours cache
            $response->headers->set('Cache-Control', 'public, max-age=7200, must-revalidate');
        }

        return $response;
    }

    /**
     * Check if path is a static asset
     */
    private function isStaticAsset(string $path): bool
    {
        return preg_match('/\.(js|css|jpg|jpeg|png|gif|svg|webp|woff|woff2|ttf|ico)$/i', $path);
    }

    /**
     * Check if path is blog content
     */
    private function isBlogContent(string $path): bool
    {
        return str_starts_with($path, 'blog/');
    }

    /**
     * Check if path is a service page
     */
    private function isServicePage(string $path): bool
    {
        return str_starts_with($path, 'services/');
    }

    /**
     * Check if path is homepage
     */
    private function isHomepage(string $path): bool
    {
        return $path === '/' || $path === '';
    }

    /**
     * Check if path is an authentication-dependent page
     * These pages should never be cached as they display different content
     * based on whether the user is logged in or not
     */
    private function isAuthDependentPage(string $path): bool
    {
        $authDependentPaths = [
            'place-order',
            'orders',
            'dashboard',
            'profile',
            'login',
            'register',
        ];

        foreach ($authDependentPaths as $authPath) {
            if ($path === $authPath || str_starts_with($path, $authPath . '/')) {
                return true;
            }
        }

        return false;
    }
}

