<?php

namespace App\Http\Middleware;

use App\Models\WriterQualificationTest;
use App\Models\WriterTestResponse;
use Closure;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Symfony\Component\HttpFoundation\Response;

class CheckExpiredQuestionTests
{
    /**
     * The probability (0-100) that this middleware will run on any given request.
     * This helps distribute the load of checking for expired tests.
     */
    const RUN_PROBABILITY = 10; // 10% chance to run on any request

    /**
     * The maximum number of expired tests to process in a single request.
     */
    const BATCH_SIZE = 5;

    /**
     * Handle an incoming request.
     *
     * @param  \Closure(\Illuminate\Http\Request): (\Symfony\Component\HttpFoundation\Response)  $next
     */
    public function handle(Request $request, Closure $next): Response
    {
        // Only run this middleware with a certain probability to reduce database load
        if (rand(1, 100) <= self::RUN_PROBABILITY) {
            $this->processExpiredTests();
        }

        return $next($request);
    }

    /**
     * Process expired question tests.
     *
     * @return void
     */
    private function processExpiredTests(): void
    {
        try {
            // Find question tests that have expired but haven't been processed
            $expiredTests = WriterQualificationTest::where('questions_test_status', 'questions_pending')
                ->whereNotNull('expected_end_at')
                ->where('expected_end_at', '<', now())
                ->whereNull('auto_processed_at')
                ->limit(self::BATCH_SIZE)
                ->get();

            if ($expiredTests->isEmpty()) {
                return;
            }

            Log::info("Processing " . $expiredTests->count() . " expired question tests");

            foreach ($expiredTests as $test) {
                // Get the writer profile and user ID
                $writerProfile = $test->writerProfile;
                if (!$writerProfile || !$writerProfile->user_id) {
                    Log::warning("Expired test ID {$test->id} has no valid writer profile");
                    continue;
                }

                // Get any responses that might have been submitted
                $responses = WriterTestResponse::where('user_id', $writerProfile->user_id)
                    ->whereDate('created_at', '>=', $test->started_at->toDateString())
                    ->get();

                if ($responses->isNotEmpty()) {
                    // Process partial submission
                    $answers = $responses->pluck('selected_answer', 'question_id')->toArray();
                    $test->processPartialSubmission($answers);
                    Log::info("Auto-processed partial submission for question test ID: {$test->id}");
                } else {
                    // Mark as expired with 3-month retry period if no answers were submitted
                    $test->markAsExpired('Auto-processed by middleware - no answers submitted');
                    Log::info("Marked question test ID: {$test->id} as expired with 3-month retry period");
                }
            }
        } catch (\Exception $e) {
            // Log the error but don't interrupt the request
            Log::error("Error processing expired question tests: " . $e->getMessage());
            Log::error($e->getTraceAsString());
        }
    }
}
