<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Symfony\Component\HttpFoundation\Response;

class ValidateBlogUpload
{
    /**
     * Handle an incoming request.
     */
    public function handle(Request $request, Closure $next): Response
    {
        // Only validate if files are being uploaded
        if (!$request->hasFile('blog_files')) {
            return $next($request);
        }

        $config = config('blog_uploads');
        $files = $request->file('blog_files');

        // Handle single file upload
        if (!is_array($files)) {
            $files = [$files];
        }

        // Validate file count
        if (count($files) > $config['file_validation']['max_files_per_upload']) {
            return response()->json([
                'error' => 'Too many files uploaded. Maximum allowed: ' . $config['file_validation']['max_files_per_upload']
            ], 422);
        }

        foreach ($files as $file) {
            // Validate file extension
            if (!$this->validateFileExtension($file, $config)) {
                return response()->json([
                    'error' => 'Invalid file extension. Allowed: ' . implode(', ', $config['file_validation']['allowed_extensions'])
                ], 422);
            }

            // Validate MIME type
            if (!$this->validateMimeType($file, $config)) {
                return response()->json([
                    'error' => 'Invalid file type. File must be a JavaScript file.'
                ], 422);
            }

            // Validate file size
            if (!$this->validateFileSize($file, $config)) {
                return response()->json([
                    'error' => 'File too large. Maximum size: ' . $this->formatBytes($config['file_validation']['max_file_size'])
                ], 422);
            }

            // Validate file content
            $contentValidation = $this->validateFileContent($file, $config);
            if (!$contentValidation['valid']) {
                // Log security event
                Log::warning('Suspicious file upload detected', [
                    'filename' => $file->getClientOriginalName(),
                    'issues' => $contentValidation['issues'],
                    'ip' => $request->ip(),
                    'user_agent' => $request->userAgent(),
                ]);

                return response()->json([
                    'error' => 'File content validation failed: ' . implode(', ', $contentValidation['issues'])
                ], 422);
            }
        }

        return $next($request);
    }

    /**
     * Validate file extension
     */
    private function validateFileExtension($file, $config): bool
    {
        $extension = strtolower($file->getClientOriginalExtension());
        return in_array($extension, $config['file_validation']['allowed_extensions']);
    }

    /**
     * Validate MIME type
     */
    private function validateMimeType($file, $config): bool
    {
        $mimeType = $file->getMimeType();
        return in_array($mimeType, $config['file_validation']['allowed_mime_types']);
    }

    /**
     * Validate file size
     */
    private function validateFileSize($file, $config): bool
    {
        return $file->getSize() <= $config['file_validation']['max_file_size'];
    }

    /**
     * Validate file content for security issues
     */
    private function validateFileContent($file, $config): array
    {
        $content = file_get_contents($file->getRealPath());
        $issues = [];

        // Check for malicious patterns
        if ($config['security']['scan_for_malicious_code']) {
            foreach ($config['security']['blocked_patterns'] as $pattern) {
                if (preg_match('/' . $pattern . '/i', $content)) {
                    $issues[] = "Suspicious pattern detected: {$pattern}";
                }
            }
        }

        // Basic JavaScript syntax validation
        if (!$this->validateJavaScriptSyntax($content)) {
            $issues[] = "Invalid JavaScript syntax";
        }

        // Check for required export structure
        if (!$this->validateExportStructure($content)) {
            $issues[] = "Invalid export structure - must export default object";
        }

        return [
            'valid' => empty($issues),
            'issues' => $issues
        ];
    }

    /**
     * Basic JavaScript syntax validation
     */
    private function validateJavaScriptSyntax($content): bool
    {
        // Check for basic JavaScript structure
        // This is a simple check - for production, consider using a proper JS parser

        // Must contain export default
        if (!preg_match('/export\s+default\s+/i', $content)) {
            return false;
        }

        // Check for balanced braces
        $openBraces = substr_count($content, '{');
        $closeBraces = substr_count($content, '}');

        if ($openBraces !== $closeBraces) {
            return false;
        }

        // Check for balanced parentheses
        $openParens = substr_count($content, '(');
        $closeParens = substr_count($content, ')');

        if ($openParens !== $closeParens) {
            return false;
        }

        return true;
    }




    /**
     * Validate export structure
     */
    private function validateExportStructure($content): bool
    {
        // Remove comments first
        $content = $this->removeComments($content);

        // Check for direct export: export default { ... }
        if (preg_match('/export\s+default\s+\{[\s\S]*\}/i', $content)) {
            return true;
        }

        // Check for variable export: const post = { ... }; export default post;
        if (preg_match('/const\s+\w+\s*=\s*\{[\s\S]*\};\s*export\s+default\s+\w+;?/i', $content)) {
            return true;
        }

        return false;
    }

    /**
     * Remove JavaScript comments
     */
    private function removeComments($content): string
    {
        // Remove single-line comments
        $content = preg_replace('/\/\/.*$/m', '', $content);

        // Remove multi-line comments
        $content = preg_replace('/\/\*[\s\S]*?\*\//', '', $content);

        return $content;
    }



    /**
     * Format bytes to human readable format
     */
    private function formatBytes($bytes): string
    {
        $units = ['B', 'KB', 'MB', 'GB'];

        for ($i = 0; $bytes > 1024 && $i < count($units) - 1; $i++) {
            $bytes /= 1024;
        }

        return round($bytes, 2) . ' ' . $units[$i];
    }
}
