<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Symfony\Component\HttpFoundation\Response;
use Illuminate\Support\Facades\Auth;

class WriterMiddleware
{
    public function handle(Request $request, Closure $next): Response
    {
        if (!Auth::check()) {
            return redirect()->route('login');
        }

        $user = Auth::user();

        // Allow broadcasting routes for all authenticated users
        // Broadcasting needs to work across all user types for messaging
        if ($this->isBroadcastingRoute($request)) {
            return $next($request);
        }

        // Check if user is a writer
        if ($user->user_type !== 'writer') {
            // Check if we're already on a dashboard route to prevent redirect loops
            $currentRoute = $request->route()->getName();

            switch ($user->user_type) {
                case 'admin':
                case 'super_admin':
                    $targetRoute = 'admin.dashboard';
                    break;
                case 'client':
                    $targetRoute = 'dashboard';
                    break;
                default:
                    $targetRoute = 'login';
                    break;
            }

            // Only redirect if we're not already on the target route
            if ($currentRoute !== $targetRoute) {
                return redirect()->route($targetRoute);
            }

            // If we're already on the correct route, abort with 403 (Forbidden)
            abort(403, 'You do not have access to this section.');
        }

        // Writer is authenticated, now check test status
        $writerProfile = $user->writerProfile;

        // If writer profile doesn't exist, create it
        if (!$writerProfile) {
            $writerProfile = $user->writerProfile()->create([]);
        }

        // Get the current route name
        $routeName = $request->route()->getName();

        // Define routes that are accessible regardless of test status
        $alwaysAccessible = [
            'writer.dashboard',
            'writer.test.instructions',
            'writer.test.start',
            'writer.test.results',
            'profile.edit',
            'profile.update',
            'profile.destroy',
            'logout'
        ];

        // If the route is always accessible, allow access
        if (in_array($routeName, $alwaysAccessible)) {
            return $next($request);
        }

        // Get qualification test status
        $questionsStatus = $writerProfile->qualificationTest ?
            $writerProfile->qualificationTest->questions_test_status : 'not_started';

        // Get essay test status
        $essayStatus = $writerProfile->essayTest ?
            $writerProfile->essayTest->essay_test_status : 'not_started';

        // Handle questions test-specific routes based on questions test status
        if (str_starts_with($routeName, 'writer.test.')) {
            switch ($questionsStatus) {
                case 'not_started':
                    // If writer hasn't started the test, only allow access to test instructions and start routes
                    if ($routeName !== 'writer.test.instructions' && $routeName !== 'writer.test.start') {
                        return redirect()->route('writer.test.instructions');
                    }
                    break;

                case 'questions_pending':
                    // If writer is taking the test, only allow access to test taking page
                    if (
                        $routeName !== 'writer.test.take' && $routeName !== 'writer.test.submit' &&
                        $routeName !== 'writer.test.ping' && $routeName !== 'writer.test.record-activity'
                    ) {
                        return redirect()->route('writer.test.take');
                    }
                    break;

                case 'questions_failed':
                    // If writer failed the test, check if they can retry
                    if (
                        $writerProfile->qualificationTest->can_retry_after &&
                        now()->lt($writerProfile->qualificationTest->can_retry_after)
                    ) {
                        // If they can't retry yet, only allow access to results page
                        if ($routeName !== 'writer.test.results') {
                            return redirect()->route('writer.test.results');
                        }
                    }
                    break;

                case 'passed':
                    // If writer passed the questions test, they can access all test-related routes
                    break;

                default:
                    // For any other status, redirect to dashboard
                    return redirect()->route('writer.dashboard');
            }

            return $next($request);
        }

        // Handle essay test-specific routes based on both test statuses
        if (str_starts_with($routeName, 'writer.essay.')) {
            // Writer must have passed the questions test to access essay routes
            if ($questionsStatus !== 'passed') {
                return redirect()->route('writer.dashboard')
                    ->with('error', 'You must pass the multiple-choice test before taking the essay test.');
            }

            switch ($essayStatus) {
                case 'not_started':
                    // If writer hasn't started the essay, only allow access to essay instructions
                    if (!in_array($routeName, ['writer.essay.instructions', 'writer.essay.start'])) {
                        return redirect()->route('writer.essay.instructions');
                    }
                    break;

                case 'essay_pending':
                    // If writer is taking the essay, only allow access to essay writing page
                    if (!in_array($routeName, [
                        'writer.essay.write',
                        'writer.essay.submit',
                        'writer.essay.upload-file',
                        'writer.essay.check-time',
                        'writer.essay.auto-submit'
                    ])) {
                        return redirect()->route('writer.essay.write');
                    }
                    break;

                case 'essay_submitted':
                    // If writer submitted essay and is waiting for review
                    if ($routeName !== 'writer.essay.confirmation') {
                        return redirect()->route('writer.essay.confirmation');
                    }
                    break;

                case 'essay_failed':
                case 'essay_expired':
                    // If writer failed the essay test
                    if ($writerProfile->essayTest->can_retry_after && now()->lt($writerProfile->essayTest->can_retry_after)) {
                        // If they can't retry yet, only allow access to essay confirmation page
                        if ($routeName !== 'writer.essay.confirmation') {
                            return redirect()->route('writer.essay.confirmation');
                        }
                    }
                    break;

                case 'passed':
                    // If writer passed the essay test, they can access all essay-related routes
                    break;

                default:
                    // For any other status, redirect to dashboard
                    return redirect()->route('writer.dashboard');
            }

            return $next($request);
        }

        // For order and bidding routes, writer must have passed both tests
        if (
            str_starts_with($routeName, 'writer.orders') ||
            str_starts_with($routeName, 'writer.bids') ||
            $routeName === 'writer.availableOrders'
        ) {
            // Use the isApproved method to check if writer has passed both tests
            if (!$writerProfile->isApproved()) {
                return redirect()->route('writer.dashboard')
                    ->with('error', 'You must complete the qualification process before accessing orders.');
            }
        }

        return $next($request);
    }

    /**
     * Check if the current request is for a broadcasting route
     * Broadcasting routes need to be accessible to all authenticated users
     * for real-time messaging functionality
     */
    private function isBroadcastingRoute(Request $request): bool
    {
        $routeName = $request->route()?->getName();
        $path = $request->path();

        return $routeName === 'broadcasting.auth' || 
               str_starts_with($path, 'broadcasting/') ||
               str_starts_with($routeName ?? '', 'broadcasting.');
    }
}
