<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Str;

class BlogCategory extends Model
{
    use HasFactory;

    protected $table = 'blog_categories';

    protected $fillable = [
        'name',
        'slug',
        'description',
        'color',
        'icon',
        'meta_title',
        'meta_description',
        'meta_keywords',
        'featured_image',
        'featured_image_alt',
        'is_active',
        'sort_order',
        'parent_id',
    ];

    protected $casts = [
        'is_active' => 'boolean',
        'sort_order' => 'integer',
    ];

    protected $appends = [
        'url',
        'full_name',
    ];

    // Relationships
    public function posts(): HasMany
    {
        return $this->hasMany(BlogPost::class, 'category_id');
    }

    public function publishedPosts(): HasMany
    {
        return $this->hasMany(BlogPost::class, 'category_id')
            ->where('status', 'published')
            ->where('published_at', '<=', now());
    }

    public function featuredPosts(): HasMany
    {
        return $this->publishedPosts()
            ->where('featured', true)
            ->latest('published_at');
    }

    public function parent()
    {
        return $this->belongsTo(BlogCategory::class, 'parent_id');
    }

    public function children(): HasMany
    {
        return $this->hasMany(BlogCategory::class, 'parent_id');
    }

    // Scopes
    public function scopeActive(Builder $query): Builder
    {
        return $query->where('is_active', true);
    }

    public function scopeWithPublishedPosts(Builder $query): Builder
    {
        return $query->whereHas('publishedPosts');
    }

    public function scopeOrderedBySort(Builder $query): Builder
    {
        return $query->orderBy('sort_order', 'asc')->orderBy('name', 'asc');
    }

    public function scopeParent(Builder $query): Builder
    {
        return $query->whereNull('parent_id');
    }

    public function scopeChild(Builder $query): Builder
    {
        return $query->whereNotNull('parent_id');
    }

    // Accessors
    public function getRouteKeyName()
    {
        return 'slug';
    }

    public function getUrlAttribute(): string
    {
        return url("/blog/category/{$this->slug}");
    }

    public function getFullNameAttribute(): string
    {
        if ($this->parent) {
            return $this->parent->name . ' > ' . $this->name;
        }

        return $this->name;
    }

    public function getMetaTitleAttribute($value): string
    {
        return $value ?: ($this->name . ' - Academic Writing Blog | Academic Scribe');
    }

    public function getMetaDescriptionAttribute($value): string
    {
        return $value ?: ("Explore {$this->name} articles and guides to improve your academic writing skills. Expert tips and strategies for students.");
    }

    public function getMetaKeywordsAttribute($value): string
    {
        return $value ?: (strtolower($this->name) . ', academic writing, essay tips, writing guides, student resources');
    }

    // Methods
    public function getPostCount(): int
    {
        return $this->publishedPosts()->count();
    }

    public function getLatestPost()
    {
        return $this->publishedPosts()->latest('published_at')->first();
    }

    public function hasChildren(): bool
    {
        return $this->children()->exists();
    }

    public function isParent(): bool
    {
        return is_null($this->parent_id);
    }

    public function isChild(): bool
    {
        return !is_null($this->parent_id);
    }

    public function generateSlug(): string
    {
        $slug = Str::slug($this->name);
        $originalSlug = $slug;
        $counter = 1;

        while (static::where('slug', $slug)->where('id', '!=', $this->id ?? 0)->exists()) {
            $slug = $originalSlug . '-' . $counter++;
        }

        return $slug;
    }

    // Boot method to auto-generate slug
    protected static function boot()
    {
        parent::boot();

        static::creating(function ($category) {
            if (empty($category->slug)) {
                $category->slug = $category->generateSlug();
            }
        });

        static::updating(function ($category) {
            if ($category->isDirty('name') && empty($category->slug)) {
                $category->slug = $category->generateSlug();
            }
        });
    }

    // SEO Methods
    public function getSeoTitle(): string
    {
        return $this->meta_title ?: ($this->name . ' - Academic Writing Blog | Academic Scribe');
    }

    public function getSeoDescription(): string
    {
        return $this->meta_description ?: ("Explore {$this->name} articles and guides to improve your academic writing skills. Expert tips and strategies for students.");
    }

    public function getSeoKeywords(): string
    {
        return $this->meta_keywords ?: (strtolower($this->name) . ', academic writing, essay tips, writing guides, student resources');
    }

    public function getCanonicalUrl(): string
    {
        return $this->url;
    }

    public function getOgImage(): string
    {
        return $this->featured_image ?: url('/images/blog/category-default-og.jpg');
    }

    public function getTwitterImage(): string
    {
        return $this->featured_image ?: url('/images/blog/category-default-twitter.jpg');
    }

    // Breadcrumb generation
    public function getBreadcrumb(): array
    {
        $breadcrumb = [
            ['name' => 'Home', 'url' => url('/')],
            ['name' => 'Blog', 'url' => url('/blog')],
        ];

        if ($this->parent) {
            $breadcrumb[] = ['name' => $this->parent->name, 'url' => $this->parent->url];
        }

        $breadcrumb[] = ['name' => $this->name, 'url' => $this->url];

        return $breadcrumb;
    }

    // Cache key generation
    public function getCacheKey(string $suffix = ''): string
    {
        return "blog_category_{$this->id}" . ($suffix ? "_{$suffix}" : '');
    }
}
