<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;

class BlogPost extends Model
{
    use HasFactory;

    protected $fillable = [
        'title',
        'slug',
        'subtitle',
        'excerpt',
        'content',
        'featured_image',
        'featured_image_alt',
        'category_id',
        'author_id',
        'status',
        'published_at',
        'read_time',
        'difficulty',
        'featured',
        'enable_comments',
        'enable_ratings',
        'meta_data',
        'structured_data',
        'social_sharing',
        'analytics',
        'table_of_contents',
    ];

    protected $casts = [
        'published_at' => 'datetime',
        'featured' => 'boolean',
        'enable_comments' => 'boolean',
        'enable_ratings' => 'boolean',
        'meta_data' => 'array',
        'structured_data' => 'array',
        'social_sharing' => 'array',
        'analytics' => 'array',
        'table_of_contents' => 'array',
    ];

    /**
     * Get the route key for the model.
     */
    public function getRouteKeyName()
    {
        return 'slug';
    }

    /**
     * Get the category that owns the blog post.
     */
    public function category(): BelongsTo
    {
        return $this->belongsTo(BlogCategory::class);
    }

    /**
     * Get the author that owns the blog post.
     */
    public function author(): BelongsTo
    {
        return $this->belongsTo(BlogAuthor::class);
    }

    /**
     * Get the tags for the blog post.
     */
    public function tags(): BelongsToMany
    {
        return $this->belongsToMany(
            BlogTag::class,
            'blog_post_tags',
            'blog_post_id',
            'tag_id'
        );
    }

    /**
     * Scope a query to only include published posts.
     */
    public function scopePublished($query)
    {
        return $query->where('status', 'published')
            ->where('published_at', '<=', now());
    }

    /**
     * Scope a query to only include featured posts.
     */
    public function scopeFeatured($query)
    {
        return $query->where('featured', true);
    }

    /**
     * Get the post's full URL.
     */
    public function getUrlAttribute()
    {
        return url("/blog/{$this->slug}");
    }

    /**
     * Get the post's edit URL.
     */
    public function getEditUrlAttribute()
    {
        return route('admin.blog.posts.edit', $this->id);
    }
}
