<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class ClientPreference extends Model
{
    protected $fillable = [
        'user_id',
        'default_academic_level_id',
        'default_paper_type_id',
        'default_subject_id',
        'default_spacing',
        'default_citation_style',
        'default_language_id',
        'default_currency',
        'auto_assign_writer',
        'preferred_writer_ids',
        'blacklisted_writer_ids',
        'default_deadline_buffer_hours',
        'allow_writer_contact_after_completion',
        'show_order_history_to_writers',
        'profile_visible_to_writers',
        'preferred_communication_method',
        'timezone',
        'date_format',
        'time_format',
        'theme',
        'auto_send_invoices',
        'invoice_format',
    ];

    protected $casts = [
        'preferred_writer_ids' => 'array',
        'blacklisted_writer_ids' => 'array',
        'auto_assign_writer' => 'boolean',
        'allow_writer_contact_after_completion' => 'boolean',
        'show_order_history_to_writers' => 'boolean',
        'profile_visible_to_writers' => 'boolean',
        'auto_send_invoices' => 'boolean',
        'default_deadline_buffer_hours' => 'integer',
    ];

    /**
     * Get the user that owns the preferences.
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Get the default academic level.
     */
    public function defaultAcademicLevel(): BelongsTo
    {
        return $this->belongsTo(AcademicLevel::class, 'default_academic_level_id');
    }

    /**
     * Get the default paper type.
     */
    public function defaultPaperType(): BelongsTo
    {
        return $this->belongsTo(PaperType::class, 'default_paper_type_id');
    }

    /**
     * Get the default subject.
     */
    public function defaultSubject(): BelongsTo
    {
        return $this->belongsTo(Subject::class, 'default_subject_id');
    }

    /**
     * Get the default language.
     */
    public function defaultLanguage(): BelongsTo
    {
        return $this->belongsTo(Language::class, 'default_language_id');
    }

    /**
     * Get or create preferences for a user.
     */
    public static function getOrCreateForUser(int $userId): self
    {
        return static::firstOrCreate(
            ['user_id' => $userId],
            [
                'default_spacing' => 'double',
                'default_currency' => 'USD',
                'default_deadline_buffer_hours' => 24,
                'allow_writer_contact_after_completion' => true,
                'profile_visible_to_writers' => true,
                'preferred_communication_method' => 'in_app',
                'date_format' => 'Y-m-d',
                'time_format' => 'H:i',
                'theme' => 'auto',
                'auto_send_invoices' => true,
                'invoice_format' => 'pdf',
            ]
        );
    }
}