<?php

namespace App\Models;

use Carbon\Carbon;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Support\Str;

class Coupon extends Model
{
    use HasFactory, SoftDeletes;

    // Coupon types
    const TYPE_FIRST_ORDER = 'first_order';
    const TYPE_FLASH = 'flash';
    const TYPE_REFERRAL = 'referral';
    const TYPE_LOYALTY = 'loyalty';
    const TYPE_ANNIVERSARY = 'anniversary';
    const TYPE_SEASONAL = 'seasonal';
    const TYPE_VOLUME = 'volume';
    const TYPE_DEFAULT = 'default';

    // Discount types
    const DISCOUNT_FIXED = 'fixed';
    const DISCOUNT_PERCENTAGE = 'percentage';

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'coupon_code',
        'discount_type',
        'discount_amount',
        'description',
        'coupon_type',
        'loyalty_tier_id',
        'seasonal_type_id',
        'start_date',
        'expiry_date',
        'max_uses',
        'uses_count',
        'per_user_limit',
        'min_order_amount',
        'min_pages',
        'min_years',
        'is_first_order_only',
        'user_specific',
        'referrer_user_id',
        'is_referral',
        'is_active',
        'created_by',
        'updated_by',
        'deleted_by',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'discount_amount' => 'decimal:2',
        'start_date' => 'datetime',
        'expiry_date' => 'datetime',
        // Remove integer casts to preserve null values for unlimited
        // 'max_uses' => 'integer',
        'uses_count' => 'integer',
        // 'per_user_limit' => 'integer',  
        'min_order_amount' => 'decimal:2',
        'min_pages' => 'integer',
        'min_years' => 'integer',
        'is_first_order_only' => 'boolean',
        'user_specific' => 'boolean',
        'is_referral' => 'boolean',
        'is_active' => 'boolean',
    ];

    /**
     * Get the loyalty tier associated with this coupon.
     */
    public function loyaltyTier(): BelongsTo
    {
        return $this->belongsTo(LoyaltyTier::class);
    }

    /**
     * Get the seasonal type associated with this coupon.
     */
    public function seasonalType(): BelongsTo
    {
        return $this->belongsTo(SeasonalType::class);
    }

    /**
     * Get the referrer user associated with this coupon.
     */
    public function referrer(): BelongsTo
    {
        return $this->belongsTo(User::class, 'referrer_user_id');
    }

    /**
     * Get the user who created this coupon.
     */
    public function creator(): BelongsTo
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    /**
     * Get the user who last updated this coupon.
     */
    public function updater(): BelongsTo
    {
        return $this->belongsTo(User::class, 'updated_by');
    }

    /**
     * Get the user who deleted this coupon.
     */
    public function deleter(): BelongsTo
    {
        return $this->belongsTo(User::class, 'deleted_by');
    }

    /**
     * Get the user coupon usages.
     */
    public function userUsages()
    {
        return $this->hasMany(UserCoupon::class);
    }

    /**
     * Get orders that used this coupon.
     */
    public function orders()
    {
        return $this->hasMany(Order::class, 'coupon_id');
    }

    /**
     * Scope a query to only include active coupons.
     */
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    /**
     * Scope a query to only include valid coupons by date.
     */
    public function scopeValidByDate($query)
    {
        $now = Carbon::now();
        return $query->where(function ($q) use ($now) {
            $q->whereNull('start_date')
                ->orWhere('start_date', '<=', $now);
        })->where(function ($q) use ($now) {
            $q->whereNull('expiry_date')
                ->orWhere('expiry_date', '>=', $now);
        });
    }

    /**
     * Scope a query to only include coupons that haven't exceeded their max uses.
     */
    public function scopeWithinUsageLimit($query)
    {
        return $query->where(function ($q) {
            $q->whereNull('max_uses')
                ->orWhereRaw('uses_count < max_uses');
        });
    }

    /**
     * Check if the coupon is valid by date.
     *
     * @return bool
     */
    public function isValidByDate(): bool
    {
        $now = Carbon::now();

        if ($this->start_date && $now->lt($this->start_date)) {
            return false;
        }

        if ($this->expiry_date && $now->gt($this->expiry_date)) {
            return false;
        }

        return true;
    }

    /**
     * Check if the coupon has exceeded its usage limit.
     *
     * @return bool
     */
    public function hasExceededUsageLimit(): bool
    {
        if ($this->max_uses === null) {
            return false;
        }

        return $this->uses_count >= $this->max_uses;
    }

    /**
     * Check if the user has exceeded their per-user limit for this coupon.
     *
     * @param int $userId
     * @return bool
     */
    public function hasUserExceededLimit(int $userId): bool
    {
        if ($this->per_user_limit === null) {
            return false;
        }

        $userUsageCount = $this->userUsages()
            ->where('user_id', $userId)
            ->count();

        return $userUsageCount >= $this->per_user_limit;
    }

    /**
     * Calculate the discount amount for a given order total.
     *
     * @param float $orderTotal
     * @return float
     */
    public function calculateDiscount(float $orderTotal): float
    {
        if ($this->discount_type === self::DISCOUNT_PERCENTAGE) {
            return ($orderTotal * $this->discount_amount) / 100;
        }

        return min($this->discount_amount, $orderTotal);
    }

    /**
     * Validate if the coupon is applicable for a given order context.
     *
     * @param array $context Order context with user_id, order_total, pages, etc.
     * @return array ['valid' => bool, 'message' => string]
     */
    public function validateForOrder(array $context): array
    {
        if (!$this->is_active) {
            return ['valid' => false, 'message' => 'This coupon is not active.'];
        }

        if (!$this->isValidByDate()) {
            return ['valid' => false, 'message' => 'This coupon has expired or is not yet active.'];
        }

        if ($this->hasExceededUsageLimit()) {
            return ['valid' => false, 'message' => 'This coupon has reached its usage limit.'];
        }

        if (isset($context['user_id']) && $this->hasUserExceededLimit($context['user_id'])) {
            return ['valid' => false, 'message' => 'You have already used this coupon the maximum number of times.'];
        }

        if ($this->min_order_amount && $context['order_total'] < $this->min_order_amount) {
            return [
                'valid' => false,
                'message' => "This coupon requires a minimum order amount of {$this->min_order_amount}."
            ];
        }

        if ($this->min_pages && $context['pages'] < $this->min_pages) {
            return [
                'valid' => false,
                'message' => "This coupon requires a minimum of {$this->min_pages} pages."
            ];
        }

        if ($this->is_first_order_only && isset($context['previous_orders']) && $context['previous_orders'] > 0) {
            return ['valid' => false, 'message' => 'This coupon is for first-time customers only.'];
        }

        // For loyalty coupons, check if user has the required loyalty tier
        if ($this->coupon_type === self::TYPE_LOYALTY && $this->loyalty_tier_id) {
            $userOrders = $context['previous_orders'] ?? 0;
            $loyaltyTier = LoyaltyTier::find($this->loyalty_tier_id);

            if ($loyaltyTier && $userOrders < $loyaltyTier->required_orders) {
                return [
                    'valid' => false,
                    'message' => "This coupon requires at least {$loyaltyTier->required_orders} previous orders."
                ];
            }
        }

        // For referral-specific coupons, check if the user is the intended recipient
        if ($this->user_specific && isset($context['user_id']) && $this->referrer_user_id != $context['user_id']) {
            return ['valid' => false, 'message' => 'This coupon is not available for your account.'];
        }

        return ['valid' => true, 'message' => 'Coupon is valid.'];
    }

    /**
     * Track the usage of this coupon.
     *
     * @param int|null $userId
     * @return void
     */
    public function trackUsage(?int $userId = null): void
    {
        $this->increment('uses_count');

        if ($userId) {
            UserCoupon::create([
                'user_id' => $userId,
                'coupon_id' => $this->id,
                'used_at' => Carbon::now(),
            ]);
        }
    }

    /**
     * Generate a unique coupon code.
     *
     * @param int $length
     * @return string
     */
    public static function generateUniqueCode(int $length = 8): string
    {
        do {
            $characters = '0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ';
            $code = '';
            for ($i = 0; $i < $length; $i++) {
                $code .= $characters[random_int(0, strlen($characters) - 1)];
            }
        } while (self::where('coupon_code', $code)->exists());

        return $code;
    }

    /**
     * Check if this coupon is a first-order coupon.
     *
     * @return bool
     */
    public function isFirstOrderCoupon(): bool
    {
        return $this->coupon_type === self::TYPE_FIRST_ORDER || $this->is_first_order_only;
    }

    /**
     * Check if this coupon is seasonal.
     *
     * @return bool
     */
    public function isSeasonalCoupon(): bool
    {
        return $this->coupon_type === self::TYPE_SEASONAL && $this->seasonal_type_id !== null;
    }

    /**
     * Check if this coupon is for loyalty rewards.
     *
     * @return bool
     */
    public function isLoyaltyCoupon(): bool
    {
        return $this->coupon_type === self::TYPE_LOYALTY && $this->loyalty_tier_id !== null;
    }

    /**
     * Check if this coupon is for volume discounts.
     *
     * @return bool
     */
    public function isVolumeCoupon(): bool
    {
        return $this->coupon_type === self::TYPE_VOLUME && $this->min_pages > 0;
    }

    /**
     * Check if this coupon is for anniversaries.
     *
     * @return bool
     */
    public function isAnniversaryCoupon(): bool
    {
        return $this->coupon_type === self::TYPE_ANNIVERSARY;
    }

    /**
     * Check if this coupon is a referral coupon.
     *
     * @return bool
     */
    public function isReferralCoupon(): bool
    {
        return $this->coupon_type === self::TYPE_REFERRAL || $this->is_referral;
    }

    /**
     * Get all coupons for a specific type.
     *
     * @param string $type
     * @return \Illuminate\Database\Eloquent\Collection
     */
    public static function getCouponsOfType(string $type)
    {
        return self::where('coupon_type', $type)
            ->active()
            ->validByDate()
            ->withinUsageLimit()
            ->get();
    }
}
