<?php

namespace App\Models;

use Carbon\Carbon;
use App\Models\OrderFile;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Factories\HasFactory;

class Order extends Model
{
    use HasFactory;

    public $timestamps = false;

    // Payment status constants
    const PAYMENT_STATUS_UNPAID = 'unpaid';
    const PAYMENT_STATUS_PAID = 'paid';
    const PAYMENT_STATUS_REFUNDED = 'refunded';
    const PAYMENT_STATUS_PARTIALLY_REFUNDED = 'partially_refunded';

    // Order status constants for the new timeline system
    const ORDER_STATUS_PENDING = 'pending';
    const ORDER_STATUS_PENDING_PAYMENT = 'pending_payment';
    const ORDER_STATUS_BIDDING = 'bidding';
    const ORDER_STATUS_ASSIGNED = 'assigned';
    const ORDER_STATUS_IN_PROGRESS = 'in_progress';
    const ORDER_STATUS_SUBMITTED = 'submitted';
    const ORDER_STATUS_UNDER_REVIEW = 'under_review';
    const ORDER_STATUS_APPROVED = 'approved';
    const ORDER_STATUS_REVISION_REQUESTED = 'revision_requested';
    const ORDER_STATUS_CANCELLED = 'cancelled';

    // Status groups for easier querying
    const ACTIVE_STATUSES = [
        self::ORDER_STATUS_PENDING,
        self::ORDER_STATUS_PENDING_PAYMENT,
        self::ORDER_STATUS_BIDDING,
        self::ORDER_STATUS_ASSIGNED,
        self::ORDER_STATUS_IN_PROGRESS,
        self::ORDER_STATUS_SUBMITTED,
        self::ORDER_STATUS_UNDER_REVIEW,
        self::ORDER_STATUS_REVISION_REQUESTED
    ];

    const COMPLETED_STATUSES = [
        self::ORDER_STATUS_APPROVED,
        self::ORDER_STATUS_CANCELLED
    ];

    // Writer payment status constants
    const WRITER_PAYMENT_STATUS_PENDING = 'pending';
    const WRITER_PAYMENT_STATUS_PENDING_REVISION = 'pending_revision';
    const WRITER_PAYMENT_STATUS_AVAILABLE = 'available';
    const WRITER_PAYMENT_STATUS_WITHDRAWN = 'withdrawn';
    const WRITER_PAYMENT_STATUS_DISPUTED = 'disputed';

    // Submission type constants
    const SUBMISSION_TYPE_DRAFT = 'draft';
    const SUBMISSION_TYPE_COMPLETE = 'complete';

    // Writer payment status groups
    const WRITER_PENDING_STATUSES = [
        self::WRITER_PAYMENT_STATUS_PENDING,
        self::WRITER_PAYMENT_STATUS_PENDING_REVISION
    ];

    const WRITER_AVAILABLE_STATUSES = [
        self::WRITER_PAYMENT_STATUS_AVAILABLE
    ];

    protected $fillable = [
        'order_number',
        'user_id',
        'writer_id',
        'coupon_id',
        'title',
        'type_of_paper',
        'subject',
        'academic_level',
        'urgency',
        'pages',
        'instructions',
        'styles',
        'dateposted',
        'deadline',
        'writer_deadline',
        'client_timezone',
        'sources',
        'order_amount',
        'net_amount',
        'writer_amount',
        'discount',
        'currency',
        'currency_rate',
        'coupon',
        'payment_status',
        'order_status',
        'powerpoint_slides',
        'spacing',
        // New tracking fields
        'status_changed_at',
        'status_changed_by',
        'status_change_reason',
        'writer_assigned_at',
        'work_started_at',
        'work_submitted_at',
        'admin_reviewed_at',
        'client_review_started_at',
        'client_approved_at',
        'auto_approval_at',
        'revision_notes',
        'revision_count',
        // Writer payment fields
        'writer_payment_released_at',
        'withdrawal_batch_id',
        'writer_payment_calculated_at',
        'writer_payment_status',
        // Submission timing fields
        'submission_was_late',
        'submission_late_by_hours',
        'submission_type',
        'order_revision',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'status_changed_at' => 'datetime',
        'writer_assigned_at' => 'datetime',
        'work_started_at' => 'datetime',
        'work_submitted_at' => 'datetime',
        'admin_reviewed_at' => 'datetime',
        'client_review_started_at' => 'datetime',
        'client_approved_at' => 'datetime',
        'auto_approval_at' => 'datetime',
        'revision_count' => 'integer',
        'writer_payment_released_at' => 'datetime',
        'writer_payment_calculated_at' => 'datetime',
        'submission_was_late' => 'boolean',
        'submission_late_by_hours' => 'integer',
        'submission_type' => 'string',
    ];


    /**
     * Get the currency associated with this order.
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function currency()
    {
        return $this->belongsTo(Currency::class, 'currency', 'currency_code');
    }

    /**
     * Get the user that owns the order.
     */
    public function user()
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Get the payments for the order.
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function payments()
    {
        return $this->hasMany(Payment::class);
    }

    /**
     * Get the latest payment for the order.
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasOne
     */
    public function latestPayment()
    {
        return $this->hasOne(Payment::class)->latest();
    }

    /**
     * Get the completed payment for the order, if any.
     *
     * @return Payment|null
     */
    public function getCompletedPayment()
    {
        return $this->payments()
            ->where(function ($query) {
                $query->where('status', Payment::STATUS_COMPLETED)
                    ->orWhere('status', Payment::STATUS_PARTIALLY_REFUNDED);
            })
            ->latest()
            ->first();
    }

    /**
     * Check if the order has been paid.
     *
     * @return bool
     */
    public function isPaid(): bool
    {
        return in_array($this->payment_status, [
            self::PAYMENT_STATUS_PAID,
            self::PAYMENT_STATUS_PARTIALLY_REFUNDED
        ]) || $this->getCompletedPayment() !== null;
    }

    /**
     * Check if the order has been fully refunded.
     *
     * @return bool
     */
    public function isRefunded(): bool
    {
        return $this->payment_status === self::PAYMENT_STATUS_REFUNDED;
    }

    /**
     * Check if the order has been partially refunded.
     *
     * @return bool
     */
    public function isPartiallyRefunded(): bool
    {
        return $this->payment_status === self::PAYMENT_STATUS_PARTIALLY_REFUNDED;
    }

    /**
     * Get the current payment status from the latest payment.
     *
     * @return string|null
     */
    public function getCurrentPaymentStatus(): ?string
    {
        $latestPayment = $this->latestPayment;
        return $latestPayment ? $latestPayment->status : null;
    }

    /**
     * Get the total refunded amount for this order.
     *
     * @return float
     */
    public function getRefundedAmount(): float
    {
        return (float) $this->payments()
            ->with('refunds')
            ->get()
            ->sum(function ($payment) {
                return $payment->getRefundedAmount();
            });
    }

    /**
     * Get the remaining amount after refunds.
     *
     * @return float
     */
    public function getRemainingAmount(): float
    {
        return max(0, (float) $this->net_amount - $this->getRefundedAmount());
    }

    /**
     * Update the order's payment status after a refund.
     *
     * @param Payment $payment The payment that was refunded
     * @param float $refundAmount The amount that was refunded
     * @return bool
     */
    public function updatePaymentStatusAfterRefund(Payment $payment, float $refundAmount): bool
    {
        // Calculate the total refunded amount
        $totalRefunded = $this->getRefundedAmount();

        // If the entire order amount has been refunded (with a small epsilon for floating point comparison)
        if (abs($totalRefunded - (float) $this->net_amount) < 0.01) {
            $this->payment_status = self::PAYMENT_STATUS_REFUNDED;
        }
        // If some money is refunded but not all
        elseif ($totalRefunded > 0) {
            $this->payment_status = self::PAYMENT_STATUS_PARTIALLY_REFUNDED;
        }

        return $this->save();
    }

    /**
     * Add a new payment record for this order.
     *
     * @param string $method
     * @param float $amount
     * @param string $status
     * @param array $metadata
     * @return Payment
     */
    public function addPayment(string $method, float $amount, string $status = Payment::STATUS_PENDING, array $metadata = []): Payment
    {
        return $this->payments()->create([
            'payment_method' => $method,
            'amount' => $amount,
            'currency' => 'USD', // Default currency
            'status' => $status,
            'payment_date' => now(),
            'metadata' => $metadata,
        ]);
    }

    // Add the method to get the maximum order_id
    public static function getMaxOrderId()
    {
        return self::max('order_number');
    }

    public function orderFiles()
    {
        return $this->hasMany(OrderFile::class);
    }

    /**
     * Accessor to ensure orderFiles is available in JSON responses
     */
    public function getOrderFilesAttribute()
    {
        // Return the relationship data if it's already loaded
        if ($this->relationLoaded('orderFiles')) {
            return $this->getRelation('orderFiles');
        }
        
        // If not loaded, return an empty collection to avoid N+1 queries
        return collect();
    }

    /**
     * The accessors to append to the model's array form.
     *
     * @var array
     */
    protected $appends = ['orderFiles'];


    /**
     * Scope a query to search for orders by order number or title.
     *
     * @param  \Illuminate\Database\Eloquent\Builder  $query
     * @param  string  $search
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeSearch($query, $search)
    {
        if ($search) {
            return $query->where(function ($q) use ($search) {
                $q->where('order_number', 'like', "%{$search}%")
                    ->orWhere('title', 'like', "%{$search}%");
            });
        }

        return $query;
    }

    /**
     * Scope a query to filter orders by payment status.
     *
     * @param  \Illuminate\Database\Eloquent\Builder  $query
     * @param  string|null  $status
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopePaymentStatus($query, $status)
    {
        if (!empty($status)) { // Ensure the status is not null or empty
            return $query->where('payment_status', $status);
        }

        return $query;
    }

    /**
     * Scope a query to filter refunded orders.
     *
     * @param  \Illuminate\Database\Eloquent\Builder  $query
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeRefunded($query)
    {
        return $query->where('payment_status', self::PAYMENT_STATUS_REFUNDED);
    }

    /**
     * Scope a query to filter partially refunded orders.
     *
     * @param  \Illuminate\Database\Eloquent\Builder  $query
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopePartiallyRefunded($query)
    {
        return $query->where('payment_status', self::PAYMENT_STATUS_PARTIALLY_REFUNDED);
    }

    /**
     * Scope a query to filter orders that have any type of refund.
     *
     * @param  \Illuminate\Database\Eloquent\Builder  $query
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeAnyRefund($query)
    {
        return $query->whereIn('payment_status', [
            self::PAYMENT_STATUS_REFUNDED,
            self::PAYMENT_STATUS_PARTIALLY_REFUNDED
        ]);
    }

    /**
     * Scope a query to filter orders by order status.
     *
     * @param  \Illuminate\Database\Eloquent\Builder  $query
     * @param  string|null  $status
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeOrderStatus($query, $status)
    {
        if (!empty($status)) { // Ensure the status is not null or empty
            return $query->where('order_status', $status);
        }

        return $query;
    }

    public function calculateWriterAmount($netAmount)
    {
        $adminSettings = AdminSetting::where('id', 1)->first();

        if ($adminSettings) {
            //  writer_fee is stored as a percentage value (e.g., 20.00 means 20%)
            $writerFeePercentage = $adminSettings->writer_fee;
            $writerAmount = ($netAmount * $writerFeePercentage) / 100;

            return $writerAmount;
        }

        // Return a default value or throw an exception if settings not found
        return 0;
    }

    /**
     * Calculate the writer deadline based on the order deadline
     *
     * @param string $orderDeadline The order deadline in datetime format
     * @return string The writer deadline in datetime format
     */
    public function calculateWriterDeadline($orderDeadline)
    {
        $adminSettings = AdminSetting::where('id', 1)->first();

        if ($adminSettings && $orderDeadline) {
            // Convert the deadline string to a Carbon instance
            $deadline = Carbon::parse($orderDeadline);

            // Get the total duration from now until the order deadline in hours
            $totalDuration = now()->diffInHours($deadline);

            // Calculate writer's time allocation (e.g., if writer_deadline is 70.00,
            // the writer gets 70% of the total time, meaning 30% less time than the client deadline)
            $writerTimePercentage = $adminSettings->writer_deadline;
            $writerDuration = ($totalDuration * $writerTimePercentage) / 100;

            // Calculate the writer deadline by adding the writer duration to now
            $writerDeadline = now()->addHours($writerDuration);

            // Format the writer deadline to match the required format
            return $writerDeadline->format('Y-m-d H:i:s');
        }

        // If no settings found or invalid deadline, return the original deadline
        return $orderDeadline;
    }

    public function bids()
    {
        return $this->hasMany(Bid::class, 'order_id');
    }

    public function writer()
    {
        return $this->belongsTo(User::class, 'writer_id');
    }

    /**
     * Get the writer infractions for this order.
     */
    public function writerInfractions()
    {
        return $this->hasMany(WriterInfraction::class);
    }

    /**
     * Get the status history for this order.
     */
    public function statusHistory()
    {
        return $this->hasMany(OrderStatusHistory::class);
    }

    /**
     * Get the latest status change.
     */
    public function latestStatusChange()
    {
        return $this->hasOne(OrderStatusHistory::class)->latest();
    }

    /**
     * Check if the order is in an active status.
     */
    public function isActive(): bool
    {
        return in_array($this->order_status, self::ACTIVE_STATUSES);
    }

    /**
     * Check if the order is completed.
     */
    public function isCompleted(): bool
    {
        return in_array($this->order_status, self::COMPLETED_STATUSES);
    }

    /**
     * Check if the order is in bidding phase.
     */
    public function isBidding(): bool
    {
        return $this->order_status === self::ORDER_STATUS_BIDDING;
    }

    /**
     * Check if the order is assigned to a writer.
     */
    public function isAssigned(): bool
    {
        return $this->order_status === self::ORDER_STATUS_ASSIGNED;
    }

    /**
     * Check if the writer is working on the order.
     */
    public function isInProgress(): bool
    {
        return $this->order_status === self::ORDER_STATUS_IN_PROGRESS;
    }

    /**
     * Check if the work has been submitted.
     */
    public function isSubmitted(): bool
    {
        return $this->order_status === self::ORDER_STATUS_SUBMITTED;
    }

    /**
     * Check if the order is under client review.
     */
    public function isUnderReview(): bool
    {
        return $this->order_status === self::ORDER_STATUS_UNDER_REVIEW;
    }

    /**
     * Check if the order is approved.
     */
    public function isApproved(): bool
    {
        return $this->order_status === self::ORDER_STATUS_APPROVED;
    }

    /**
     * Check if the order needs revision.
     */
    public function needsRevision(): bool
    {
        return $this->order_status === self::ORDER_STATUS_REVISION_REQUESTED;
    }

    /**
     * Check if the order is cancelled.
     */
    public function isCancelled(): bool
    {
        return $this->order_status === self::ORDER_STATUS_CANCELLED;
    }

    /**
     * Get the next possible statuses for this order.
     */
    public function getNextPossibleStatuses(): array
    {
        switch ($this->order_status) {
            case self::ORDER_STATUS_PENDING:
                return [self::ORDER_STATUS_BIDDING, self::ORDER_STATUS_CANCELLED];
            
            case self::ORDER_STATUS_PENDING_PAYMENT:
                return [self::ORDER_STATUS_BIDDING, self::ORDER_STATUS_CANCELLED];
            
            case self::ORDER_STATUS_BIDDING:
                return [self::ORDER_STATUS_ASSIGNED, self::ORDER_STATUS_CANCELLED];
            
            case self::ORDER_STATUS_ASSIGNED:
                return [self::ORDER_STATUS_IN_PROGRESS, self::ORDER_STATUS_CANCELLED];
            
            case self::ORDER_STATUS_IN_PROGRESS:
                return [self::ORDER_STATUS_SUBMITTED, self::ORDER_STATUS_CANCELLED];
            
            case self::ORDER_STATUS_SUBMITTED:
                return [self::ORDER_STATUS_UNDER_REVIEW, self::ORDER_STATUS_REVISION_REQUESTED];
            
            case self::ORDER_STATUS_UNDER_REVIEW:
                return [self::ORDER_STATUS_APPROVED, self::ORDER_STATUS_REVISION_REQUESTED];
            
            case self::ORDER_STATUS_REVISION_REQUESTED:
                return [self::ORDER_STATUS_IN_PROGRESS, self::ORDER_STATUS_CANCELLED];
            
            default:
                return [];
        }
    }

    /**
     * Check if a status transition is valid.
     */
    public function canTransitionTo(string $newStatus): bool
    {
        return in_array($newStatus, $this->getNextPossibleStatuses());
    }

    /**
     * Get the human-readable status name.
     */
    public function getStatusDisplayName(): string
    {
        $statusNames = [
            self::ORDER_STATUS_PENDING => 'Pending',
            self::ORDER_STATUS_PENDING_PAYMENT => 'Pending Payment',
            self::ORDER_STATUS_BIDDING => 'Bidding',
            self::ORDER_STATUS_ASSIGNED => 'Assigned',
            self::ORDER_STATUS_IN_PROGRESS => 'In Progress',
            self::ORDER_STATUS_SUBMITTED => 'Submitted',
            self::ORDER_STATUS_UNDER_REVIEW => 'Under Review',
            self::ORDER_STATUS_APPROVED => 'Approved',
            self::ORDER_STATUS_REVISION_REQUESTED => 'Revision Requested',
            self::ORDER_STATUS_CANCELLED => 'Cancelled',
        ];

        return $statusNames[$this->order_status] ?? 'Unknown';
    }

    /**
     * Get the status color for UI display.
     */
    public function getStatusColor(): string
    {
        $statusColors = [
            self::ORDER_STATUS_PENDING => 'gray',
            self::ORDER_STATUS_PENDING_PAYMENT => 'orange',
            self::ORDER_STATUS_BIDDING => 'blue',
            self::ORDER_STATUS_ASSIGNED => 'purple',
            self::ORDER_STATUS_IN_PROGRESS => 'yellow',
            self::ORDER_STATUS_SUBMITTED => 'indigo',
            self::ORDER_STATUS_UNDER_REVIEW => 'orange',
            self::ORDER_STATUS_APPROVED => 'green',
            self::ORDER_STATUS_REVISION_REQUESTED => 'red',
            self::ORDER_STATUS_CANCELLED => 'gray',
        ];

        return $statusColors[$this->order_status] ?? 'gray';
    }

    /**
     * Get the status icon for UI display.
     */
    public function getStatusIcon(): string
    {
        $statusIcons = [
            self::ORDER_STATUS_PENDING => 'clock',
            self::ORDER_STATUS_PENDING_PAYMENT => 'credit-card',
            self::ORDER_STATUS_BIDDING => 'users',
            self::ORDER_STATUS_ASSIGNED => 'user-check',
            self::ORDER_STATUS_IN_PROGRESS => 'edit-3',
            self::ORDER_STATUS_SUBMITTED => 'upload',
            self::ORDER_STATUS_UNDER_REVIEW => 'eye',
            self::ORDER_STATUS_APPROVED => 'check-circle',
            self::ORDER_STATUS_REVISION_REQUESTED => 'refresh-cw',
            self::ORDER_STATUS_CANCELLED => 'x-circle',
        ];

        return $statusIcons[$this->order_status] ?? 'file-text';
    }

    /**
     * Get the writer payment records for this order.
     */
    public function writerPaymentRecords()
    {
        return $this->hasMany(WriterPaymentRecord::class);
    }

    /**
     * Get the latest writer payment record for this order.
     */
    public function latestWriterPaymentRecord()
    {
        return $this->hasOne(WriterPaymentRecord::class)->latest();
    }

    /**
     * Update writer payment status based on order status.
     */
    public function updateWriterPaymentStatus()
    {
        if (!$this->writer_id) {
            return; // No writer assigned
        }

        $newPaymentStatus = $this->calculateWriterPaymentStatus();
        
        if ($newPaymentStatus !== $this->writer_payment_status) {
            $this->update([
                'writer_payment_status' => $newPaymentStatus,
                'writer_payment_calculated_at' => now(),
            ]);

            // If payment becomes available, set release timestamp
            if ($newPaymentStatus === self::WRITER_PAYMENT_STATUS_AVAILABLE && !$this->writer_payment_released_at) {
                $this->update(['writer_payment_released_at' => now()]);
            }
        }
    }

    /**
     * Calculate what the writer payment status should be based on current order status.
     */
    public function calculateWriterPaymentStatus(): string
    {
        // If order is approved, payment is available
        if ($this->order_status === self::ORDER_STATUS_APPROVED) {
            return self::WRITER_PAYMENT_STATUS_AVAILABLE;
        }

        // If order needs revision, payment is pending revision
        if ($this->order_status === self::ORDER_STATUS_REVISION_REQUESTED) {
            return self::WRITER_PAYMENT_STATUS_PENDING_REVISION;
        }

        // If order is cancelled, no payment
        if ($this->order_status === self::ORDER_STATUS_CANCELLED) {
            return self::WRITER_PAYMENT_STATUS_PENDING; // Or could be a different status
        }

        // For all other statuses (assigned, in_progress, submitted, under_review), payment is pending
        if (in_array($this->order_status, [
            self::ORDER_STATUS_ASSIGNED,
            self::ORDER_STATUS_IN_PROGRESS,
            self::ORDER_STATUS_SUBMITTED,
            self::ORDER_STATUS_UNDER_REVIEW
        ])) {
            return self::WRITER_PAYMENT_STATUS_PENDING;
        }

        // Default to pending for any other cases
        return self::WRITER_PAYMENT_STATUS_PENDING;
    }

    /**
     * Check if writer payment is pending.
     */
    public function isWriterPaymentPending(): bool
    {
        return in_array($this->writer_payment_status, self::WRITER_PENDING_STATUSES);
    }

    /**
     * Check if writer payment is available for withdrawal.
     */
    public function isWriterPaymentAvailable(): bool
    {
        return $this->writer_payment_status === self::WRITER_PAYMENT_STATUS_AVAILABLE;
    }

    /**
     * Check if writer payment has been withdrawn.
     */
    public function isWriterPaymentWithdrawn(): bool
    {
        return $this->writer_payment_status === self::WRITER_PAYMENT_STATUS_WITHDRAWN;
    }

    /**
     * Create a payment record for this order when writer is assigned.
     */
    public function createWriterPaymentRecord()
    {
        if (!$this->writer_id || !$this->writer_amount) {
            return null;
        }

        // Check if payment record already exists
        $existingRecord = WriterPaymentRecord::where('order_id', $this->id)
            ->where('writer_id', $this->writer_id)
            ->first();

        if ($existingRecord) {
            return $existingRecord;
        }

        return WriterPaymentRecord::create([
            'order_id' => $this->id,
            'writer_id' => $this->writer_id,
            'payment_amount' => $this->writer_amount,
            'currency' => $this->currency ?? 'USD',
            'status' => WriterPaymentRecord::STATUS_PENDING,
            'earned_at' => $this->order_status === self::ORDER_STATUS_APPROVED ? now() : null,
        ]);
    }

    /**
     * Scope to get orders with pending writer payments for a specific writer.
     */
    public function scopeWithPendingWriterPayment($query, $writerId = null)
    {
        $query = $query->whereIn('writer_payment_status', self::WRITER_PENDING_STATUSES);
        
        if ($writerId) {
            $query->where('writer_id', $writerId);
        }

        return $query;
    }

    /**
     * Scope to get orders with available writer payments for a specific writer.
     */
    public function scopeWithAvailableWriterPayment($query, $writerId = null)
    {
        $query = $query->whereIn('writer_payment_status', self::WRITER_AVAILABLE_STATUSES);
        
        if ($writerId) {
            $query->where('writer_id', $writerId);
        }

        return $query;
    }

    /**
     * Get the rating for this order.
     */
    public function rating()
    {
        return $this->hasOne(OrderRating::class);
    }

    /**
     * Check if this order has been rated by the client.
     */
    public function hasRating(): bool
    {
        return $this->rating()->exists();
    }

    /**
     * Check if this order is eligible for rating (completed and not already rated).
     */
    public function isEligibleForRating(): bool
    {
        return $this->order_status === self::ORDER_STATUS_APPROVED && !$this->hasRating();
    }

    /**
     * Get the rating deadline (when rating can no longer be submitted).
     */
    public function getRatingDeadline(): ?Carbon
    {
        if (!$this->client_approved_at) {
            return null;
        }
        
        // Allow rating for 30 days after order approval
        return $this->client_approved_at->addDays(30);
    }

    /**
     * Check if rating period has expired.
     */
    public function isRatingExpired(): bool
    {
        $deadline = $this->getRatingDeadline();
        return $deadline && now()->isAfter($deadline);
    }

    /**
     * Get the revisions for this order.
     */
    public function revisions()
    {
        return $this->hasMany(OrderRevision::class);
    }

    /**
     * Get the active revisions for this order.
     */
    public function activeRevisions()
    {
        return $this->hasMany(OrderRevision::class)->whereIn('revision_status', [
            OrderRevision::STATUS_REQUESTED, 
            OrderRevision::STATUS_ACKNOWLEDGED, 
            OrderRevision::STATUS_IN_PROGRESS
        ]);
    }

    /**
     * Get the reassignments for this order.
     */
    public function reassignments()
    {
        return $this->hasMany(OrderReassignment::class);
    }

    /**
     * Get the latest reassignment for this order.
     */
    public function latestReassignment()
    {
        return $this->hasOne(OrderReassignment::class)->latest();
    }

    /**
     * Check if the order has been reassigned.
     */
    public function hasBeenReassigned(): bool
    {
        return $this->reassignments()->exists();
    }

    /**
     * Check if the order has active revisions.
     */
    public function hasActiveRevisions(): bool
    {
        return $this->activeRevisions()->exists();
    }

    /**
     * Get the total revision count for this order.
     */
    public function getRevisionCount(): int
    {
        return $this->revisions()->count();
    }

    /**
     * Get high priority revisions for this order.
     */
    public function highPriorityRevisions()
    {
        return $this->hasMany(OrderRevision::class)->highPriority();
    }

    /**
     * Get overdue revisions for this order.
     */
    public function overdueRevisions()
    {
        return $this->hasMany(OrderRevision::class)->overdue();
    }

    /**
     * Check if the order has high priority revisions.
     */
    public function hasHighPriorityRevisions(): bool
    {
        return $this->highPriorityRevisions()->exists();
    }

    /**
     * Check if the order has overdue revisions.
     */
    public function hasOverdueRevisions(): bool
    {
        return $this->overdueRevisions()->exists();
    }

    /**
     * Get revisions by type.
     */
    public function revisionsByType(string $type)
    {
        return $this->hasMany(OrderRevision::class)->byType($type);
    }

    /**
     * Get revisions by priority.
     */
    public function revisionsByPriority(string $priority)
    {
        return $this->hasMany(OrderRevision::class)->byPriority($priority);
    }

    /**
     * Get revisions by status.
     */
    public function revisionsByStatus(string $status)
    {
        return $this->hasMany(OrderRevision::class)->byStatus($status);
    }

    /**
     * Check if order can request revision.
     */
    public function canRequestRevision(): bool
    {
        return in_array($this->order_status, [
            self::ORDER_STATUS_SUBMITTED,
            self::ORDER_STATUS_UNDER_REVIEW
        ]);
    }

    /**
     * Check if order can be submitted as complete.
     */
    public function canSubmitAsComplete(): bool
    {
        return in_array($this->order_status, [
            self::ORDER_STATUS_ASSIGNED,
            self::ORDER_STATUS_IN_PROGRESS
        ]);
    }

    /**
     * Check if order can be submitted as draft.
     */
    public function canSubmitAsDraft(): bool
    {
        return in_array($this->order_status, [
            self::ORDER_STATUS_ASSIGNED,
            self::ORDER_STATUS_IN_PROGRESS
        ]);
    }

    /**
     * Get the next revision number for this order.
     */
    public function getNextRevisionNumber(): int
    {
        return $this->revisions()->count() + 1;
    }

    /**
     * Check if the order has a draft submission.
     */
    public function hasDraftSubmission(): bool
    {
        return $this->submission_type === self::SUBMISSION_TYPE_DRAFT;
    }

    /**
     * Check if the order has a complete submission.
     */
    public function hasCompleteSubmission(): bool
    {
        return $this->submission_type === self::SUBMISSION_TYPE_COMPLETE;
    }

    /**
     * Get draft files for this order.
     */
    public function draftFiles()
    {
        return $this->hasMany(OrderFile::class)->where('file_category', 'draft');
    }

    /**
     * Get complete files for this order.
     */
    public function completeFiles()
    {
        return $this->hasMany(OrderFile::class)->where('file_category', 'complete');
    }
}
