<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class OrderRating extends Model
{
    use HasFactory;

    protected $fillable = [
        'order_id',
        'client_id',
        'writer_id',
        'overall_rating',
        'quality_rating',
        'communication_rating',
        'timeliness_rating',
        'value_rating',
        'review_title',
        'review_text',
        'is_anonymous',
        'is_verified',
        'rating_categories',
        'rating_tags',
        'metadata',
        'rated_at'
    ];

    protected $casts = [
        'overall_rating' => 'integer',
        'quality_rating' => 'integer',
        'communication_rating' => 'integer',
        'timeliness_rating' => 'integer',
        'value_rating' => 'integer',
        'is_anonymous' => 'boolean',
        'is_verified' => 'boolean',
        'rating_categories' => 'array',
        'rating_tags' => 'array',
        'metadata' => 'array',
        'rated_at' => 'datetime'
    ];

    // Rating constants
    const MIN_RATING = 1;
    const MAX_RATING = 5;
    const DEFAULT_RATING = 5;

    // Rating categories
    const RATING_CATEGORIES = [
        'quality' => 'quality_rating',
        'communication' => 'communication_rating',
        'timeliness' => 'timeliness_rating',
        'value' => 'value_rating'
    ];

    // Rating tags
    const RATING_TAGS = [
        'professional', 'creative', 'thorough', 'original', 'well-researched',
        'on-time', 'responsive', 'helpful', 'excellent', 'outstanding',
        'meets-expectations', 'exceeds-expectations', 'good-value', 'recommended'
    ];

    /**
     * Get the order that this rating belongs to.
     */
    public function order(): BelongsTo
    {
        return $this->belongsTo(Order::class);
    }

    /**
     * Get the client who gave this rating.
     */
    public function client(): BelongsTo
    {
        return $this->belongsTo(User::class, 'client_id');
    }

    /**
     * Get the writer who received this rating.
     */
    public function writer(): BelongsTo
    {
        return $this->belongsTo(User::class, 'writer_id');
    }

    /**
     * Scope to get only verified ratings.
     */
    public function scopeVerified($query)
    {
        return $query->where('is_verified', true);
    }

    /**
     * Scope to get ratings by writer.
     */
    public function scopeByWriter($query, $writerId)
    {
        return $query->where('writer_id', $writerId);
    }

    /**
     * Scope to get ratings by client.
     */
    public function scopeByClient($query, $clientId)
    {
        return $query->where('client_id', $clientId);
    }

    /**
     * Scope to get ratings within a rating range.
     */
    public function scopeRatingRange($query, $minRating, $maxRating = null)
    {
        if ($maxRating === null) {
            $maxRating = $minRating;
        }
        return $query->whereBetween('overall_rating', [$minRating, $maxRating]);
    }

    /**
     * Scope to get recent ratings.
     */
    public function scopeRecent($query, $days = 30)
    {
        return $query->where('rated_at', '>=', now()->subDays($days));
    }

    /**
     * Get the average rating for a specific category.
     */
    public function getCategoryRating(string $category): ?float
    {
        $field = self::RATING_CATEGORIES[$category] ?? null;
        return $field ? $this->$field : null;
    }

    /**
     * Check if this rating has a specific category rating.
     */
    public function hasCategoryRating(string $category): bool
    {
        return isset($this->rating_categories[$category]) && 
               $this->getCategoryRating($category) !== null;
    }

    /**
     * Get all category ratings as an array.
     */
    public function getAllCategoryRatings(): array
    {
        $ratings = [];
        foreach (self::RATING_CATEGORIES as $category => $field) {
            if ($this->$field !== null) {
                $ratings[$category] = $this->$field;
            }
        }
        return $ratings;
    }

    /**
     * Get the rating tags as an array.
     */
    public function getRatingTags(): array
    {
        return $this->rating_tags ?? [];
    }

    /**
     * Check if this rating has a specific tag.
     */
    public function hasTag(string $tag): bool
    {
        return in_array($tag, $this->getRatingTags());
    }

    /**
     * Get the display name for the client (anonymous or actual name).
     */
    public function getClientDisplayName(): string
    {
        if ($this->is_anonymous) {
            return 'Anonymous Client';
        }
        return $this->client->name ?? 'Unknown Client';
    }

    /**
     * Validate rating values.
     */
    public static function validateRating(int $rating): bool
    {
        return $rating >= self::MIN_RATING && $rating <= self::MAX_RATING;
    }

    /**
     * Get the rating display text based on the overall rating.
     */
    public function getRatingDisplayText(): string
    {
        return match($this->overall_rating) {
            5 => 'Excellent',
            4 => 'Very Good',
            3 => 'Good',
            2 => 'Fair',
            1 => 'Poor',
            default => 'Unknown'
        };
    }

    /**
     * Get the rating color class for UI display.
     */
    public function getRatingColorClass(): string
    {
        return match($this->overall_rating) {
            5 => 'text-green-600 bg-green-100',
            4 => 'text-blue-600 bg-blue-100',
            3 => 'text-yellow-600 bg-yellow-100',
            2 => 'text-orange-600 bg-orange-100',
            1 => 'text-red-600 bg-red-100',
            default => 'text-gray-600 bg-gray-100'
        };
    }

    /**
     * Check if this is a positive rating (4 or 5 stars).
     */
    public function isPositive(): bool
    {
        return $this->overall_rating >= 4;
    }

    /**
     * Check if this is a negative rating (1 or 2 stars).
     */
    public function isNegative(): bool
    {
        return $this->overall_rating <= 2;
    }

    /**
     * Check if this is a neutral rating (3 stars).
     */
    public function isNeutral(): bool
    {
        return $this->overall_rating === 3;
    }
}
