<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class OrderReassignment extends Model
{
    use HasFactory;

    protected $fillable = [
        'order_id',
        'original_writer_id',
        'new_writer_id',
        'reassigned_by',
        'reassignment_reason',
        'detailed_reason',
        'admin_notes',
        'progress_percentage',
        'order_status_at_reassignment',
        'reassigned_at',
        'work_handover_completed_at',
        'new_writer_started_at',
        'original_writer_payment',
        'new_writer_payment',
        'total_payment',
        'work_handover_notes',
        'files_transferred',
        'requirements_summary',
        'original_work_quality_rating',
        'quality_assessment_notes',
        'delay_hours',
        'additional_cost',
    ];

    protected $casts = [
        'reassigned_at' => 'datetime',
        'work_handover_completed_at' => 'datetime',
        'new_writer_started_at' => 'datetime',
        'progress_percentage' => 'decimal:2',
        'original_writer_payment' => 'decimal:2',
        'new_writer_payment' => 'decimal:2',
        'total_payment' => 'decimal:2',
        'files_transferred' => 'array',
        'original_work_quality_rating' => 'integer',
        'delay_hours' => 'decimal:2',
        'additional_cost' => 'decimal:2',
    ];

    /**
     * Get the order that was reassigned.
     */
    public function order(): BelongsTo
    {
        return $this->belongsTo(Order::class);
    }

    /**
     * Get the original writer who was assigned to the order.
     */
    public function originalWriter(): BelongsTo
    {
        return $this->belongsTo(User::class, 'original_writer_id');
    }

    /**
     * Get the new writer who was assigned to the order.
     */
    public function newWriter(): BelongsTo
    {
        return $this->belongsTo(User::class, 'new_writer_id');
    }

    /**
     * Get the admin who made the reassignment.
     */
    public function reassignedBy(): BelongsTo
    {
        return $this->belongsTo(User::class, 'reassigned_by');
    }

    /**
     * Check if the work handover has been completed.
     */
    public function isHandoverCompleted(): bool
    {
        return !is_null($this->work_handover_completed_at);
    }

    /**
     * Check if the new writer has started working.
     */
    public function hasNewWriterStarted(): bool
    {
        return !is_null($this->new_writer_started_at);
    }

    /**
     * Calculate the total delay caused by reassignment.
     */
    public function getTotalDelayHours(): float
    {
        $delay = $this->delay_hours;
        
        // Add additional delay if handover is not completed
        if (!$this->isHandoverCompleted()) {
            $delay += now()->diffInHours($this->reassigned_at);
        }
        
        return $delay;
    }

    /**
     * Get the payment distribution summary.
     */
    public function getPaymentSummary(): array
    {
        return [
            'original_writer' => $this->original_writer_payment,
            'new_writer' => $this->new_writer_payment,
            'total' => $this->total_payment,
            'additional_cost' => $this->additional_cost,
        ];
    }

    /**
     * Check if this reassignment is cost-effective.
     */
    public function isCostEffective(): bool
    {
        return $this->total_payment <= $this->order->writer_amount + $this->additional_cost;
    }

    /**
     * Scope for recent reassignments.
     */
    public function scopeRecent($query, $days = 30)
    {
        return $query->where('reassigned_at', '>=', now()->subDays($days));
    }

    /**
     * Scope for reassignments by reason.
     */
    public function scopeByReason($query, $reason)
    {
        return $query->where('reassignment_reason', $reason);
    }

    /**
     * Scope for reassignments by writer.
     */
    public function scopeByWriter($query, $writerId)
    {
        return $query->where(function ($q) use ($writerId) {
            $q->where('original_writer_id', $writerId)
              ->orWhere('new_writer_id', $writerId);
        });
    }

    /**
     * Scope for reassignments with quality issues.
     */
    public function scopeQualityIssues($query)
    {
        return $query->where('reassignment_reason', 'quality_issues');
    }

    /**
     * Scope for reassignments with deadline violations.
     */
    public function scopeDeadlineViolations($query)
    {
        return $query->where('reassignment_reason', 'deadline_violation');
    }
} 